/* File     : get_argument.c
 * Author   : Karyl F. Stein <steinkf@cs.purdue.edu>
 * Purpose  : Functions for reading arguments from an input stream.  The
 *            individual functions are detailed below.
 *
 * Xnew is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "config.h"
#include "list_func.h"
#include <stdio.h>

/* Set defaults if not defined previously or not valid.  Do not change. */
#ifndef INBUF
# define INBUF 512
#elsif (INBUF < 2)
# define INBUF 512
#endif


/* Function: count_arg
 * Input   : A string to count the number of arguments of.  Arguments are
 *           defined as strings seperated by one or more spaces or tabs ending
 *           with a new line or NULL.  If a # character is the first character
 *           in an argument, the rest of the string is determined to be a
 *           comment and ignored.
 * Return  : The number of arguments in the passed string.
 */
int count_arg (char *str) {
  char *ptr;
  int flag = 0, retval = 0;

  if (str == NULL)
    return(0);

  /* Count the number of arguments */
  for (ptr = str; *ptr != '\0'; ++ptr) {
    if (((*ptr == '#') && !flag) || (*ptr == '\n'))
      break;
    if ((*ptr == ' ') || (*ptr == '\t')) {
      if (flag)
	++retval;
      flag = 0;
    } else flag = 1;
  }

  if (flag)
    ++retval;
  return(retval);
}


/* Function: read_arguments
 * Input   : An open file pointer from which to read.
 * Return  : Everything to and including the first \n from the passed file
 *           pointer.  The trailing \n is converted to a \0, (meaning the
 *           returned string ends in \0\0);
 */
char *read_arguments (FILE *ifp) {
  char *retval = NULL;
  extern int line;
  int read_len = INBUF, start = 0, tmpint = 0;

  /* Loop until \n is found reading INBUF chunks at a time */
  do {
    if ((retval = (char *) realloc(retval, INBUF * ++tmpint)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    if ((fgets(retval + start, read_len, ifp)) == NULL) {
      fprintf(stderr, "Unknown Read Error (line %d)\n", line);
      exit(1);
    }
    if (read_len == INBUF)
      ++read_len;
    if (start == 0)
      start = INBUF - 1;
    else start += INBUF;
  } while (retval[strlen(retval) - 1] != '\n');

  /* Increment the line counter, convert the trailing \n to \0, and return */
  ++line;
  retval[strlen(retval) - 1] = '\0';
  return(retval);
}


/* Function: get_arguments
 * Input   : A file pointer to read data from
 * Return  : An array of character pointers where each pointer points to an
 *           argument string or NULL if no arguments.  Arguments are expanded
 *           here.
 */
char **get_arguments (FILE *ifp) {
  char **retval;
  char *head, *inbuf, *ptr, *tmpstr;
  int arg_count, tmpint = 0;

  inbuf = read_arguments(ifp);
  arg_count = count_arg(inbuf);

  /* Fill the argument array */
  if ((retval = (char **) malloc(sizeof(char *) * (arg_count + 1))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  ptr = inbuf;
  while (tmpint < arg_count) {
    while ((*ptr == ' ') || (*ptr == '\t'))
      ++ptr;
    head = ptr;
    while ((*ptr != '\0') && (*ptr != ' ') && (*ptr != '\t'))
      ++ptr;
    *ptr = '\0';

    /* Expand variables and module results */
    if ((*head == '$') || (*head == '&'))
      tmpstr = get_result((char *) (head + ((*head == '$') ? 0 : 1)));
    else tmpstr = head;

    /* Add the appropiate value to the argument array.  NULL values are */
    /* translated to the string "NULL".                                 */
    if ((tmpstr == NULL) || (*tmpstr == '\0')) {
      if ((retval[tmpint] = (char *) malloc(strlen("NULL") + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(retval[tmpint++], "NULL");
    } else {
      if ((retval[tmpint] = (char *) malloc(strlen(tmpstr) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(retval[tmpint++], tmpstr);
    }

    ++ptr;
  }

  free(inbuf);
  retval[tmpint] = NULL;
  return(retval);
}
