/* File     : do_command.c
 * Author   : Karyl F. Stein <xenon@xenos.net>
 * Purpose  : The do_command() function and supporting functions.  The
 *            individual functions are detailed below.
 *
 * Xnew is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"      /* INBUF, DATA_DIR   */
#include "xnew.h"
#include "xnew_func.h"   /* clear_arguments() */
#include "do_command.h"  /* Local prototypes  */
#include <stdio.h>
#include <errno.h>       /* errno             */
#include <sys/types.h>   /* open()            */
#include <sys/stat.h>    /* open()            */
#include <fcntl.h>       /* open()            */
#include <sys/wait.h>    /* wait()            */
#include <string.h>      /* strrchr()         */
#include <termios.h>     /* termios()         */
#include <unistd.h>      /* termios()         */

#define PID_LEN 6        /* Number of significant digits of the PID to */
                         /* use when creating temporary files.         */


/* Function: command_type
 * Input   : A command.
 * Return  : SYSTEM is command is "system", EXEC if command is "exec",
 *           SET if command is "set", otherwise MODULE.
 */
int command_type (char *command) {
  if (strcmp(command, "system") == 0)
    return(SYSTEM);
  if (strcmp(command, "exec") == 0)
    return(EXEC);
  if (strcmp(command, "set") == 0)
    return(SET);
  return(MODULE);
}

  
/* Function: arg_count
 * Input   : A pointer to a NULL terminated array of argument strings.
 * Return  : The number of arguments in the array.
 */
int arg_count (char *arguments[]) {
  char **arg_ptr;
  int retval = 0;

  if (arguments != NULL)
    for (arg_ptr = arguments; *arg_ptr != NULL; ++arg_ptr, ++retval);
  return(retval);
}


/* Function: create_argument_list
 * Input   : A pointer to an array and an offset.  Each entry in the array
 *           should point to a string argument except for the last entry
 *           which should be NULL.  The offset defines how many empty entries
 *           should be left in the front of the returned array.
 * Return  : A pointer to an array holding the arguments passed with the
 *           first offset entries set to NULL.
 */
char **create_argument_list (char *args[], int offset) {
  char **arg_ptr = args, **retval;
  int place = offset;

  /* Create the new argument array */
  if ((retval = (char **) malloc(sizeof(char *) *
				 (arg_count(args) + offset + 1))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  /* Fill the new argument array */
  if (args != NULL) {
    for ( ; *arg_ptr != NULL; ++arg_ptr, ++place) {
      if ((retval[place] = (char *) malloc(strlen(*arg_ptr) + 1)) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      strcpy(retval[place], *arg_ptr);
    }
  }
  retval[place] = NULL;

  /* Set the first offset entries to NULL */
  while (--offset >= 0)
    retval[offset] = NULL;

  return(retval);
}


/* Function: create_command_name
 * Input   : A command.
 * Return  : If the passed command begins with a /, or DATA_DIR is undefined,
 *           then the command as passed is returned.  Otherwise, DATA_DIR/
 *           followed by the passed command is returned.
 */
char *create_command_name (char *command) {
  char *retval;

#ifdef DATA_DIR
  if (*command == '/') {
    if ((retval = (char *) malloc(strlen(command) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(retval, command);
  } else {
    if ((retval = (char *) malloc(strlen(DATA_DIR) + strlen(command) + 2))
	== NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    sprintf(retval, "%s/%s", DATA_DIR, command);
  }
#else
  if ((retval = (char *) malloc(strlen(command) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, command);
#endif  /* DATA_DIR */

  return(retval);
}


/* Function: create_tmp_file
 * Input   : An argument list to add the new data file to.
 * Return  : A file descripter of a unique temporary file.
 * Notes   : Creates the file used for communication between the main program
 *           and the called module.  This may be exploitable if TMP_DIR is on
 *           an NFS partition.
 */
char *create_tmp_file (void) {
  char pid_str[PID_LEN];
  char *datafile;
  int ifd, tmpint, umask_save = umask(0077);
  pid_t pid = getpid(), tmppid = pid;

  if ((datafile = (char *) malloc(strlen(TMP_DIR) + strlen("/xnew.") +
				  PID_LEN + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }

  /* Create a unique temporary file */
  while (1) {
    for (tmpint = 0; tmpint < PID_LEN - 1; ++tmpint, tmppid /= 10)
      pid_str[tmpint] = (tmppid % 10) + '0';
    pid_str[tmpint] = '\0';
    sprintf(datafile, "%s/xnew.%s", TMP_DIR, pid_str);
    if ((ifd = open(datafile, O_RDONLY | O_CREAT | O_EXCL,
		    S_IRUSR | S_IWUSR)) == -1) {
      if (errno == EEXIST) {
	++pid;
	continue;
      } else {
	fprintf(stderr, "Unable to open file %s\n", datafile);
	exit(1);
      }
    } else break;
  }
  umask(umask_save);

  close(ifd);
  return(datafile);
}


/* Function: do_command
 * Input   : The name of the command to run and a NULL terminated list of
 *           arguments to pass to the command.  If command is system or exec,
 *           then the first argument in the passed argument list is taken as
 *           the system command to run.  Otherwise, the passed command is
 *           considered to be a module name and the default module path is
 *           placed in front of the command, (unless the command begins with
 *           the / character).
 * Output  :
 * Return  :
 */
char *do_command (char *command, char **args) {
  char **arg_list, **arg_ptr;
  char *module, *retval = NULL;
  char inbuf[INBUF];
  extern int errno;
  int count = 0, flag, ifd, status;
  pid_t tmppid;
  struct termios save_scr;

  flag = command_type(command);
  arg_list = create_argument_list(args, (flag & MODULE) ? 2 : 0);

  /* Set an environment or terminal variable */
  if (flag & SET) {
    if (args[0] == NULL) {
      fprintf(stderr, "Syntax Error: set declared with no parameters\n");
      exit(1);
    }
    if (strcmp(args[0], "erase") == 0) {
      if (tcgetattr(0, &save_scr) == -1) {
	fprintf(stderr, "Fatal Error: Unable to get terminal settings\n");
	exit(1);
      }
      save_scr.c_cc[VERASE] = *args[1];
      if (tcsetattr(0, TCSANOW, &save_scr) == -1) {
	fprintf(stderr, "Fatal Error: Unable to set terminal settings\n");
	exit(1);
      }
    }
    else {
      if ((module = (char *) malloc(strlen(args[0]) + strlen(args[1]) + 2))
	  == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }
      sprintf(module, "%s=%s", args[0], args[1]);
      if (putenv(module) == -1) {
	fprintf(stderr, "Out Of Environment Memory\n");
	exit(1);
      }
    }
    return(NULL);
  }

  if ((flag & SYSTEM) || (flag & EXEC))
    if (arg_list[0] == NULL)
      return(NULL);
    else {
      free(arg_list[0]);
      arg_list[0] = strip_path(args[0]);
    }

  /* Run a "system" command */
  if (flag & SYSTEM) {
    if ((tmppid = fork()) == -1) {
      fprintf(stderr, "Unable to fork()\n");
      exit(1);
    }
    if (tmppid == 0)
      execv(args[0], arg_list);
    else waitpid(tmppid, &status, 0);
    free(clear_arguments(arg_list));
    return(NULL);
  } 

  /* Run an "exec" command */
  if (flag & EXEC)
    execv(args[0], arg_list);

  /* Finish the argument list */
  module = create_command_name(command);
  arg_list[0] = strip_path(command);
  tmp_file = arg_list[1] = create_tmp_file();

  /* Open the temporary file */
  if ((ifd = open(arg_list[1], O_RDONLY)) == -1) {
    fprintf(stderr, "Unable To Open %s For Reading\n", arg_list[1]);
    remove(arg_list[1]);
    exit(1);
  }

  /* Run the module */
  if ((tmppid = fork()) == -1) {
    fprintf(stderr, "Unable to fork()\n");
    exit(1);
  }

  if (tmppid == 0) {
    if (execv(module, arg_list) == -1) {
      fprintf(stderr, "Unable to run module %s\n", arg_list[0]);
      exit(1);
    }
  } else {
    waitpid(tmppid, &status, 0);

    /* Check return value.  Put more checks in here. */
    if ((WIFEXITED(status) != 0) && (WEXITSTATUS(status) != 0)) {
      remove(arg_list[1]);
      fprintf(stderr, "Fatal Error in module %s\n", module);
      exit(1);
    }

    /* Read the data */
    while ((count = read(ifd, inbuf, INBUF - 1)) != 0) {
      if (count == -1) {
	remove(arg_list[1]);
	fprintf(stderr, "Error Reading %s\n", arg_list[1]);
	exit(1);
      }

      inbuf[count] = '\0';
      if (retval == NULL) {
	if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
	  fprintf(stderr, "Out Of Memory\n");
	  remove(arg_list[1]);
	  exit(1);
	}
	strcpy(retval, inbuf);
      } else {
	if ((retval = (char *) realloc(retval, strlen(retval) + strlen(inbuf)
				       + 1)) == NULL) {
	  fprintf(stderr, "Out Of Memory\n");
	  remove(arg_list[1]);
	  exit(1);
	}
	strcat(retval, inbuf);
      }
    }

    /* Clean up */
    close(ifd);
    remove(arg_list[1]);
    tmp_file = NULL;
    free(module);
    free(clear_arguments(arg_list));
    return(retval);
  }
}


/* Function: strip_path
 * Input   : A command name.
 * Return  : The command name with the leading path (if any) stripped from it.
 */
char *strip_path (char *command) {
  char *ptr, *retval;

  if (command == NULL)
    return(NULL);

  if ((ptr = strrchr(command, '/')) == NULL)
    ptr = command;
  else ++ptr;

  if ((retval = (char *) malloc(strlen(ptr) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, ptr);

  return(retval);
}
