/*
 * XOTPCalc, an S/Key and OPIE compatible one-time password calculator
 * Copyright (C) 1997 Ivan Nejgebauer <ian@uns.ns.ac.yu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <Xm/Xm.h>
#include <Xm/MessageB.h>
#include "xotpcalc.h"
#include "hosts.h"

/* global data */
XtAppContext app;
Widget top;
PrivResRec AppRes;
int save_on_exit = 0;

/* local functions */
static void display_check(Display *dpy);
static Boolean insecure_warn(char *msg);
static Boolean ReallyQuit(XtPointer client_data);
static void SetDialogAnswer(Widget w, XtPointer client_data,
			    XtPointer call_data);
static void save_settings(void);
static void merge_geometry(Display *dpy, char *home);
static void save_geometry(void);

/* local data */
static XtWorkProcId wpid;
static Boolean dlg_done, dlg_answer;

static XtActionsRec Actions[] = {
    { "MainMenuPopup", MainMenuPopup },
    { "DoNewHost", DoNewHost },
    { "DoEditHost", DoEditHost },
    { "DoDeleteHost", DoDeleteHost },
    { "DoQuit", DoQuit },
};

static XtResource resources[] = {
    { "allowInsecureOperation", "AllowInsecureOperation",
      XtRBoolean, sizeof(Boolean),
      XtOffsetOf(PrivResRec, allow_insecure),
      XtRImmediate, (XtPointer) False },
    { "savePosition", "SavePosition",
      XtRBoolean, sizeof(Boolean),
      XtOffsetOf(PrivResRec, save_position),
      XtRImmediate, (XtPointer) True },
    { "saveDimensions", "SaveDimensions",
      XtRBoolean, sizeof(Boolean),
      XtOffsetOf(PrivResRec, save_dimensions),
      XtRImmediate, (XtPointer) True },
    { "responseFont", "ResponseFont",
      XtRString, sizeof(String),
      XtOffsetOf(PrivResRec, response_font),
      XtRString, NULL },
    { "initialSequence", "InitialSequence",
      XtRInt, sizeof(int),
      XtOffsetOf(PrivResRec, initial_seq),
      XtRImmediate, (XtPointer) 499 },
};

main(int argc, char *argv[])
{
    Display *dpy;
    Arg av[1];
    int ac;
    char *emsg;

    XtToolkitInitialize();
    app = XtCreateApplicationContext();
    XtAppSetFallbackResources(app, defres);
    XtAppAddActions(app, Actions, XtNumber(Actions));

    dpy = XtOpenDisplay(app, NULL, "xotpcalc", "XOTPCalc",
			NULL, 0, &argc, argv);
    if (dpy == NULL) {
	fprintf(stderr, "%s: Cannot open display\n", argv[0]);
	exit(1);
    }
    merge_geometry(dpy, getenv("HOME"));

    ac = 0;
    XtSetArg(av[ac], XtNmappedWhenManaged, False); ac++;
    top = XtAppCreateShell("xotpcalc", "XOTPCalc", applicationShellWidgetClass,
			   dpy, av, ac);
    XtGetApplicationResources(top, &AppRes, resources, XtNumber(resources),
			      NULL, 0);

    display_check(dpy);
    if (hosts_init(&emsg) < 0) {
	err_report(top, "Host list initialization failed, hosts_init reported:",
		   emsg);
	exit(1);
    }
    list_create();
    dialog_create();
    if (atexit(save_settings) == 0)
	save_on_exit = 1;
    XtRealizeWidget(top);
    list_init_geometry();
    XtMapWidget(top);
    XmProcessTraversal(hostlist, XmTRAVERSE_CURRENT);
    XtAppMainLoop(app);
}

/*
 * Simple no/yes question, with Cancel as the default.
 */
Boolean
ask_noyes(char *msg)
{
    Widget mb;
    Arg av[5];
    int ac;
    XmString s;

    ac = 0;
    s = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
    XtSetArg(av[ac], XmNmessageString, s); ac++;
    XtSetArg(av[ac], XmNtitle, "xotpcalc: Question"); ac++;
    XtSetArg(av[ac], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); ac++;
    mb = XmCreateQuestionDialog(top, "ask_yesno", av, ac);
    XmStringFree(s);
    XtUnmanageChild(XmMessageBoxGetChild(mb, XmDIALOG_HELP_BUTTON));

    XtAddCallback(mb, XmNokCallback, SetDialogAnswer, (XtPointer) 1);
    XtAddCallback(mb, XmNcancelCallback, SetDialogAnswer, (XtPointer) 0);

    XtManageChild(mb);
    XtPopup(XtParent(mb), XtGrabExclusive);

    dlg_done = dlg_answer = 0;
    while (!dlg_done)
	XtAppProcessEvent(app, XtIMAll);

    XtUnmanageChild(mb);
    XtPopdown(XtParent(mb));

    return dlg_answer;
}

/*
 * Report an error or warning.
 */
void
err_report(Widget w, char *msg, char *emsg)
{
    Widget mb;
    Arg av[5];
    int ac;
    char message[512];
    XmString s;

    ac = 0;
    sprintf(message, "%.376s\n\n%.128s", msg, emsg);
    s = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
    XtSetArg(av[ac], XmNmessageString, s); ac++;
    XtSetArg(av[ac], XmNtitle, "xotpcalc: Error"); ac++;
    mb = XmCreateErrorDialog(w, "err_report", av, ac);
    XmStringFree(s);
    s = XmStringCreateSimple("Dismiss");
    XtVaSetValues(mb, XmNokLabelString, s, NULL);
    XmStringFree(s);
    XtUnmanageChild(XmMessageBoxGetChild(mb, XmDIALOG_HELP_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(mb, XmDIALOG_CANCEL_BUTTON));

    XtAddCallback(mb, XmNokCallback, SetDialogAnswer, (XtPointer) 0);

    XtManageChild(mb);
    XtPopup(XtParent(mb), XtGrabExclusive);

    dlg_done = 0;
    while (!dlg_done)
	XtAppProcessEvent(app, XtIMAll);

    XtUnmanageChild(mb);
    XtPopdown(XtParent(mb));
}

/*
 * Start a clean exit from the program.
 */
void
DoQuit(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
    if (w != hostlist && XtParent(XtParent(w)) != top)
	return;
    XtDestroyWidget(top);
    wpid = XtAppAddWorkProc(app, ReallyQuit, NULL);
}

/*
 * Finish exiting.
 */
static Boolean
ReallyQuit(XtPointer client_data)
{
    XtRemoveWorkProc(wpid);
    XtDestroyApplicationContext(app);
    exit(0);

    return True;
}

/*
 * Check if the display is local.  The test is simplistic, as it doesn't
 * take into account the possibility of working over loopback or a local
 * interface address, but it should cover vastly more than 99% of expected
 * uses ;)  If you have an overwhelming reason for running X over loopback,
 * send me the patches, but I refuse to complicate the code for other
 * kinds of interfaces -- i.e., I want to stay away from SIOCGIFCONF and
 * the like.
 *
 * This check is no panacea, of course; for example, it will be gladly
 * fooled by a proxy running over an AF_UNIX socket.
 */
static void
display_check(Display *dpy)
{
    struct sockaddr sa;
    int salen = sizeof sa;
    Boolean proceed;

    getsockname(ConnectionNumber(dpy), &sa, &salen);
    if (sa.sa_family == AF_UNIX)
	return;
    proceed = insecure_warn(
	"The program is connected to a remote display.  This means that\n"
	"it is unsafe for entering sensitive information, such as passwords,\n"
	"as your keystrokes could be monitored and recorded unless the\n"
	"connection is encrypted (which is unlikely).");
    if (proceed)
	return;
    exit(1);
}

/*
 * Warn about an insecure condition.  If allow_insecure is set, the user
 * can proceed at her own risk, otherwise the dialog is formed as an
 * error notification and always returns false.
 */
static Boolean
insecure_warn(char *msg)
{
    Widget mb;
    Widget (*dlg_func)(Widget, String, ArgList, Cardinal);
    Arg av[5];
    int ac;
    char title[32];
    XmString s, ol;

    dlg_func = AppRes.allow_insecure ? XmCreateWarningDialog
				     : XmCreateErrorDialog;
    ac = 0;
    XtSetArg(av[ac], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); ac++;
    sprintf(title, "xotpcalc: %s",
	    AppRes.allow_insecure ? "Warning" : "Error");
    XtSetArg(av[ac], XmNtitle, title); ac++;
    mb = dlg_func(top, "insecure_warn", av, ac);
    XtUnmanageChild(XmMessageBoxGetChild(mb, XmDIALOG_HELP_BUTTON));

    XtAddCallback(mb, XmNcancelCallback, SetDialogAnswer, (XtPointer) 0);
    if (AppRes.allow_insecure)
	XtAddCallback(mb, XmNokCallback, SetDialogAnswer, (XtPointer) 1);

    ac = 0;
    ol = 0;
    s = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
    XtSetArg(av[ac], XmNmessageString, s); ac++;
    if (AppRes.allow_insecure) {
	ol = XmStringCreateSimple("Proceed");
	XtSetArg(av[ac], XmNokLabelString, ol); ac++;
    } else
	XtUnmanageChild(XmMessageBoxGetChild(mb, XmDIALOG_OK_BUTTON));
    XtSetValues(mb, av, ac);
    XmStringFree(s);
    if (ol)
	XmStringFree(ol);

    XtManageChild(mb);
    XtPopup(XtParent(mb), XtGrabExclusive);

    dlg_done = dlg_answer = 0;
    while (!dlg_done)
	XtAppProcessEvent(app, XtIMAll);

    XtUnmanageChild(mb);
    XtPopdown(XtParent(mb));

    return dlg_answer;
}

/*
 * Dialog callback.
 */
static void
SetDialogAnswer(Widget w, XtPointer client_data, XtPointer call_data)
{
    dlg_answer = (unsigned) client_data;
    dlg_done = True;
}

/*
 * Try to read saved geometry and merge it into our resources.
 */
static void
merge_geometry(Display *dpy, char *home)
{
    XrmDatabase dflt;
    char gf[] = "/.xotpcalc/geometry";
    char *fullname;

    if (!home)
	return;
    if ((fullname = (char *) malloc(strlen(home) + sizeof gf + 1)) == NULL)
	return;
    dflt = XtDatabase(dpy);
    strcpy(fullname, home);
    strcat(fullname, gf);
    XrmCombineFileDatabase(fullname, &dflt, True);
    free(fullname);
}

/*
 * Save host list and call the geometry-saving routine.
 */
static void
save_settings(void)
{
    char *emsg;

    if (!save_on_exit)
	return;
    if (hosts_write(&emsg) < 0)
	fprintf(stderr, "xotpcalc: error saving host table: %s\n", emsg);
    if (AppRes.save_position || AppRes.save_dimensions)
	save_geometry();
}

/*
 * Save top-level shell geometry.  We are already in ~/.xotpcalc at the
 * time this function is called.
 */
static void
save_geometry(void)
{
    Dimension x, y, w, h;
    FILE *gfp;

    if ((gfp = fopen("geometry", "w")) == NULL)
	return;
    get_shell_geometry(&x, &y, &w, &h);
    fprintf(gfp, "XOTPCalc.geometry: ");
    if (AppRes.save_dimensions)
	fprintf(gfp, "%dx%d", w, h);
    if (AppRes.save_position)
	fprintf(gfp, "+%d+%d", x, y);
    fputc('\n', gfp);
    fclose(gfp);
}
