/*
 * XOTPCalc, an S/Key and OPIE compatible one-time password calculator
 * Copyright (C) 1997 Ivan Nejgebauer <ian@uns.ns.ac.yu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Main dialog creation and handling.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include "xotpcalc.h"
#include "hosts.h"

/* global data */
Widget hostlist;

/* local functions */
static void make_item_visible(char *label, int nvisible);
static void ResizeHandler(Widget w, XtPointer closure, XEvent *ev,
			  Boolean *cont);
static void List_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void Calculate_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void ListKeyHandler(Widget w, XtPointer closure, XEvent *ev,
			  Boolean *cont);

/* local data */
static Widget frame, button;
static int selpos;
static Dimension lih;		/* list item height */
static Dimension fld;		/* frame/list delta */
static Dimension stw = 0;	/* stored top-level width */
static Dimension sth = 0;	/* stored top-level height */
static Dimension stx, sty;

#define SEARCH_MAX_DELTA 1000	/* msec between successive keys */

/*
 * Build the list and the top-level shell.
 */
void
list_create(void)
{
    Widget form, label;
    Arg av[15];
    int ac;
    XmString *s = NULL, sel = NULL;
    int i;

    if (h_count) {
	s = (XmString *) malloc(h_count * sizeof(XmString));
	if (!s) {
	    err_report(top, "list_create: XmString array creation failed",
		       "Out of memory");
	    exit(1);
	}
    }
    for (i = 0; i < h_count; i++) {
	s[i] = XmStringCreateLtoR(h_list[i]->name, XmSTRING_DEFAULT_CHARSET);
	if (h_list[i] == h_selected)
	    selpos = i + 1;
    }
    if (h_selected)
	sel = XmStringCreateLtoR(h_selected->name, XmSTRING_DEFAULT_CHARSET);

    ac = 0;
    XtSetArg(av[ac], XmNverticalSpacing, 4); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 4); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 2); ac++;
    form = XtCreateManagedWidget(
	"main_form", xmFormWidgetClass,
	top,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 2); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 2); ac++;
    frame = XtCreateManagedWidget(
	"main_frame", xmFrameWidgetClass,
	form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNchildType, XmFRAME_TITLE_CHILD); ac++;
    label = XtCreateManagedWidget(
	"Hosts", xmLabelWidgetClass,
	frame,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNitems, h_count ? s : NULL); ac++;
    XtSetArg(av[ac], XmNitemCount, h_count); ac++;
    XtSetArg(av[ac], XmNselectedItems, h_count ? &sel : NULL); ac++;
    XtSetArg(av[ac], XmNselectedItemCount, h_count ? 1 : 0); ac++;
    XtSetArg(av[ac], XmNlistSizePolicy, XmCONSTANT); ac++;
    XtSetArg(av[ac], XmNscrollBarDisplayPolicy, XmSTATIC); ac++;
    XtSetArg(av[ac], XmNselectionPolicy, XmBROWSE_SELECT); ac++;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, frame); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNleftWidget, frame); ac++;
    XtSetArg(av[ac], XmNvisibleItemCount, 9); ac++;
    XtSetArg(av[ac], XmNwidth, 150); ac++;
    hostlist = XmCreateScrolledList(frame, "main_list", av, ac);
    XtManageChild(hostlist);
    XtAddCallback(hostlist, XmNbrowseSelectionCallback, List_CB, NULL);
    XtAddCallback(hostlist, XmNdefaultActionCallback, List_CB, NULL);
    XtAddEventHandler(hostlist, KeyPressMask, False, ListKeyHandler, NULL);
    menu_create();
    if (h_count == 0)
	sensitize_edit_delete(False);

    ac = 0;
    XtSetArg(av[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNbottomOffset, 2); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 0); ac++;
    XtSetArg(av[ac], XmNmarginLeft, 10); ac++;
    XtSetArg(av[ac], XmNmarginRight, 10); ac++;
    XtSetArg(av[ac], XmNmarginTop, 2); ac++;
    XtSetArg(av[ac], XmNmarginBottom, 2); ac++;
    button = XtCreateManagedWidget(
	"Calculate", xmPushButtonWidgetClass,
	form,
	av, ac);
    XtVaSetValues(form, XmNdefaultButton, button, NULL);
    XtAddCallback(button, XmNactivateCallback, Calculate_CB, (XtPointer) 0);
    if (h_count == 0)
	XtSetSensitive(button, False);

    /* initialize resize processing */
    XtAddEventHandler(top, StructureNotifyMask, False, ResizeHandler, NULL);

    /* free up strings */
    for (i = 0; i < h_count; i++)
	XmStringFree(s[i]);
    if (sel)
	XmStringFree(sel);
}

/*
 * Calculate the values for proper handling of resize requests.  There
 * _ought_ to be a better way to find out what's needed than groveling
 * through the widgets' guts, or, still better, a more natural widget
 * layout.  Small wonder that people prefer Windows.  This code is too
 * ugly for words.
 */
void
list_init_geometry(void)
{
    Dimension fh;
    XmFontContext fctx;
    XmFontList fl;
    XmFontListEntry fle;
    XmFontType ft;
    XFontStruct *xfs;
    int nvi;
    XEvent ev;		/* for the call to ResizeHandler */

    XtVaGetValues(hostlist, XmNvisibleItemCount, &nvi, XmNfontList, &fl, NULL);
    XmFontListInitFontContext(&fctx, fl);
    fle = XmFontListNextEntry(fctx);
    xfs = (XFontStruct *) XmFontListEntryGetFont(fle, &ft);
    lih = xfs->ascent + xfs->descent + 3;
    XmFontListFreeFontContext(fctx);
    XtVaGetValues(frame, XmNheight, &fh, NULL);
    fld = fh - (nvi * lih);

    ev.type = ConfigureNotify;
    ResizeHandler(NULL, NULL, &ev, NULL);
}

/*
 * Return top-level shell geometry data to save_geometry().
 */
void
get_shell_geometry(Dimension *x, Dimension *y, Dimension *w, Dimension *h)
{
    *x = stx;
    *y = sty;
    *w = stw;
    *h = sth;
}

/*
 * Add a host to list.  Widget w is the parent for error messages.
 */
void add_host_to_list(Widget w, struct hostinfo *h)
{
    XmString s;
    int sel, ipos, nvi;
    char *emsg;

    sel = host_add(h, True, &emsg);
    if (sel < 0) {
	err_report(w, "Host insertion failed, host_add reported:", emsg);
	return;
    }
    selpos = sel;
    ipos = (sel == h_count) ? 0 : sel;
    s = XmStringCreateLtoR(h->name, XmSTRING_DEFAULT_CHARSET);
    XmListAddItems(hostlist, &s, 1, ipos);
    XmStringFree(s);
    make_item_visible(h->name, 0);
    if (h_count == 1) {
	sensitize_edit_delete(True);
	XtSetSensitive(button, True);
    }
}

/*
 * Delete a host.
 */
void DoDeleteHost(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
    char msg[100];
    XmString s;
    int sel, nvi;
    char *emsg;

    if (w != hostlist && XtParent(XtParent(w)) != top)
	return;
    if (h_count == 0)
	return;
    sprintf(msg, "Delete host \"%.80s\"?", h_selected->name);
    if (!ask_noyes(msg))
	return;
    s = XmStringCreateLtoR(h_selected->name, XmSTRING_DEFAULT_CHARSET);
    XmListDeleteItems(hostlist, &s, 1);
    XmStringFree(s);
    sel = host_delete(h_selected->name, &emsg);
    if (sel < 0) {
	err_report(top, "Host deletion failed, host_delete reported:", emsg);
	save_on_exit = 0;
	return;
    }
    if (sel == 0) {
	sensitize_edit_delete(False);
	XtSetSensitive(button, False);
    } else {
	selpos = sel;
	make_item_visible(h_selected->name, 0);
    }
}

/*
 * Resize processing.
 */
static void
ResizeHandler(Widget w, XtPointer closure, XEvent *ev, Boolean *cont)
{
    Dimension fh, tw, th, tb, bw, bh;
    int nnvi, nvi;

    if (ev->type != ConfigureNotify)
	return;
    XtVaGetValues(top, XmNwidth, &tw, XmNheight, &th, XmNborderWidth, &tb,
		  XmNx, &stx, XmNy, &sty, NULL);
    XtVaGetValues(frame, XmNheight, &fh, NULL);
    XtVaGetValues(hostlist, XmNvisibleItemCount, &nvi, NULL);
    XtVaGetValues(button, XmNwidth, &bw, XmNheight, &bh, NULL);
    if (tw == stw && th == sth)
	return;
    nnvi = (th - bh - 4 - fld) / lih;
    if (nnvi > 0) {
	XtVaSetValues(hostlist, XmNvisibleItemCount, nnvi, NULL);
	fh += (nnvi - nvi) * (int) lih;
	nvi = nnvi;
    }
    make_item_visible(NULL, nvi);
    XtVaSetValues(button,
		  XmNleftPosition, 50 * (tw - bw) / tw,
		  XmNbottomOffset, (th - fh - bh) / 2,
		  NULL);
    XtResizeWidget(top, tw, th, tb);
    stw = tw;
    sth = th;
}

/*
 * (Optionally) select a list item and make it visible in the list.
 * If label is NULL, we don't need to select the item; if nvisible is
 * 0, we must determine the actual number of visible items.
 */
static void
make_item_visible(char *label, int nvisible)
{
    XmString s;
    int nvi;

    if (nvisible == 0)
	XtVaGetValues(hostlist, XmNvisibleItemCount, &nvi, NULL);
    else
	nvi = nvisible;
    if (label) {
	s = XmStringCreateLtoR(label, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(hostlist, XmNselectedItems, &s, XmNselectedItemCount, 1,
		      NULL);
	XmStringFree(s);
    }
    if (selpos > nvi)
	XmListSetPos(hostlist, selpos - nvi / 2);
    else
	XmListSetPos(hostlist, 1);
}
    
/*
 * List callback.
 */
static void
List_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    XmListCallbackStruct *lcb = (XmListCallbackStruct *) call_data;
    unsigned cdata = 0;

    switch (lcb->reason) {
	case XmCR_BROWSE_SELECT:
	    selpos = lcb->item_position;
	    h_selected = h_list[selpos - 1];
	    break;
	case XmCR_DEFAULT_ACTION:
	    if (lcb->event && lcb->event->type == KeyPress)
		cdata = 1;
	    Calculate_CB(w, (XtPointer) cdata, NULL);
	    break;
    }
}

/*
 * Calculate button callback.
 */
static void
Calculate_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    static Boolean ignore_next = False;
    unsigned ignorep = (unsigned) client_data;

    if (ignore_next) {
	ignore_next = False;
	return;
    }
    if (ignorep)
	ignore_next = True;
    do_calculate();
}

/*
 * KeyPress event handler, implements interactive search.
 */
static void
ListKeyHandler(Widget w, XtPointer closure, XEvent *ev, Boolean *cont)
{
    char c;
    KeySym ks;
    XKeyEvent *kev = (XKeyEvent *) ev;
    Time delta;
    int sel;
    static char sstr[16];
    static int spos = 0;
    static Boolean last_not_found = False;
    static Time last_key_time;

    if (kev->state != 0 && kev->state != ShiftMask)
	return;
    if (XLookupString(kev, &c, 1, &ks, NULL) != 1)
	return;
    if (!isalnum(c) && c != '.' && c != '-')
	return;
    delta = kev->time - last_key_time;
    last_key_time = kev->time;
    if (delta > SEARCH_MAX_DELTA) {
	last_not_found = False;
	spos = 0;
    }
    else if (spos == sizeof sstr - 1) {
	XBell(XtDisplay(top), 0);
	return;
    }
    sstr[spos++] = tolower(c);
    sstr[spos] = '\0';
    if (spos > 1 && h_selected && strncmp(h_selected->name, sstr, spos) == 0)
	return;
    if (last_not_found || (sel = host_partial_lookup(sstr)) == -1) {
	last_not_found = True;
	XBell(XtDisplay(top), 0);
	return;
    }
    h_selected = h_list[sel];
    selpos = sel + 1;
    make_item_visible(h_selected->name, 0);
}
