/*
 * XOTPCalc, an S/Key and OPIE compatible one-time password calculator
 * Copyright (C) 1997 Ivan Nejgebauer <ian@uns.ns.ac.yu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Host table and array manipulation functions.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "hosts.h"

extern int errno;

/* global data */
struct hostinfo **h_list = NULL;
struct hostinfo *h_selected = NULL;
int h_count = 0;

/* local functions */
static void hosts_sort(void);
static int hosts_cmp(const void *h1, const void *h2);
static int hosts_pcmp(const void *h1, const void *h2);

/* local data */
static int h_init = 1;
static int h_dupes = 0;
static char err_string[128];
static int partial_size;

/*
 * Read the host table from ~/.xotpcalc/hosts, create ~/.xotpcalc if
 * necessary and chdir() there.  Returns 0 on success, -1 on error.
 */
int
hosts_init(char **err_return)
{
    FILE *hfp;
    char *s, *t, *u;
    char line[256];
    struct hostinfo h;
    int lineno = 0;

    *err_return = err_string;
    if ((s = getenv("HOME")) == NULL) {
	strcpy(err_string, "HOME environment variable undefined");
	return -1;
    }
    if (chdir(s) < 0) {
	sprintf(err_string, "Error changing to home directory: %.90s",
		strerror(errno));
	return -1;
    }
    if (chdir(".xotpcalc") < 0) {
	if (mkdir(".xotpcalc", 0700) < 0) {
	    sprintf(err_string, "Error creating ~/.xotpcalc: %.90s",
		    strerror(errno));
	    return -1;
	}
	if (chdir(".xotpcalc") < 0) {
	    sprintf(err_string, "Error changing to ~/.xotpcalc: %.90s",
		    strerror(errno));
	    return -1;
	}
    }
    if ((hfp = fopen("hosts", "r")) == NULL) {
	h_init = 0;
	return 0;
    }
    while (fgets(line, sizeof line, hfp) == line) {
	int selected = 0;

	lineno++;
	if (line[0] == '#' || line[0] == '\n')
	    continue;

	/* hostname */
	if ((s = strtok(line, " \t\n")) == NULL) {
	    sprintf(err_string, "Malformed entry in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}
	if (*s == '*') {
	    selected = 1;
	    s++;
	}
	h.name = s;
	for (t = u = h.name; *t; t++)
	    if (isalnum(*t) || *t == '-' || *t == '.')
		*u++ = tolower(*t);
	*u = '\0';
	if (u == h.name) {
	    sprintf(err_string, "Empty name in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}

	/* algorithm */
	if ((s = strtok(NULL, " \t\n")) == NULL) {
	    sprintf(err_string, "Malformed entry in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}
	if (strcmp(s, "md4") == 0)
	    h.algorithm = MD4;
	else if (strcmp(s, "md5") == 0)
	    h.algorithm = MD5;
	else {
	    sprintf(err_string, "Invalid algorithm specified "
		    "in ~/.xotpcalc/hosts, line %d", lineno);
	    return -1;
	}

	/* sequence */
	if ((s = strtok(NULL, " \t\n")) == NULL) {
	    sprintf(err_string, "Malformed entry in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}
	if ((h.sequence = atoi(s)) < 0) {
	    sprintf(err_string, "Invalid sequence in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}
	h.sequence %= 10000;

	/* seed */
	if ((s = strtok(NULL, " \t\n")) == NULL) {
	    sprintf(err_string, "Malformed entry in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}
	h.seed = s;
	for (t = u = h.seed; *t; t++)
	    if (isprint(*t) && *t != ' ')
		*u++ = *t;
	*u = '\0';
	if (u == h.seed) {
	    sprintf(err_string, "Empty seed in ~/.xotpcalc/hosts, "
		    "line %d", lineno);
	    return -1;
	}

	if (host_add(&h, selected, err_return) < 0)
	    return -1;
    }
    fclose(hfp);
    h_init = 0;
    hosts_sort();
    if (h_dupes) {
	sprintf(err_string, "Duplicate hosts found in ~/.xotpcalc/hosts");
	return -1;
    }
    if (h_count && !h_selected)
	h_selected = h_list[0];
    return 0;
}

/*
 * Add a host to array.  If selected is set, copy the pointer to hostinfo
 * structure to h_selected.  Returns host index (or 0 during initialization)
 * on success, -1 on error.
 */
int
host_add(struct hostinfo *h, int selected, char **err_return)
{
    struct hostinfo *g;
    void *t;

    *err_return = err_string;
    if (!h_init && host_lookup(h->name) >= 0) {
	sprintf(err_string, "Host already exists");
	return -1;
    }
    h_count++;
    t = h_list ? realloc(h_list, h_count * sizeof(struct hostinfo *))
	       : malloc(sizeof(struct hostinfo *));
    if (!t) {
	sprintf(err_string, "Out of memory");
	return -1;
    }
    h_list = (struct hostinfo **) t;
    t = malloc(sizeof(struct hostinfo));
    if (!t) {
	sprintf(err_string, "Out of memory");
	return -1;
    }
    *((struct hostinfo *) t) = *h;
    h_list[h_count - 1] = g = t;
    if (selected)
	h_selected = t;
    if ((g->name = (char *) malloc(strlen(h->name) + 1)) == NULL) {
	sprintf(err_string, "Out of memory");
	return -1;
    }
    strcpy(g->name, h->name);
    if ((g->seed = (char *) malloc(strlen(h->seed) + 1)) == NULL) {
	sprintf(err_string, "Out of memory");
	return -1;
    }
    strcpy(g->seed, h->seed);
    if (h_init)
	return 0;
    hosts_sort();
    return host_lookup(h->name) + 1;
}

/*
 * Delete a host from array, moving the selection and rearranging the array
 * as necessary.  Returns the index of the selected item after deletion
 * (incremented by 1) on success, -1 on error.
 */
int
host_delete(char *name, char **err_return)
{
    int h_index;
    void *t;

    *err_return = err_string;
    if (!h_count) {
	sprintf(err_string, "Internal inconsistency: empty array");
	return -1;
    }
    if ((h_index = host_lookup(name)) < 0) {
	sprintf(err_string, "Internal inconsistency: host not found");
	return -1;
    }
    if (h_list[h_index] == h_selected)	/* must move selection */
	if (h_index > 0)
	    if (h_index == h_count - 1)
		h_selected = h_list[h_index - 1];
	    else
		h_selected = h_list[h_index + 1];
	else if (h_count > 1)
	    h_selected = h_list[1];
	else
	    h_selected = NULL;
    free(h_list[h_index]->name);
    free(h_list[h_index]->seed);
    free(h_list[h_index]);
    if (h_index < h_count - 1)		/* must rearrange */
	memmove(h_list + h_index, h_list + h_index + 1,
		(h_count - h_index - 1) * sizeof(struct hostinfo *));
    if (--h_count)
	t = realloc(h_list, h_count * sizeof(struct hostinfo *));
    else {
	free(h_list);
	h_list = NULL;
    }
    if (h_count && !t) {
	sprintf(err_string, "Error reallocating memory");
	return -1;
    }
    if (h_count)
	h_list = (struct hostinfo **) t;
    return h_count ? host_lookup(h_selected->name) + 1 : 0;
}

/*
 * Look up a host by its name.  Returns index to h_list where the hostinfo
 * pointer is, or -1 on error.
 */
int
host_lookup(char *name)
{
    struct hostinfo h;
    struct hostinfo *t;

    if (!h_count)
	return -1;
    h.name = name;
    t = &h;
    t = (struct hostinfo *)
	bsearch(&t, h_list, h_count, sizeof(struct hostinfo *), hosts_cmp);
    return t ? ((char *) t - (char *) h_list) / sizeof(struct hostinfo *)
	     : -1;
}

/*
 * Do a partial host lookup, i.e., find the first host in the array
 * with the start of its name matching the lookup string.  Returns -1
 * if nothing is found.
 */
int
host_partial_lookup(char *part)
{
    struct hostinfo h;
    struct hostinfo *t;
    int hpos;

    if (!h_count)
	return -1;
    h.name = part;
    partial_size = strlen(part);
    t = &h;
    t = (struct hostinfo *)
	bsearch(&t, h_list, h_count, sizeof(struct hostinfo *), hosts_pcmp);
    if (!t)
	return -1;
    hpos = ((char *) t - (char *) h_list) / sizeof(struct hostinfo *);
    while (hpos > 0 && strncmp(h_list[hpos - 1]->name, part, partial_size) == 0)
	hpos--;
    return hpos;
}

/*
 * Rewrite the host table.  Returns 0 on success, -1 on error.
 */
int
hosts_write(char **err_return)
{
    FILE *hfp;
    int i;

    *err_return = err_string;
    if ((hfp = fopen("hosts", "w")) == NULL) {
	sprintf(err_string, "Error opening ~/.xotpcalc/hosts for writing: "
		"%.80s", strerror(errno));
	return -1;
    }
    fprintf(hfp,
	"# OTP hosts.  This file is read on startup and rewritten on exit.\n"
	"# Fields are hostname, algorithm, sequence, seed.  An asterisk in\n"
	"# front of a hostname means that the host will be selected in the\n"
	"# main window list when the program starts.\n\n");
    for (i = 0; i < h_count; i++)
	fprintf(hfp, "%s%.64s %s %d %.30s\n",
		h_list[i] == h_selected ? "*" : "",
		h_list[i]->name,
		h_list[i]->algorithm == MD4 ? "md4" : "md5",
		h_list[i]->sequence,
		h_list[i]->seed);
    fclose(hfp);
    return 0;
}

/*
 * Sort the host array.
 */
static void
hosts_sort(void)
{
    if (h_count < 2)
	return;
    qsort(h_list, h_count, sizeof(struct hostinfo *), hosts_cmp);
}

/*
 * Comparison function for qsort() and bsearch().  Sets h_dupes to 1 if
 * h_init is set and any two names compare equal; this is used during
 * initialization to detect duplicates, and is otherwise ignored.
 */
static int
hosts_cmp(const void *h1, const void *h2)
{
    int result;

    result = strcmp((*((struct hostinfo **) h1))->name,
		    (*((struct hostinfo **) h2))->name);
    if (h_init && result == 0)
	h_dupes = 1;
    return result;
}

/*
 * Partial name lookup.  We use strncmp instead of strcmp, and a static
 * variable holds the length to avoid excessive strlens.
 */
static int
hosts_pcmp(const void *h1, const void *h2)
{
    return strncmp((*((struct hostinfo **) h1))->name,
		   (*((struct hostinfo **) h2))->name,
		   partial_size);
}
