/*
 * XOTPCalc, an S/Key and OPIE compatible one-time password calculator
 * Copyright (C) 1997 Ivan Nejgebauer <ian@uns.ns.ac.yu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * The guts.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include "xotpcalc.h"
#include "hosts.h"
#include "mdglobal.h"
#include "md4.h"
#include "md5.h"

/* from btoe.c */
char *opiebtoe(char *engout, char *c);

/* local functions */
static void add_or_edit_host(void);
static void fill_host_data(struct hostinfo *h);
static void update_host_data(struct hostinfo *h);
static void set_name_editable(Boolean editable);
static void PFocusChange(Widget w, XtPointer client_data, XtPointer call_data);
static Boolean PRemoveFocus(XtPointer client_data);
static void PVerify(Widget w, XtPointer client_data, XtPointer call_data);
static void PActivate(Widget w, XtPointer client_data, XtPointer call_data);
static void FVerify(Widget w, XtPointer client_data, XtPointer call_data);
static void Host_Exit_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void Calc_Exit_CB(Widget w, XtPointer client_data, XtPointer call_data);
static Boolean RetainFocus(XtPointer client_data);
static void Form_Unmap_CB(Widget w, XtPointer client_data, XtPointer call_data);

/* local data */
static char passphrase[64];
static int passlen;
static XtWorkProcId wpid;

static Boolean dlg_done, dlg_answer;
static Boolean adding_host;

enum { ERR_NHFOUND, ERR_NHEMPTY, ERR_NSEMPTY, ERR_CSEMPTY };
enum { F_NAME, F_SEQ, F_SEED };

static Widget shell;
static Widget form;

static struct {
    Widget frame;
    Widget frame_label;
    Widget form;
    Widget name_label;
    Widget edit_label;
    Widget name_field;
    Widget md4, md5;
    Widget sequence;
    Widget seed;
} host;

static struct {
    Widget frame;
    Widget pass_field;
} pass;

static struct {
    Widget frame;
    Widget resp_label;
} resp;

static struct {
    Widget ok_pb;
    Widget cancel_pb;
} buttons;

/*
 * Create a new host entry (bound to Alt+N).
 */
void
DoNewHost(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
    if (w != hostlist && XtParent(XtParent(w)) != top)
	return;
    adding_host = True;
    add_or_edit_host();
}

/*
 * Edit an entry (bound to Alt+E).
 */
void
DoEditHost(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
    if (w != hostlist && XtParent(XtParent(w)) != top)
	return;
    if (h_count == 0)
	return;
    adding_host = False;
    add_or_edit_host();
}

/*
 * The Calculate dialog.  Called from the Calculate button callback.
 */
void
do_calculate(void)
{
    Arg av[15];
    int ac;
    XmString s;

    XtVaSetValues(shell, XmNtitle, "xotpcalc: Calculate", NULL);
    fill_host_data(h_selected);

    XtManageChild(host.frame_label);
    XtManageChild(pass.frame);
    XtManageChild(resp.frame);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, resp.frame); ac++;
    XtSetArg(av[ac], XmNtopOffset, 4); ac++;
    XtSetArg(av[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNbottomOffset, 4); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 35); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNrightPosition, 65); ac++;
    XtSetArg(av[ac], XmNmarginTop, 2); ac++;
    XtSetArg(av[ac], XmNmarginBottom, 2); ac++;
    buttons.ok_pb = XtCreateManagedWidget(
	"Leave", xmPushButtonWidgetClass,
	form,
	av, ac);
    XtAddCallback(buttons.ok_pb, XmNactivateCallback, Calc_Exit_CB, NULL);

    s = XmStringCreateLtoR(" ", XmSTRING_DEFAULT_CHARSET);
    XtVaSetValues(resp.resp_label, XmNlabelString, s, NULL);
    XmStringFree(s);

    XtManageChild(form);
    set_name_editable(False);
    XmProcessTraversal(pass.pass_field, XmTRAVERSE_CURRENT);
    XtPopup(shell, XtGrabExclusive);

    dlg_done = dlg_answer = False;
    while (!dlg_done)
	XtAppProcessEvent(app, XtIMAll);
    if (dlg_answer)
	update_host_data(h_selected);

    XtUnmanageChild(form);
    XtPopdown(shell);
}

/*
 * Build all dialog elements except buttons.  New, Edit and Calculate
 * dialogs have a lot in common and use the same basic layout.
 */
void
dialog_create(void)
{
    Arg av[15];
    int ac;
    Widget label, rowcol;
    XmFontList fl;
    XmFontListEntry fle;

    ac = 0;
    XtSetArg(av[ac], XmNallowShellResize, False); ac++;
    XtSetArg(av[ac], XmNmappedWhenManaged, False); ac++;
    XtSetArg(av[ac], XmNdeleteResponse, XmUNMAP); ac++;
    shell = XmCreateDialogShell(top, "calc_shell", av, ac);
    
    ac = 0;
    XtSetArg(av[ac], XmNverticalSpacing, 4); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 4); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 2); ac++;
    XtSetArg(av[ac], XmNautoUnmanage, False); ac++;
    form = XtCreateWidget(
	"calc_form", xmFormWidgetClass,
	shell,
	av, ac);
    XtAddCallback(form, XmNunmapCallback, Form_Unmap_CB, NULL);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 2); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 2); ac++;
    host.frame = XtCreateManagedWidget(
	"calc_host_frame", xmFrameWidgetClass,
	form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNverticalSpacing, 0); ac++;
    host.form = XtCreateManagedWidget(
	"calc_host_form", xmFormWidgetClass,
	host.frame,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNchildType, XmFRAME_TITLE_CHILD); ac++;
    host.frame_label = XtCreateManagedWidget(
	"Host", xmLabelWidgetClass,
	host.frame,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 22); ac++;
    XtSetArg(av[ac], XmNcolumns, 30); ac++;
    host.name_field = XtCreateManagedWidget(
	"calc_host_name", xmTextFieldWidgetClass,
	host.form,
	av, ac);
    XtAddCallback(host.name_field, XmNmodifyVerifyCallback, FVerify,
		  (XtPointer) F_NAME);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNtopOffset, 8); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNrightPosition, 22); ac++;
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_END); ac++;
    host.name_label = XtCreateManagedWidget(
	"Name:", xmLabelWidgetClass,
	host.form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNtopOffset, 8); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 22); ac++;
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_END); ac++;
    host.edit_label = XtCreateManagedWidget(
	"calc_edit_label", xmLabelWidgetClass,
	host.form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNradioBehavior, True); ac++;
    XtSetArg(av[ac], XmNorientation, XmHORIZONTAL); ac++;
    XtSetArg(av[ac], XmNnumColumns, 1); ac++;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.name_field); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 22); ac++;
    rowcol = XtCreateManagedWidget(
	"calc_host_algrc", xmRowColumnWidgetClass,
	host.form,
	av, ac);

    host.md4 = XtCreateManagedWidget(
	"MD4", xmToggleButtonWidgetClass,
	rowcol,
	NULL, 0);

    host.md5 = XtCreateManagedWidget(
	"MD5", xmToggleButtonWidgetClass,
	rowcol,
	NULL, 0);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.name_field); ac++;
    XtSetArg(av[ac], XmNtopOffset, 7); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNrightWidget, rowcol); ac++;
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_END); ac++;
    label = XtCreateManagedWidget(
	"Algorithm:", xmLabelWidgetClass,
	host.form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, rowcol); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 22); ac++;
    XtSetArg(av[ac], XmNcolumns, 5); ac++;
    XtSetArg(av[ac], XmNmaxLength, 4); ac++;
    host.sequence = XtCreateManagedWidget(
	"calc_host_sequence", xmTextFieldWidgetClass,
	host.form,
	av, ac);
    XtAddCallback(host.sequence, XmNmodifyVerifyCallback, FVerify,
		  (XtPointer) F_SEQ);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, rowcol); ac++;
    XtSetArg(av[ac], XmNtopOffset, 7); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNrightWidget, host.sequence); ac++;
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_END); ac++;
    label = XtCreateManagedWidget(
	"Sequence:", xmLabelWidgetClass,
	host.form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.sequence); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 22); ac++;
    XtSetArg(av[ac], XmNcolumns, 10); ac++;
    host.seed = XtCreateManagedWidget(
	"calc_host_seed", xmTextFieldWidgetClass,
	host.form,
	av, ac);
    XtAddCallback(host.seed, XmNmodifyVerifyCallback, FVerify,
		  (XtPointer) F_SEED);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.sequence); ac++;
    XtSetArg(av[ac], XmNtopOffset, 7); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNrightWidget, host.seed); ac++;
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_END); ac++;
    label = XtCreateManagedWidget(
	"Seed:", xmLabelWidgetClass,
	host.form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.frame); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 2); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 1); ac++;
    pass.frame = XtCreateManagedWidget(
	"calc_pass_frame", xmFrameWidgetClass,
	form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNchildType, XmFRAME_TITLE_CHILD); ac++;
    label = XtCreateManagedWidget(
	"Passphrase", xmLabelWidgetClass,
	pass.frame,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNmaxLength, 63); ac++;
    pass.pass_field = XtCreateManagedWidget(
	"calc_pass_pass", xmTextFieldWidgetClass,
	pass.frame,
	av, ac);
    XtAddCallback(pass.pass_field, XmNfocusCallback, PFocusChange, NULL);
    XtAddCallback(pass.pass_field, XmNlosingFocusCallback, PFocusChange, NULL);
    XtAddCallback(pass.pass_field, XmNmodifyVerifyCallback, PVerify, NULL);
    XtAddCallback(pass.pass_field, XmNactivateCallback, PActivate, NULL);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, pass.frame); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNmarginWidth, 2); ac++;
    XtSetArg(av[ac], XmNmarginHeight, 2); ac++;
    resp.frame = XtCreateManagedWidget(
	"calc_resp_frame", xmFrameWidgetClass,
	form,
	av, ac);

    ac = 0;
    XtSetArg(av[ac], XmNchildType, XmFRAME_TITLE_CHILD); ac++;
    label = XtCreateManagedWidget(
	"Response", xmLabelWidgetClass,
	resp.frame,
	av, ac);

    ac = 0;
    fle = XmFontListEntryLoad(XtDisplay(top), AppRes.response_font,
			      XmFONT_IS_FONT, XmFONTLIST_DEFAULT_TAG);
    if (fle) {
	fl = XmFontListAppendEntry(NULL, fle);
	if (fl)
	    XtSetArg(av[ac], XmNfontList, fl); ac++;
	XmFontListEntryFree(&fle);
    }
    XtSetArg(av[ac], XmNalignment, XmALIGNMENT_CENTER); ac++;
    resp.resp_label = XtCreateManagedWidget(
	"calc_resp_label", xmLabelWidgetClass,
	resp.frame,
	av, ac);
}

/*
 * The actual New/Edit dialog.
 */
static void
add_or_edit_host(void)
{
    Arg av[15];
    int ac;

    if (adding_host) {
	XtVaSetValues(shell, XmNtitle, "xotpcalc: New Host", NULL);
	fill_host_data(NULL);
    } else {
	XtVaSetValues(shell, XmNtitle, "xotpcalc: Edit Host", NULL);
	fill_host_data(h_selected);
    }

    XtUnmanageChild(host.frame_label);
    XtUnmanageChild(pass.frame);
    XtUnmanageChild(resp.frame);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.frame); ac++;
    XtSetArg(av[ac], XmNtopOffset, 4); ac++;
    XtSetArg(av[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNbottomOffset, 4); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 3); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNrightPosition, 33); ac++;
    XtSetArg(av[ac], XmNmarginTop, 2); ac++;
    XtSetArg(av[ac], XmNmarginBottom, 2); ac++;
    buttons.ok_pb = XtCreateManagedWidget(
	"OK", xmPushButtonWidgetClass,
	form,
	av, ac);
    XtAddCallback(buttons.ok_pb, XmNactivateCallback, Host_Exit_CB,
		  (XtPointer) 1);
    XtVaSetValues(form, XmNdefaultButton, buttons.ok_pb, NULL);

    ac = 0;
    XtSetArg(av[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
    XtSetArg(av[ac], XmNtopWidget, host.frame); ac++;
    XtSetArg(av[ac], XmNtopOffset, 4); ac++;
    XtSetArg(av[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
    XtSetArg(av[ac], XmNbottomOffset, 4); ac++;
    XtSetArg(av[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNleftPosition, 67); ac++;
    XtSetArg(av[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
    XtSetArg(av[ac], XmNrightPosition, 97); ac++;
    XtSetArg(av[ac], XmNmarginTop, 2); ac++;
    XtSetArg(av[ac], XmNmarginBottom, 2); ac++;
    buttons.cancel_pb = XtCreateManagedWidget(
	"Cancel", xmPushButtonWidgetClass,
	form,
	av, ac);
    XtAddCallback(buttons.cancel_pb, XmNactivateCallback, Host_Exit_CB,
		  (XtPointer) 0);
    XtVaSetValues(form, XmNcancelButton, buttons.cancel_pb, NULL);

    XtManageChild(form);
    if (adding_host)
	set_name_editable(True);
    else
	set_name_editable(False);
    XmProcessTraversal(adding_host ? host.name_field : host.sequence,
		       XmTRAVERSE_CURRENT);
    XtPopup(shell, XtGrabExclusive);

    dlg_answer = dlg_done = False;
    while (!dlg_done)
	XtAppProcessEvent(app, XtIMAll);
    if (dlg_answer)
	update_host_data(adding_host ? NULL : h_selected);

    XtUnmanageChild(form);
    XtPopdown(shell);
}

/*
 * Initialize the dialog.
 */
static void
fill_host_data(struct hostinfo *h)
{
    char sseq[15];

    if (!h) {
	sprintf(sseq, "%d", abs(AppRes.initial_seq) % 10000);
	XmTextFieldSetString(host.name_field, "");
	XmTextFieldSetString(host.sequence, sseq);
	XmTextFieldSetString(host.seed, "");
	XmToggleButtonSetState(host.md5, True, True);
	XtVaSetValues(host.sequence, XmNcursorPosition, strlen(sseq), NULL);
	return;
    }
    XmTextFieldSetString(host.name_field, h->name);
    sprintf(sseq, "%d", h->sequence);
    XmTextFieldSetString(host.sequence, sseq);
    XmTextFieldSetString(host.seed, h->seed);
    switch (h->algorithm) {
	case MD4:
	    XmToggleButtonSetState(host.md4, True, True);
	    break;
	case MD5:
	    XmToggleButtonSetState(host.md5, True, True);
	    break;
    }
    XtVaSetValues(host.sequence, XmNcursorPosition, strlen(sseq), NULL);
    XtVaSetValues(host.seed, XmNcursorPosition, strlen(h->seed), NULL);
}

/*
 * Update an existing or add a new host entry, using the form data.
 */
static void
update_host_data(struct hostinfo *h)
{
    struct hostinfo nh;
    char *fname, *fseq, *fseed;
    int alg, seq;

    fname = XmTextFieldGetString(host.name_field);
    fseed = XmTextFieldGetString(host.seed);
    fseq = XmTextFieldGetString(host.sequence);
    seq = abs(atoi(fseq)) % 10000;
    alg = XmToggleButtonGetState(host.md4) ? MD4 : MD5;
    if (h) {	/* update; don't touch name */
	if (h->seed)
	    free(h->seed);
	h->seed = (char *) malloc(strlen(fseed) + 1);
	if (h->seed)
	    strcpy(h->seed, fseed);
	h->sequence = seq;
	h->algorithm = alg;
    } else {
	nh.name = fname;
	nh.algorithm = alg;
	nh.sequence = seq;
	nh.seed = fseed;
	add_host_to_list(shell, &nh);
    }
    XtFree(fname);
    XtFree(fseq);
    XtFree(fseed);
}

/*
 * We only allow the user to change the host name in the New dialog.
 * This function also resizes the shell to the same width every time
 * it is realized.  (Maybe someday I'll understand how Motif _really_
 * works.)
 */
static void
set_name_editable(Boolean editable)
{
    Dimension sh, sb;
    XmString s;
    char *t;
    static Dimension sw;	/* saved width */
    static int inited = 0;

    if (!inited) {
	XtVaGetValues(shell, XmNwidth, &sw, NULL);
	XtVaSetValues(shell, XmNmaxWidth, sw, NULL);
	inited = 1;
    }
    XtVaGetValues(shell, XmNheight, &sh, XmNborderWidth, &sb, NULL);
    if (editable) {
	XtUnmanageChild(host.edit_label);
	XtManageChild(host.name_field);
	XtResizeWidget(shell, sw, sh, sb);
    } else {
	t = XmTextFieldGetString(host.name_field);
	s = XmStringCreateLtoR(t, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(host.edit_label, XmNlabelString, s, NULL);
	XtFree(t);
	XmStringFree(s);
	XtManageChild(host.edit_label);
	XtUnmanageChild(host.name_field);
	XtResizeWidget(shell, sw, sh, sb);
    }
}

/*
 * Wipe out the Passphrase field (and the actual value) whenever the
 * field loses or gains input focus.  If we're gaining focus, try to
 * grab the keyboard (and bail out if unsuccessful).  Remove the grab
 * when we lose focus.
 */
static void
PFocusChange(Widget w, XtPointer client_data, XtPointer call_data)
{
    /* not strictly true but shouldn't matter */
    XmAnyCallbackStruct *cbs = (XmAnyCallbackStruct *) call_data;
    int gr;

    memset(passphrase, '\0', sizeof passphrase);
    passlen = 0;
    XmTextFieldSetString(pass.pass_field, "");
    switch (cbs->reason) {
	case XmCR_FOCUS:
	    gr = XGrabKeyboard(XtDisplay(shell), XtWindow(shell),
			       False,
			       GrabModeAsync,
			       GrabModeAsync,
			       CurrentTime);
	    if (gr != GrabSuccess) {
		err_report(shell,
	"The program has failed to grab (i.e., take exclusive control of)\n"
	"the keyboard.  This means that any other program running on\n"
	"your display could record the keystrokes if it wished to do so.\n"
	"As you are about to enter a supposedly secret password, it\n"
	"would be inadvisable to permit that even as a possibility.",
	"(Maybe some xterm has \"Secure Keyboard\" turned on?)");
		wpid = XtAppAddWorkProc(app, PRemoveFocus, NULL);
	    }
	    break;
	case XmCR_LOSING_FOCUS:
	    XUngrabKeyboard(XtDisplay(shell), CurrentTime);
	    break;
    }
}

/*
 * Remove focus from the Passphrase field.
 */
static Boolean
PRemoveFocus(XtPointer client_data)
{
    XtRemoveWorkProc(wpid);
    XmProcessTraversal(buttons.ok_pb, XmTRAVERSE_CURRENT);
    return True;
}

/*
 * Process passphrase modifications.  Briefly: only single-character input
 * is allowed, all input characters are displayed as asterisks.
 */
static void
PVerify(Widget w, XtPointer client_data, XtPointer call_data)
{
    XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) call_data;
    int c, i;

    if (!cbs->event) {			/* programmatic deletion */
	if (cbs->text->length != 0)
	    cbs->doit = False;
	return;
    }
    if (cbs->event->xkey.send_event) {	/* no synthetic events */
	cbs->doit = False;
	return;
    }
    if (cbs->text->length > 0) {	/* insertion */
	if (cbs->text->length > 1) {
	    cbs->doit = False;
	    return;
	}
	c = *cbs->text->ptr;
	*cbs->text->ptr = '*';
	if (cbs->startPos == passlen)
	    passphrase[passlen++] = c;
	else {
	    for (i = passlen; i > cbs->startPos; i--)
		passphrase[i + 1] = passphrase[i];
	    passphrase[cbs->startPos] = c;
	    passlen++;
	}
    } else {				/* deletion */
	if (cbs->endPos == passlen)
	    passlen = cbs->startPos;
	else {
	    for (i = 0; i < passlen - cbs->endPos; i++)
		passphrase[cbs->startPos + i] = passphrase[cbs->endPos + i];
	    passlen -= cbs->endPos - cbs->startPos;
	}
    }
}

/*
 * Given sequence, seed and passphrase, calculate the response and display
 * it in the Response frame.  The response is also placed in CUT_BUFFER0
 * of the X server, so it can be conveniently pasted in the login window.
 * If you press Ctrl-Return instead of Return, you will be prompted to
 * retype the passphrase for verification.
 */
static void
PActivate(Widget w, XtPointer client_data, XtPointer call_data)
{
    XmAnyCallbackStruct *cbs = (XmAnyCallbackStruct *) call_data;
    XmString s;
    char *t;
    char *ss;
    char respstr[32];
    char sseq[15];
    unsigned digest[4];
    unsigned opieval[2];
    int seq, i;
    MD4_CTX ctx4;
    MD5_CTX ctx5;
    static unsigned vdigest[4];		/* for passphrase verification */
    static Boolean do_verify = False;

    if (cbs->event->xkey.send_event)	/* no synthetic events */
	return;
    passphrase[passlen] = '\0';
    if (!do_verify && cbs->event->xkey.state & ControlMask) {
	MD5Init(&ctx5);
	MD5Update(&ctx5, (unsigned char *) passphrase, passlen);
	MD5Final((unsigned char *) vdigest, &ctx5);
	memset(passphrase, '\0', sizeof passphrase);
	passlen = 0;
	s = XmStringCreateLtoR("Retype the passphrase",
			       XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(resp.resp_label, XmNlabelString, s, NULL);
	XmStringFree(s);
	XmTextFieldSetString(pass.pass_field, "");
	do_verify = True;
	return;
    }
    if (do_verify) {
	int result;

	do_verify = False;
	MD5Init(&ctx5);
	MD5Update(&ctx5, (unsigned char *) passphrase, passlen);
	MD5Final((unsigned char *) digest, &ctx5);
	result = memcmp(digest, vdigest, sizeof digest);
	memset(vdigest, '\0', sizeof vdigest);
	if (result != 0) {
	    s = XmStringCreateLtoR("Passphrases don\'t match",
				   XmSTRING_DEFAULT_CHARSET);
	    XtVaSetValues(resp.resp_label, XmNlabelString, s, NULL);
	    XmStringFree(s);
	    XmTextFieldSetString(pass.pass_field, "");
	    memset(passphrase, '\0', sizeof passphrase);
	    passlen = 0;
	    return;
	}
    }

    memset(respstr, '\0', sizeof respstr);
    t = XmTextFieldGetString(host.seed);
    seq = atoi(ss = XmTextFieldGetString(host.sequence));
    XtFree(ss);

    if (XmToggleButtonGetState(host.md4)) {
	MD4Init(&ctx4);
	MD4Update(&ctx4, (unsigned char *) t, strlen(t));
	XtFree(t);
	MD4Update(&ctx4, (unsigned char *) passphrase, passlen);
	MD4Final((unsigned char *) digest, &ctx4);
	opieval[0] = digest[0] ^ digest[2];
	opieval[1] = digest[1] ^ digest[3];

	for (i = 0; i < seq; i++) {
	    MD4Init(&ctx4);
	    MD4Update(&ctx4, (unsigned char *) opieval, sizeof opieval);
	    MD4Final((unsigned char *) digest, &ctx4);
	    opieval[0] = digest[0] ^ digest[2];
	    opieval[1] = digest[1] ^ digest[3];
	}
    } else {
	MD5Init(&ctx5);
	MD5Update(&ctx5, (unsigned char *) t, strlen(t));
	XtFree(t);
	MD5Update(&ctx5, (unsigned char *) passphrase, passlen);
	MD5Final((unsigned char *) digest, &ctx5);
	opieval[0] = digest[0] ^ digest[2];
	opieval[1] = digest[1] ^ digest[3];

	for (i = 0; i < seq; i++) {
	    MD5Init(&ctx5);
	    MD5Update(&ctx5, (unsigned char *) opieval, sizeof opieval);
	    MD5Final((unsigned char *) digest, &ctx5);
	    opieval[0] = digest[0] ^ digest[2];
	    opieval[1] = digest[1] ^ digest[3];
	}
    }
    opiebtoe(respstr, (char *) opieval);

    if (seq > 0)
	seq--;
    else
	seq = 0;
    sprintf(sseq, "%d", seq);
    XmTextFieldSetString(host.sequence, sseq);
    XtVaSetValues(host.sequence, XmNcursorPosition, strlen(sseq), NULL);

    s = XmStringCreateLtoR(respstr, XmSTRING_DEFAULT_CHARSET);
    XtVaSetValues(resp.resp_label, XmNlabelString, s, NULL);
    XmStringFree(s);
    XChangeProperty(XtDisplay(w), DefaultRootWindow(XtDisplay(w)),
		    XA_CUT_BUFFER0, XA_STRING, 8, PropModeReplace,
		    respstr, strlen(respstr));
    wpid = XtAppAddWorkProc(app, PRemoveFocus, NULL);
}

/*
 * Verify field input.  Name accepts alphanumeric characters, dot and
 * dash.  Sequence is numeric only.  Seed doesn't accept blanks.  Name
 * is forced to lowercase.
 */
static void
FVerify(Widget w, XtPointer client_data, XtPointer call_data)
{
    XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) call_data;
    int ftype = (unsigned) client_data;
    int status = True;
    int c, i;

    if (!cbs->event)			/* programmatic insertion/deletion */
	return;
    for (i = 0; i < cbs->text->length; i++) {
	c = cbs->text->ptr[i];
	switch (ftype) {
	    case F_NAME:
		status = status && (isalnum(c) || c == '-' || c == '.');
		if (status && isupper(c))
		    cbs->text->ptr[i] = tolower(c);
		break;
	    case F_SEQ:
		status = status && isdigit(c);
		break;
	    case F_SEED:
		status = status && (isprint(c) && !(c == ' '));
		break;
	    default:
		status = False;
		break;
	}
	if (!status)
	    break;
    }
    cbs->doit = status;
}
    
/*
 * OK/Cancel callback for New and Edit.  Input fields are checked for
 * validity.
 */
static void
Host_Exit_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    char *h, *s;

    dlg_answer = (unsigned) client_data;
    if (dlg_answer) {
	h = XmTextFieldGetString(host.name_field);
	s = XmTextFieldGetString(host.seed);
	if (adding_host && host_lookup(h) != -1) {
	    wpid = XtAppAddWorkProc(app, RetainFocus, (XtPointer) ERR_NHFOUND);
	    XtFree(h);
	    XtFree(s);
	    return;
	}
	if (adding_host && strlen(h) == 0) {
	    wpid = XtAppAddWorkProc(app, RetainFocus, (XtPointer) ERR_NHEMPTY);
	    XtFree(h);
	    XtFree(s);
	    return;
	}
	if (strlen(s) == 0) {
	    wpid = XtAppAddWorkProc(app, RetainFocus,
				    adding_host ? (XtPointer) ERR_NSEMPTY
						: (XtPointer) ERR_CSEMPTY);
	    XtFree(h);
	    XtFree(s);
	    return;
	}
	XtFree(h);
	XtFree(s);
    }
    dlg_done = True;
}

/*
 * Leave button callback for Calculate.  Check the seed.
 */
static void
Calc_Exit_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    char *s = NULL;

    s = XmTextFieldGetString(host.seed);
    if (strlen(s) == 0) {
	wpid = XtAppAddWorkProc(app, RetainFocus, (XtPointer) ERR_CSEMPTY);
	XtFree(s);
	return;
    }
    XtFree(s);
    dlg_done = dlg_answer = True;
}

/*
 * Move focus to the field that needs editing.
 */
static Boolean
RetainFocus(XtPointer client_data)
{
    Widget where;

    XtRemoveWorkProc(wpid);
    switch ((unsigned) client_data) {
	case ERR_NHFOUND:
	    err_report(shell, "The name you have entered already exists.",
		       "Please enter a different name.");
	    where = host.name_field;
	    break;
	case ERR_NHEMPTY:
	    err_report(shell, "You haven't entered a name for the new host.",
		       "Please enter the host name.");
	    where = host.name_field;
	    break;
	case ERR_NSEMPTY:
	    err_report(shell, "You haven't entered a seed for the new host.",
		       "Please enter the seed.");
	    where = host.seed;
	    break;
	case ERR_CSEMPTY:
	    err_report(shell, "The seed value for the host must not be empty.",
		       "Please re-enter the seed.");
	    where = host.seed;
	    break;
    }
    XmProcessTraversal(where, XmTRAVERSE_CURRENT);
    return True;
}

/*
 * Form unmap callback.  We destroy the buttons here since we don't know
 * how the user will choose to exit.
 */
static void
Form_Unmap_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
    if (buttons.ok_pb) {
	XtDestroyWidget(buttons.ok_pb);
	buttons.ok_pb = NULL;
    }
    if (buttons.cancel_pb) {
	XtDestroyWidget(buttons.cancel_pb);
	buttons.cancel_pb = NULL;
    }
    dlg_done = True;
    dlg_answer = False;
}
