/* File   : uwatch_func.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Functions use with the uwatch program.  The individual functions
 *          are detailed below.
 *
 * Uwatch is Copyright (C)1998 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "config.h"
#include "uwatch.h"
#include "uwatch_func.h"
#include <unistd.h>
#include <sys/utsname.h>  /* uname()     */
#include <utmp.h>         /* struct utmp */
#include <time.h>         /* localtime() */
#include <string.h>       /* strrchr()   */


/* Function: get_argument
 * Input   : An open file pointer from which to read.
 * Return  : Everything until EOF, a new line, or a comment is reached.
 */
char *get_argument (FILE *ifp) {
  char c;
  char *inbuf = NULL, *retval;
  int count = 0, flag = 0, size = 0;

  /* Skip over any leading white space */
  skip(ifp, SPACE);

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if (c == '\n')
      break;
    if ((flag) && (c == '#')) {
      skip(ifp, TO_EOL);
      break;
    }
    if ((c == ' ') || (c == '\t'))
      flag = 1;
    else flag = 0;
    if ((++count >= size) &&
        ((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    inbuf[count - 1] = c;
  }

  if (count == 0)
    return(NULL);
  inbuf[count] = '\0';

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Fatal Error: Out of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);
  return(retval);
}


/* Function: get_command
 * Input   : A file pointer to read data from.
 * Return  : The next command in the file or NULL if error, EOF, or white
 *           space.
 */
char *get_command (FILE *ifp) {
  char c;
  int size = 0;
  char *inbuf = NULL, *retval;
  int count = 0;

  /* Read and store input until whitespace or EOF is found */
  while ((c = getc(ifp)) != EOF) {
    if ((c == ' ') || (c == '\n') || (c == '\t'))
      break;
    if ((++count >= size) &&
      ((inbuf = (char *) realloc(inbuf, size += INBUF)) == NULL)) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    inbuf[count - 1] = c;
  }
  if (count == 0)
    return(NULL);

  inbuf[count] = '\0';
  if (ungetc(c, ifp) == EOF) {
    fprintf(stderr, "Unable to ungetc()\n");
    exit(1);
  }

  /* Return the data read */
  if ((retval = (char *) malloc(strlen(inbuf) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  strcpy(retval, inbuf);
  free(inbuf);                     /* just in case... */
  return(retval);
}


/* Function: get_end
 * Input   : A file to seek to the end of and a pointer to a file position
 *           structure.
 * Output  : The position of the end of the passed file is stored in the
 *           passed position structure.
 */
void get_end (char *file, fpos_t *position) {
  FILE *ifp = NULL;

  /* Sort of makes the use of fpos_t pointless */
  *position = 0;

  if (((ifp = fopen(file, "r")) != NULL) &&
      (fseek(ifp, 0, SEEK_END) != -1))
    fgetpos(ifp, position);
  fclose(ifp);
}


/* Function: parse_config
 * Input   : The name of the configuration file to parse.
 * Output  : The passed file is parsed and the appropriate settings are made.
 */
void parse_config (char *file) {
  char *command, *argument;
  FILE *ifp;

  /* Open the file */
  if ((file == NULL) || ((ifp = fopen(file, "r")) == NULL))
    return;

  /* Parse the file */
  skip(ifp, WHITE|COMMENTS);
  while ((command = get_command(ifp)) != NULL) {
    argument = get_argument(ifp);
    skip(ifp, WHITE|COMMENTS);

    if (strcmp(command, "in_message") == 0) {
      if (flag & GOT_IN_MSG)
	free(in_format);
      in_format = argument;
      flag |= GOT_IN_MSG;
    } else if (strcmp(command, "out_message") == 0) {
      if (flag & GOT_OUT_MSG)
	free(out_format);
      out_format = argument;
      flag |= GOT_OUT_MSG;
    } else if (strcmp(command, "in_program") == 0) {
      if (flag & GOT_IN_PROG)
	free(in_program);
      in_program = argument;
      flag |= GOT_IN_PROG;
    } else if (strcmp(command, "out_program") == 0) {
      if (flag & GOT_OUT_PROG)
	free(out_program);
      out_program = argument;
      flag |= GOT_OUT_PROG;
    } else {
      if (strcmp(command, "watch_list") == 0) {
        parse_name_file(argument);
        flag |= RESTRICT;
      } else if (strcmp(command, "ignore_list") == 0)
        ;
      else if (strcmp(command, "quick") == 0)
	flag |= QUICK;
      else if (strcmp(command, "debug") == 0)
	flag |= DEBUG;
      else if (strcmp(command, "quiet") == 0)
	flag |= QUIET;
      else if (strcmp(command, "nokillonexit") == 0)
	flag &= ~KILL_ON_EXIT;
      else if (strcmp(command, "killonexit") == 0)
	flag |= KILL_ON_EXIT;
      else if (strcmp(command, "norunprogram") == 0)
	flag |= NO_RUN_PROG;
      else if (strcmp(command, "runprogram") == 0)
	flag &= ~NO_RUN_PROG;
      else if (strcmp(command, "sleep") == 0)
	sleep_time = atoi(argument);
      free(argument);
    }
    free(command);
  }
}


/* Function: parse_name_file
 * Input   : A file to parse.
 * Output  : The names in the file are added to the name hash.
 */
void parse_name_file (char *file) {
  char *name;
  FILE *ifp;

  /* Open the input file */
  if ((ifp = fopen(file, "r")) == NULL) {
    if (!flag & QUIET)
      fprintf(stderr, "Warning: Unable to open %s for reading\n", file);
    return;
  }

  /* Parse the file */
  skip(ifp, WHITE|COMMENTS);
  while ((name = get_command(ifp)) != NULL) {
    if (insert_name(name) == -1)
      free(name);
    skip(ifp, WHITE|COMMENTS);
  }
}


/* Function: print_name
 * Input   : A format string and a name to print.  The format string may
 *           contain the following special directives:
 *             %%  Print a % character
 *             %b  Print a bell
 *             %e  Print an escape character
 *             %h  Print the local host name
 *             %n  Print a new line
 *             %p  Print the port (tty) that the user is on
 *             %r  Print the remote host name
 *             %t  Print the current time
 *             %u  Print the passed name
 *           If no format string is given, then just the name is printed.
 * Output  : The passed name in the passed is printed.
 */
void print_name (char *format, struct utmp *user_data) {
  char *ptr = format, time_data[9];
  int field_len = 0;
  struct utsname uname_buf;

  if (flag & QUIET)
    return;

  if (ptr == NULL)
    if (user_data != NULL)
      fprintf(stderr, "%s\n", user_data->ut_name);
    else fprintf(stderr, "Unknown");
  if (user_data == NULL)
    return;

  flag &= ~IN_EXPAND;
  while (*ptr != '\0') {
    if (flag & IN_EXPAND) {
      flag &= ~IN_EXPAND;
      switch (*ptr) {
      case '\0':
	break;
      case '0': case '1': case '2': case '3': case '4': case '5':
      case '6': case '7': case '8': case '9':
	field_len = field_len * 10 + *ptr - '0';
	flag |= IN_EXPAND;
	break;
      case '%':
	putc('%', stderr);
	break;
      case 'b':
	putc(7, stderr);
	break;
      case 'e':
	putc(27, stderr);
	break;
      case 'h':
	uname(&uname_buf);
	if (field_len)
	  fprintf(stderr, "%-*.*s", field_len, field_len, uname_buf.nodename);
	else fprintf(stderr, "%s", uname_buf.nodename);
	break;
      case 'n':
	fprintf(stderr, "\n");
	break;
      case 'p':
	if (user_data != NULL)
	  if (field_len)
	    fprintf(stderr, "%-*.*s", field_len, field_len,
		    user_data->ut_line);
	  else fprintf(stderr, "%.*s", UT_LINESIZE, user_data->ut_line);
	break;
      case 'r':
	if (user_data != NULL)
	  if (field_len)
	    fprintf(stderr, "%-*.*s", field_len, field_len,
		    user_data->ut_host);
	  else fprintf(stderr, "%.*s", UT_HOSTSIZE, user_data->ut_host);
	break;
      case 't':
	strftime(time_data, 9, "%H:%M.%S", localtime(&user_data->ut_time));
	if (field_len)
	  fprintf(stderr, "%-*.*s", field_len, field_len, time_data);
	else fprintf(stderr, "%s", time_data);
	break;
      case 'u':
	if (user_data != NULL)
	  if (field_len)
	    fprintf(stderr, "%-*.*s", field_len, field_len,
		    user_data->ut_name);
	  else fprintf(stderr, "%.*s", UT_NAMESIZE, user_data->ut_name);
	else fprintf(stderr, "Unknown");
	break;
      default:
	putc(*ptr, stderr);
      }
    } else if (*ptr == '%')
      flag |= IN_EXPAND;
    else putc(*ptr, stderr);

    if (!(flag & IN_EXPAND))
      field_len = 0;
    ++ptr;
  }
}


/* Function: run_program
 * Input   : The name of a program to run and utmp data for expanding %
 *           directives in the program name, (see print_name).
 * Output  : All % directives in the passed program name and arguments are
 *           expanded and the program is executed.
 */
void run_program (char *program, struct utmp *data) {
  char time_data[9];
  char *ptr, *run_string;
  int len = 0;

  if (program == NULL)
    return;

  /* Determine the maximum size of the expanded output string */
  for (ptr = program; *ptr != '\0'; ++ptr) {
    if (*ptr == '%') {
      ++ptr;
      if (*ptr == 'u')
	len += strlen(data->ut_name);
      else if (*ptr == 'r')
	len += strlen(data->ut_host);
      else if (*ptr == 'p')
	len += strlen(data->ut_line);
      else if (*ptr == 't')
	len += 9;
      else if ((*ptr == '%') || (*ptr == 'b') || (*ptr == 'n') ||
	       (*ptr == 'e'))
	++len;
      else len += 2;
    } else ++len;
  }

  if ((run_string = (char *) malloc(len + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  /* Fill the run string */
  for (ptr = program, len = 0; *ptr != '\0'; ++ptr, ++len) {
    run_string[len] = '\0';
    if (*ptr == '%') {
      ++ptr;
      if (*ptr == 'b') {
        run_string[len] = 7;
	continue;
      } else if (*ptr == 'e') {
	run_string[len] = 27;
	continue;
      } else if (*ptr == 'n') {
	run_string[len] = '\n';
	continue;
      } else if (*ptr == 'u')
	strncat(run_string, data->ut_name, UT_NAMESIZE);
      else if (*ptr == 'r')
	strncat(run_string, data->ut_host, UT_HOSTSIZE);
      else if (*ptr == 'p')
	strncat(run_string, data->ut_line, UT_LINESIZE);
      else if (*ptr == 't') {
	strftime(time_data, 9, "%H:%M.%S", localtime(&data->ut_time));
	strcat(run_string, time_data);
      } else {
	run_string[len++] = '%';
	run_string[len] = *ptr;
	continue;
      }
      len = strlen(run_string) - 1;
    } else run_string[len] = *ptr;
  }
  run_string[len] = '\0';

  system(run_string);
  free(run_string);
}

    
/* Function: set_user_tty
 * Output  : The global user_tty variable is set to the calling tty.
 */
void set_user_tty (void) {
  char *ptr, *str, *tty;

  if ((tty = ttyname(0)) == NULL) {
    fprintf(stderr, "Unable to get tty.  Try the -k flag.\n");
    exit(1);
  }

  if ((ptr = strrchr(tty, '/')) == NULL)
    ptr = tty;
  else ptr += 1;

  if ((user_tty = (char *) malloc(strlen(ptr) + 1)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  strcpy(user_tty, ptr);
  /* Do we need to free tty? */
}


/* Function: skip
 * Input   : An open file to read from and a flag telling what to skip over.
 *           The flag may be an or of any of the following values:
 *
 *           COMMENTS - Skip lines starting with a # character
 *           SPACE    - Skip over spaces and tabs, but not new lines
 *           TO_EOL   - Skip over everything to and including the first new
 *                      line
 *           WHITE    - Skip over white space including new lines
 *
 * Output  : File pointer is moved over things to "skip."
 */
void skip (FILE *ifp, int flag) {
  char c;

  /* Make sure a valid flag and non-null file pointer was given */
  if (!((flag & COMMENTS) || (flag & SPACE) || (flag & TO_EOL) ||
      (flag & WHITE)) || (ifp == NULL))
    return;

  /* Loop until EOF is found */
  while ((c = getc(ifp)) != EOF) {

    /* Skip over comments */
    if ((flag & COMMENTS) && (c == '#')) {
      skip_line(ifp, 1);
      continue;
    }

    /* Skip over white space */
    if (((flag & WHITE) || (flag & SPACE)) &&
      ((c == ' ') || (c == '\t') || (c == '\r')))
      continue;

    /* Test for EOL */
    if (c == '\n') {
      if (flag & WHITE)
      continue;
      else if (flag & TO_EOL)
      return;
    }

    /* Ignore everything if TO_EOL (new line is detected above) */
    if (flag & TO_EOL)
      continue;

    /* Place the last character read back into stream */
    if (ungetc(c, ifp) == EOF) {
      fprintf(stderr, "Unable to unget()\n");
      exit(1);
    }
    return;
  }
}


/* Function: skip_line
 * Input   : A file pointer to read the data from and the number of lines
 *           to skip.
 * Ouptut  : File pointer is moved appropiately
 */
void skip_line (FILE *ifp, int count) {
  char tmpin;
  int line = 0;

  while (line++ < count)
    skip(ifp, TO_EOL);
}
