/* File   : uwatch.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Gives notification when users log in or out.
 *
 * Uwatch is Copyright (C)1998 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "config.h"
#include "uwatch.h"
#include "uwatch_func.h"
#include "hash_func.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <unistd.h>
#include <utmp.h>


/* Function: user_in
 * Input   : A utmp entry holding the user data.
 * Output  : Print a user "in" message.
 */
void user_in (struct utmp *inbuf) {
  if ((~flag & RESTRICT) || (scan_name(inbuf->ut_name) != NULL)) {
    if (in_format == NULL) {

#ifdef DEF_IN_FORMAT
      print_name(DEF_IN_FORMAT, inbuf);
#else
      print_name("%bIn: %u%n", inbuf);
#endif  /* DEF_IN_FORMAT */

    } else print_name(in_format, inbuf);
    if (~flag & NO_RUN_PROG)
      run_program(in_program, inbuf);
  }
}


/* Function: user_out
 * Input   : An open file pointer, the current position, and a utmp entry
 *           holding the pty data.
 * Output  : Print a user "out" message.
 */
void user_out (FILE *ifp, fpos_t *position, struct utmp *inbuf) {
  struct utmp tmp_buf;

  /* Scan the wtmp in reverse until the name is found */
  while (1) {

    /* Check for the begining of the file */
    if (fseek(ifp, sizeof(struct utmp) * -2, SEEK_CUR) == -1)
      break;
    if (fread(&tmp_buf, sizeof(struct utmp), 1, ifp) != 1)
      break;

    /* Check for the name */

#ifdef LINUX
    if ((tmp_buf.ut_type == USER_PROCESS) &&
	(strcmp(tmp_buf.ut_line, inbuf->ut_line) == 0)) {
#endif  /* LINUX */

#ifdef BSDI
    if (strcmp(tmp_buf.ut_line, inbuf->ut_line) == 0) {
#endif  /* BSDI */

      /* Check if the calling user has logged out */
      if ((flag & KILL_ON_EXIT) && (strcmp(inbuf->ut_line, user_tty) == 0))
	exit(0);

      if ((~flag & RESTRICT) || (scan_name(tmp_buf.ut_name) != NULL)) {
	tmp_buf.ut_time = inbuf->ut_time;
	if (out_format == NULL)

#ifdef DEF_OUT_FORMAT
	  print_name(DEF_OUT_FORMAT, &tmp_buf);
#else
	  print_name("%bOut: %u%n", &tmp_buf);
#endif

	else print_name(out_format, &tmp_buf);
	if (~flag & NO_RUN_PROG)
	  run_program(out_program, &tmp_buf);
      }
      break;
    }
  }

  /* Reposition the file pointer */
  if (fsetpos(ifp, position) == -1) {
    fprintf(stderr, "Unable to position file %s", _PATH_WTMP);
    exit(1);
  }
}


/* Function: print_changes
 * Input   : A file to use, a position pointer into the passed file from which
 *           to start printing data, and a flag which determines if logout
 *           records should be printed (1) or not (0).
 */
int print_changes (char *file, fpos_t *position, short type) {
  FILE *ifp;
  struct utmp inbuf;

  /* Open the file */
  if ((ifp = fopen(file, "r")) == NULL) {
    if (~flag & QUIET)
      fprintf(stderr, "Unable to open %s for reading\n", file);
    return(-1);
  }

  /* Position ourselves in the file */
  if ((fsetpos(ifp, position) == -1) && (fsetpos(ifp, 0) == -1)) {
    if (~flag & QUIET)
      fprintf(stderr, "Unable to position file %s", file);
    return(-1);
  }

  /* Print the file data */
  while (fread(&inbuf, sizeof(struct utmp), 1, ifp) == 1) {
    if (fgetpos(ifp, position) == -1) {
      if (~flag & QUIET)
	fprintf(stderr, "Unable to get position in %s\n", file);
      return(-1);
    }

#ifdef LINUX
    if (inbuf.ut_type == USER_PROCESS)
#endif  /* LINUX */

#ifdef BSDI
    if (strcmp(inbuf.ut_name, "") != 0)
#endif  /* BSDI */

      user_in(&inbuf);

#ifdef LINUX
    else if ((inbuf.ut_type == DEAD_PROCESS) && (type == 1))
#endif  /* LINUX */

#ifdef BSDI
    else if (type == 1)
#endif  /* BSDI */

      user_out(ifp, position, &inbuf);
  }

  fclose(ifp);
}


/* Function: usage
 * Input   : A program name.
 * Output  : A usage message is printed and the program is exited.
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s\n", name);
  fprintf(stderr, "       [-c <file>] [-d] [-e <program>] [-f <file>] [-h] [-i <string>]\n");
  fprintf(stderr, "       [-o string] [-r <file>] [-s] [-q] [-x <program>] [user [user ...]]\n");
  fprintf(stderr, " -c  Use <file> as a user configuration file.\n");
  fprintf(stderr, " -d  Run in debug mode, (do not go into the background).\n");
  fprintf(stderr, " -e  Execute <program> when a user logs in.\n");
  fprintf(stderr, " -f  Load names for which to look from the file <file>.  This file should\n");
  fprintf(stderr, "     contain a list of names seperated by spaces, tabs, or new lines.\n");
  fprintf(stderr, "     If a # character is found, the rest of the line is taken as a comment\n");
  fprintf(stderr, "     and ignored.  Blank lines are also ignored.\n");
  fprintf(stderr, " -h  Print this usage message.\n");
  fprintf(stderr, " -i  Use <string> as the format for \"in\" messages.\n");
  fprintf(stderr, " -k  Do not kill the watch program when the invoking user logs out.\n");
  fprintf(stderr, " -o  Use <string> as the format for \"out\" messages.\n");
  fprintf(stderr, " -r  Redirect output to <file>\n");
  fprintf(stderr, " -s  Silent operation, (no messages are printed).\n");
  fprintf(stderr, " -q  Print a quick summary of who is online and exit.\n");
  fprintf(stderr, " -x  Execute <program> when a user logs out.\n");
  exit(0);
}


/* Function: parse_arguments
 * Input   : An argument count and the argument array.
 * Output  : The passed arguments are parsed and appropiate action is taken.
 */
void parse_arguments (int argc, char *argv[]) {
  int c;

  while ((c = getopt(argc, argv, "c:de:f:hi:ko:qsw:x:")) != EOF) {
    switch (c) {
    case 'c':
      parse_config(optarg);
      continue;
    case 'd':
      flag |= DEBUG;
      continue;
    case 'e':
      in_program = optarg;
      break;
    case 'f':
      parse_name_file(optarg);
      flag |= RESTRICT;
      continue;
    case 'i':
      in_format = optarg;
      continue;
    case 'k':
      flag &= ~KILL_ON_EXIT;
      continue;
    case 'o':
      out_format = optarg;
      continue;
    case 'q':
      flag |= QUICK;
      continue;
    case 's':
      flag |= QUIET;
      continue;
    case 'w':
      sleep_time = atoi(optarg);
      continue;
    case 'x':
      out_program = optarg;
      continue;
    default:
      usage(argv[0]);
    }
  }

  if (optind < argc) {
    flag |= RESTRICT;
    while (optind < argc)
      insert_name(argv[optind++]);
  }
}

  
int main (int argc, char *argv[]) {
  char *str;
  fpos_t position = 0, dummy = 0;
  pid_t pid;
  time_t last_mod = 0;
  struct stat stat_buf;
  struct passwd *pw;
  int tmpint;

  sleep_time = MIN_SLEEP_TIME;
  flag = KILL_ON_EXIT;
  in_format = in_program = out_format = out_program = user_tty = NULL;
  init_name_hash();

  /* Read the configuration files and the passed arguments */

#ifdef SYSTEM_CONFIG
  parse_config(SYSTEM_CONFIG);
#endif  /* SYSTEM_CONFIG */

#ifdef USER_CONFIG
  if ((pw = getpwuid(geteuid())) == NULL) {
    if (~flag & QUIET)
      fprintf(stderr, "Unable to find your user name\n");
  } else if ((str = (char *) malloc(strlen(pw->pw_dir) +
				    strlen(USER_CONFIG) + 2)) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  } else {
    sprintf(str, "%s/%s", pw->pw_dir, USER_CONFIG);
    parse_config(str);
    free(str);
  }
#endif  /* USER_CONFIG */

  parse_arguments(argc, argv);

#ifdef SECURE_CONFIG
  parse_config(SECURE_CONFIG);
#endif  /* SECURE_CONFIG */

  /* Test the validity of the options */
#ifdef MIN_SLEEP_TIME
  if (sleep_time < MIN_SLEEP_TIME) {
    sleep_time = MIN_SLEEP_TIME;
    if (~flag & QUIET)
      fprintf(stderr, "Warning: Sleep time too low - Set to %d\n", sleep_time);
  }
#endif  /* MIN_SLEEP_TIME */

  /* Put ourselves into the background */
  if (~flag & DEBUG) {
    if ((pid = fork()) == -1) {
      fprintf(stderr, "Unable to fork\n");
      exit(1);
    } else if (pid != 0)
      exit(0);
  }

  /* Print the users currently online.  In order to prevent some names  */
  /* from "falling through the cracks," we first seek to the end of the */
  /* wtmp file.  This, however, may cause a repeat announcement.        */
  get_end(_PATH_WTMP, &position);
  print_changes(_PATH_UTMP, &dummy, 0);
  if (flag & QUICK)
    exit(0);

  if (flag & KILL_ON_EXIT)
    set_user_tty();

  /* Loop forever watching for new activity */
  while (1) {
    if (stat(_PATH_WTMP, &stat_buf) < 0) {
      fprintf(stderr, "Unable to stat() %s\n", _PATH_WTMP);
      exit(1);
    }
    if (last_mod != stat_buf.st_mtime) {
      if (print_changes(_PATH_WTMP, &position, 1) == -1)
	exit(1);
      last_mod = stat_buf.st_mtime;
    }
    sleep(sleep_time);
  }
}
