/* File   : hash_func.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Functions for handline the hash tables in uwatch.  The individual
 *          functions are detailed below.
 *
 * Uwatch is Copyright (C)1998 Karyl F. Stein
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>
#include "uwatch.h"
#include "hash_func.h"


/* Function: init_name_hash
 * Output  : The global name hash is initialized.
 */
void *init_name_hash (void) {
  int count;

  for (count = 0; count < NAME_HASH_SIZE; ++count)
    watch_list[count] = NULL;
}


/* Function: insert_name
 * Input   : A name to insert.
 * Output  : The passed name is inserted into the name hash.
 */
int insert_name (char *name) {
  bucket *newnode;

  if (name == NULL)
    return(1);

  /* Scan for duplicates */
  if (scan_name(name) != NULL)
    return(-1);

  /* Create a new node */
  newnode = new_name_bucket();
  newnode->name = name;

  /* Insert the new name into the hash */
  newnode->next = watch_list[*newnode->name % 13];
  watch_list[*newnode->name % 13] = newnode;
}


/* Function: new_name_bucket
 * Return  : A new name bucket.
 */
bucket *new_name_bucket (void) {
  bucket *retval = (bucket *) malloc(sizeof(bucket));

  if (retval == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  retval->name = NULL;
  retval->next = NULL;
}


/* Function: scan_name
 * Input   : A name for which to search.
 * Return  : A pointer to the bucket holding the name, or NULL if not found.
 */
bucket *scan_name (char *name) {
  bucket *retval;

  if (name == NULL)
    return(NULL);

  retval = watch_list[*name % 13];
  while (retval != NULL) {
    if (strcmp(retval->name, name) == 0)
      return(retval);
    retval = retval->next;
  }

  return(NULL);
}
