/*Included*Headers*************************************************************/

#include "termlock.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include <plh_general.h>
#include <plh_bits.h>
#include <plh_memory.h>
#include <plh_parse.h>
#include <plh_ptr.h>
#include <plh_env.h>
#include <plh_io.h>

/*Macros***********************************************************************/

			/* Macro to cause the process to sleep for n seconds  */
#define Rest(n)		usleep( (n) * 1000000 )	

			/* Macro to print the terminal prompt for input       */
#define Prompt		printf("%s ", tprompt)

/*Routines*********************************************************************/

					/*GetOptions****************************
					* Routine to set the option variables. *
					* Currently, the options are retrieved *
					* via rc file and sets unspecified     *
					* variables using the defaults.        *
					***************************************/
void SetOptions(char *errdelay, char *badallow, char *shellfile,
		char *scriptusr, char *termprom, char *acsprom)
 {
			/* Configuration (rc) file that contains the options  */
   FILE *cfgfile = fopen(CONF_FILE, "r");
   			/* Structure for reading in the options		      */
   options *opts;
   			/* Temporary variable value holder for assignments    */
   char *value;
   
   			/* If opening rc file fails - assume system has been  */
   			/* compromised and logout.			      */
   if(IS_NULL(cfgfile))
    {
      fprintf(stderr,
              "Missing or corrupt configuration file (%s)\n", CONF_FILE);
      fprintf(stderr, "Aborting login...\n");
      Rest(DEF_EDELAY);
      exit(0);
    }
  
   			/* File opened fine- now parse the file for all       */
   			/* variables and their assigned values.		      */
   opts = ReadOptionsFile(cfgfile, REMOVE_COMMENTS | GET_VARS, '#', '.');	

			/* Check for new configure options and use in place   */
			/* of the defaults if they exist.		      */
   if(!IS_NULL(value = GetVariableValue(*opts, FILE_LABEL)))
     strcpy(shellfile, value);
   else
     strcpy(shellfile, DEF_SHFILE);
     
   if(!IS_NULL(value = GetVariableValue(*opts, SCRIPT_LABEL)))
     strcpy(scriptusr, value);
   else
     strcpy(scriptusr, DEF_SCRIPT);
     
   if(!IS_NULL(value = GetVariableValue(*opts, APROMPT_LABEL)))
     strcpy(acsprom, value);
   else
     strcpy(acsprom, DEF_APROMPT);
     
   if(!IS_NULL(value = GetVariableValue(*opts, TPROMPT_LABEL)))
     strcpy(termprom, value);
   else
     strcpy(termprom, DEF_TPROMPT);
     
   if(!IS_NULL(value = GetVariableValue(*opts, ATTEMPT_LABEL)))
     *badallow = atoi(value);
   else
     *badallow = DEF_MAXATT;
     
   if(!IS_NULL(value = GetVariableValue(*opts, EDELAY_LABEL)))
     *errdelay = atoi(value);
   else
     *errdelay = DEF_EDELAY;
     
#ifdef DEBUG
			/* For debugging: code to display the current config  */
			/* options being used.				      */
   fprintf(stderr, "Current Configuration:\n");
   fprintf(stderr, "     Shell File: %s\n", shellfile);
   fprintf(stderr, "    Term Prompt: %s\n", termprom);
   fprintf(stderr, "  Access Prompt: %s\n", acsprom);
   fprintf(stderr, "   Max Attempts: %d\n", *badallow);
   fprintf(stderr, "    Error Delay: %d\n", *errdelay);
   fprintf(stderr, "    Script User: %s\n", scriptusr);
#endif

   fclose(cfgfile);
   free(opts);
 }

					/*GetShell******************************
					* Routine to retrieve a shell from the *
					* shell file with the name passed.     *
					* Routine returns a pointer to the     *
					* shell or NULL if no user was found.  *
					****************************************
					* user	user to search for shell of    *
					***************************************/
char *GetShell(char *user, char *shfile, char edelay)
 {
			/* pointer to shell file record			      */
   FILE *shellfile = fopen(shfile, "r");
			/* buffer to read information from the shell file     */
   char buffer[LARGE_BUFFER_SIZE];
   			/* pointer to current item from input buffer          */
   char *item;
   	
   if(IS_NULL(shellfile))
    {
      fprintf(stderr,
          "Missing or corrupt shells file (%s) - aborting\n", shfile);
      Rest(edelay);
      exit(0);
    }
    
   while(FGetStringClipped(buffer, LARGE_BUFFER_SIZE, shellfile)
         && !IS_EMPTY(buffer))
    {
			/* Get the user name from appropriate field           */
      item = GetDelimItem(buffer, DELIMETER, NAME_FIELD);

			/* Check username against one from file		      */
      if(!strcmp(user, item))
       {
       			/* if the usernames match, return the user's shell    */
        free(item);
        fclose(shellfile);
        return GetDelimItem(buffer, DELIMETER, SHELL_FIELD);
       }
			/*  free buffer to avoid wasting memory		      */
      free(item);
    }   

			/* search was unsuccessful- close file & return NULL  */
   fclose(shellfile);
   return NULL;
 }
 
					/*UnjumbleKey***************************
					* Routine to 'unjumble' the key which  *
					* is jumbled to stop anyone from       *
					* simply 'more'ing the binary and      *
					* looking for character strings (this  *
					* would be a way to find the key)      *
					****************************************
					* jkey	the jumbled key		       *
					* key	buffer to unjumble key into    *
					***************************************/
void UnjumbleKey(char *jkey, char *key)
 {
			/* Length of the key being 'unjumbled'		      */
   int keylen = strlen(jkey);
   			/* Index for scanning the keys			      */
   int index;
   
   for(index = 0; index < keylen; index++)
     key[index] = *BitMirror8(jkey + index);
 }

					/*OpenShell*****************************
					* Routine to startup to REAL shell     *
					* without using environment.           *
					****************************************
					* shell		binary shell to execute*
					* argv		argument list passed   *
					*               to TermLock at startup *
					***************************************/
void OpenShell(char *shell, char **argv, char edelay)
 {
   			/* Try to execute the shell using the path	      */
   execvp(shell, argv);

   			/* Execution failed - abort login		      */
   fprintf(stderr, "Shell execution failed - aborting login\n");
   Rest(edelay);
   exit(0);
 }
 
/*Main*************************************************************************/

void main(int argc, char *argv[], char *env[])
 {
			/* Jumbled key supplied during the build process      */
   char JUMBLEKEY[] = { TEMPKEY };
			/* Buffer for holding key when it gets unjumbled      */
   char KEY[BUFFER_SIZE];
			/* User ID number (for logins)			      */
   int userid;
			/* Structure for User password entry		      */
   struct passwd *passid;
			/* Device identity of the login terminal	      */ 
   char *tdev;
			/* Delay after printing error message before logout   */
   char edelay;
			/* Maximum number of bad logins before auto-logout    */
   char maxbad;
			/* Absolute path to the user shells file	      */
   char shfile[BUFFER_SIZE];
			/* Normal terminal prompt			      */
   char tprompt[BUFFER_SIZE];
			/* Prompt that marks an access request		      */
   char aprompt[BUFFER_SIZE];
			/* User name that the script executing shell is under */
   char scruser[BUFFER_SIZE];
			/* Pointer to the name of the user who logged in      */
   char *username;
   			/* Pointer to the name of the user's REAL login shell */
   char *shellname;
			/* String to hold the name of the shell executable    */
   char *shellbin;
   			/* Buffer for reading the access request input        */
   char buffer[LARGE_BUFFER_SIZE];
			/* Counter for the number of failed attempts made     */
   char badcount = 0;
   			/* The length of the Access Prompt (saves time)       */
   int apromlen;

#ifdef DEBUG
   fprintf(stderr, "Program started.\n");
#endif

   			/* Read in rc options and set option variables, using */
   			/* default values for unspecified variables. Then,    */
   			/* calculate the length of Access Prompt length to    */
   			/* avoid multiple routine calls for the same value.   */
   SetOptions(&edelay, &maxbad, shfile, scruser, tprompt, aprompt);
   apromlen = strlen(aprompt);

#ifdef DEBUG
   fprintf(stderr, "Got options.\n");
#endif

			/* Get user and terminal information		      */
   userid = getuid();
   passid = getpwuid(userid);
   username = passid->pw_name;
   tdev = ttyname(STDIN_FILENO);

#ifdef DEBUG
   fprintf(stderr, "User and terminal information retrieved.\n");
   fprintf(stderr, "    UserID: %d\n", userid);
   fprintf(stderr, "    Username: %s\n", username);
   fprintf(stderr, "    Terminal: %s\n", tdev);
#endif
            
			/* Check for a non-login (script or shell) execution  */
   if(*argv[0] != '-')
    {
     if(IS_NULL(shellname = GetShell(scruser, shfile, edelay)))
      {
       fprintf(stderr, "User has no entry in shells file - aborting\n");
       Rest(edelay);
       exit(0);
      }
     SplitString(shellname, '/', NULL, &shellbin, SEARCH_END | DELIM_NEITHER);
     argv[0] = MimicString(shellbin);
     OpenShell(shellbin, argv, edelay);
    }
    
#ifdef DEBUG
   fprintf(stderr, "Called as login shell.\n");
#endif

			/* Unjumble key so that comparisons can be made	and   */
			/* retrieve the users real shell from the from the    */
			/* shells file.					      */     
   UnjumbleKey(JUMBLEKEY, KEY);
   if(IS_NULL(shellname = GetShell(username, shfile, edelay)))
    {
      fprintf(stderr, "User has no entry in shells file - aborting\n");
      Rest(edelay);
      exit(0);
    }
   SplitString(shellname, '/', NULL, &shellbin, SEARCH_END | DELIM_NEITHER);
   
#ifdef DEBUG
   fprintf(stderr, "Key unjumbled: %s\n", KEY);
   fprintf(stderr, "User shell retrieved: %s\n", shellname);
   fprintf(stderr, "Shell executable name: %s\n", shellbin);
#endif
   
   			/* Check to ensure that access prompt does not also   */
   			/* contain the access delimeter.		      */
   if(strchr(aprompt, ACCESS_DELIM))
    {
     fprintf(stderr,
       "Access prompt may not use the character %c - aborting\n", ACCESS_DELIM);
     Rest(edelay);
     exit(0);
    }

			/* begin accepting user input			      */    
   Prompt;
   while((badcount != maxbad) &&
         !IS_NULL(FGetStringClipped(buffer, LARGE_BUFFER_SIZE, stdin)) && 
         strcmp(buffer, "logout") && strcmp(buffer, "bye") && 
         strcmp(buffer, "quit") && strcmp(buffer, "exit"))         
    {

#ifdef DEBUG
      fprintf(stderr, "Buffer read: <%s>\n", buffer);
#endif

    			/* check all conditions for bypassing TermLock -      */
    			/* Requires user to know access prompt, access        */
    			/* delimeter and access key.			      */
      if(!strncmp(buffer, aprompt, apromlen) &&
         (buffer[apromlen] == ACCESS_DELIM) &&
         !strcmp(buffer + apromlen + 1, KEY))
       {

#ifdef DEBUG
      fprintf(stderr, "Access Granted.\n");
#endif

			/* Replace shell variable in environment with the     */
			/* appropriate REAL shell entry, then append a '-' to */
			/* front of the first argument to force REAL shell to */
			/* act as a login shell and execute the shell.        */
         ReplaceEnvVariable(env, ENV_SHELL_VAR, shellname);
			/* re-use buffer to build first argument	      */
         strcpy(buffer, "-");
         strcat(buffer, shellbin);
	 argv[0] = MimicString(buffer);

	 OpenShell(shellbin, argv, edelay);
       }
      			/* unsuccessful login attempt- increment the bad      */
      			/* attempts count and re-prompt.		      */
      badcount++;
      Prompt;
      
#ifdef DEBUG
      fprintf(stderr, "Verification failure #%d\n", badcount);
#endif
    }

#ifdef DEBUG
   fprintf(stderr, "Access failed. Logging out...\n");
#endif
    
    			/* login failed - free memory & bail out              */
   free(username);
   free(shellname);
   free(passid);
   free(tdev);
 }
 
