/*******************************************************************************
* plh_parse.h								       *
********************************************************************************
* Routines for parsing a variety of items ranging from runtime command files   *
* to direct user input lines. (as used in shells)			       *
*******************************************************************************/

			/* For FILE					      */
#include <stdio.h>

/*FLAGS************************************************************************/

/*CONSTANTS********************************************************************/

			/* Parsing flag for assuming comments and removing    */
			/* them. (1st bit)				      */
#define REMOVE_COMMENTS		0x1
			/* Parsing flag for parsing input for variables and   */
			/* their associated values. (2nd bit)		      */
#define GET_VARS		0x2
			/* Parsing flag for parsing input for directives      */
			/* (3rd bit)					      */
#define GET_DIRS		0x4
			/* These flags can be combined			      */

			/* String splitting flag to search from front of      */
			/* string.					      */
#define SEARCH_FRONT		0x0
			/* Flag to denote a search from the end of the string */
#define SEARCH_END		0x1
			/* Flags are mutually exclusive.		      */
#define SEARCH_MASK		0x1
			/* Mask to view search optin only. Uses bit 0 of the  */
			/* flags variable			              */

			/* String splitting flag denoting store delimter with */
			/* left side					      */
#define DELIM_LEFT		0x2
			/* Flag denoting store delimeter with right string    */
#define DELIM_RIGHT		0x4
			/* Flag denoting store delimeter with neither string  */
#define DELIM_NEITHER		0x6
			/* Flag denoting store delimeter with both strings    */
#define DELIM_BOTH		0x0
			/* These are mutually exclusive.		      */
#define DELIM_MASK		0x6
			/* Mask to view only the delimeter flag bits	      */
			/* Uses bits 1 & 2.				      */
			
/*MACROS***********************************************************************/

/*TYPE*DEFINITIONS*************************************************************/

			/* Structure for storing a variables label (name) and */
			/* its associated value in a singly-linked list form. */
typedef
 struct ovnode
  {
    char *label;
    char *value;
    struct ovnode *next;
  }
 optvar_node;

			/* Structure for storing a directive in a	      */
			/* singly-linked list form.			      */
typedef
 struct odnode
  {
    char *directive;
    struct odnode *next;
  }
 optdir_node;

			/* Structure to store head pointers of all option     */
			/* linked lists, which currently consist of variables */
			/* and directives.				      */
typedef
 struct
  {
    optvar_node *vars;
    optdir_node *dirs;
  }
 options;

/*EXTERNAL*GLOBAL*VARIABLES****************************************************/

/*EXTERNAL*ROUTINES************************************************************/

			/*******************************************************
			* Routine to read a previously opened options file and *
			* parse it according to the operations specified in    *
			* the flags value passed. Routine returns a pointer to *
			* a structure containing the data retrieved from the   *
			* file.						       *
			********************************************************
			* optfile	Options file to read		       *
			* flags		pre-defined bit flags ORed together    *
			* cdel		delimeter to be assumed for comments   *
			* ddel		delimeter to be assumed for directives *
			*******************************************************/
options *ReadOptionsFile(FILE *optfile, char flags, char cdel, char ddel);

			/*******************************************************
			* Routine to parse a line passed and retrieve the item *
			* (field) specified assuming the specified delimeter.  *
			* It returns an independant pointer to this item. Item *
			* numbers start reference at index 1.		       *
			********************************************************
			* line		string to parse for item	       *
			* delim		delimeter to assume when parsing       *
			* select	which item to return (starts at 1)     *
			*******************************************************/
char *GetDelimItem(char *line, char delim, int select);

			/*******************************************************
			* Routine to get the data part of a directive and      *
			* allocate storage for it. It then assigns this memory *
			* to the pointer pointer passed to it.		       *
			********************************************************
			* line		line that contains the directive       *
			* directive	pointer pointer for new storage	       *
			*******************************************************/
void ParseDirective(char *line, char **directive);

			/*******************************************************
			* Routine to separate the values of a variable         *
			* assignment into its corresponding parts- label and   *
			* value. It as allocates memory for copies of them and *
			* sets the pointer pointers to the new memory.	       *
			********************************************************
			* line		line that contains the full assignment *
			* label		pointer pointer for variable label     *
			* value		pointer pointer for variable value     *
			*******************************************************/
void ParseVariable(char *line, char **label, char **value);

			/*******************************************************
			* Routine to return the value of an environment        *
			* variable if it exists; otherwise returns NULL.       *
			* Returns pointer to part of the structure (dependant) *
			********************************************************
			* opts		Options structure with lists to search *
			* label		Label associated with value desired    *
			*******************************************************/
char *GetVariableValue(options opts, char *label);

			/*******************************************************
			* Routine to return whether or not a directive is      *
			* defined or not. (Uses 0 => False & 1 => True for     *
			* easy code testing). 				       *
			********************************************************
			* opts		Options structure with lists to search *
			* directive	Directive to search for		       *
			*******************************************************/
char IsDirectiveDefined(options opts, char *directive);

			/*******************************************************
			* Routine to split a string up along a specified       *
			* delimeter with multiple handling options. Returns    *
			* Independant strings and leaves original intact.      *
			********************************************************
			* s		String to split			       *
			* del		Delimeter to split string by	       *
			* left		pointer to assign new string memory to *
			*		for the left side of the string        *
			* right		pointer to assign new string memory to *
			*		for the right side of the string       *
			* flags		Variable for setting the execution     *
			*               options	(listed above)		       *
			*******************************************************/
void SplitString(char *s, char del, char **left, char **right, char flags);
