/*
** rjg-crack - brute-force UNIX password cracker program.
**
** Copyright(C) June, 1996, by Reinhold J. Gerharz, Tracor Inc., TASI EESD TBTC
**
** "If an army of monkeys were strumming on typewriters they _might_ write
** all the books in the British Museum." -- Sir Arthur Stanley Eddington
**
** And so it is with this program. In time, it might, nay, WILL 
** crack the password for a given 13 character encoding string.
**
**DISCRIPTION:
** This program accepts a 13 character password string and attempts to
** crack it by trying all possible character combination strings. An
** optional second argument can be used as a starting string, usually read
** from the checkpoint file to resume cracking after a crash or abort.
**
** Every 13 minutes, a checkpoint file is written into the current working
** directory, containing the encoded password and test string. The filename
** is "rjg-crack.chk". If the checkpoint file already exists, it will be
** overwritten. This file can be used to restart the cracker. (see "./go")
**
**DISCLAIMER:
** This program was inspired by the need to crack a root user password on
** a used UNIX system delivered to my facility, for which the root password
** was apparently lost. Crack 4.1 was unsuccessful. Any use of this program
** for purposes other than that described above is a violation of Federal Law!
** (Well it ought to be!) Oh, and no guarantee or warranty or even a promise
** is made for its suitability for any purpose, not even that described above.
** The author, his employer, his parents, his wife, their pet hamster,
** his friends and acquaintances cannot be held liable or responsible for any
** use of this program, unless said person is indeed the same person as has
** misused this program. In that case, off with his head!
** This disclaimer will self destruct in ten seconds.
**
** USE AT YOUR OWN RISK!
**
**NOTES:
** 1) The checkpoint file is never written while the password is in the
**    process of being incremented.
** 2) Password string is right-aligned to obviate the need to shift
**    right when a carry operation requires lengthening the string.
** 3) A table, char_tbl[256], is created from char_set[] and used to speed
**    up the process of incrementing the trial password.
** 4) In order to remove the overhead of testing for the password length
**    exceeding the maximum every time it is incremented, the test is
**    performed only when the checkfile is written. Therefore, passwd[] is
**    allocated to MAX_PASSWORD + 1 size to give a "guard" position at the
**    head of the string. Since it would take strlen(char_set) times longer
**    to overflow the guard position than it took to reach the MAX_PASSWORD
**    limit, I think this approach is pretty damned reliable, don't you?
**    (This may need to be modified when Intel releases its LightSpeed
**    Optical Pentium Cube, code-named "P2e12". But by then crypt() will have
**    been replaced with retinal, or perhaps even DNA scanners.)
** 5) Since around 63 out of 64 times through the main loop the trial password
**    will have the wrong first character, substantial time savings have
**    been achieved by comparing the first character before call strcmp().
**
** x) Written under Linux. Should work under other OSes.
**    Adding use of fcrypt() should be trivial.
** y) Under Linux ELF, this program runs faster when linked statically. 
**    The do_benchmark function takes ~3.5 seconds when linked dyanamically
**    on my PC, versus ~2.8 seconds when linked statically!
** z) It can take a LONG time!
**
**REVISION HISTORY:
** v1.00 1996-06-05, RJG -- Initial Version.
** v1.01 1996-06-07, RJG -- Changed text string increment function from a table
**		     of indexes into char_set to a lookup table of incremented
**		     char values. This eliminated one level of indirection,
**		     some casting and miscellaneous arithmetic.
** v1.02 1996-08-16, RJG -- Added alternative collation sequence w/getenv()
**                   to compare the performance to that of another brute force
**                   password cracker which I did not know existed before
**                   I began writing this one.
**/

/*
** Include headers
*/
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


/*
** Global Constant Variables and Definitions
*/
const char Version[]=
"rjg-crack -- UNIX password cracker program, version 1.02";

const char Copyright[]=
"Copyright(C) June, 1996, by Reinhold J. Gerharz, Tracor Inc., TASI EESD TBTC";

const char CheckFilename[] = "rjg-crack.chk";

/* add any punctuation characters you want. in theory, if the target
   password has punctuation an equivalent, all alphanumeric password
   should eventually be found! These are the characters allowed in the
   crypt() man page, though I found no hard-coded limits in the code for
   UFC crypt. I have successfully set my own password to contain spaces
   and punctuation characters. */
const char	*char_set;
const char	default_char_set[]=
#if 0==1
/* this is the [correct] collating sequence as used by the crypt() function */
"./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
#else
/* this collating sequence assumes passwords begin with lower-case consonants
   or numerical digits before vowels or upper-case, thus it may be faster. */
"bcdfghjklmnpqrstvwx1234567890aeiouyz./BCDFGHJKLMNPQRSTVWXAEIOUYZ";
#endif

/* Hmm, do_benchmark reports the following define runs faster than defining
   bool as unsigned char which is in turn much faster than typedefing it
   either int or unsigned char. How strange... */
#define		bool		int

#define		TRUE		(1==1)
#define		FALSE		(1==0)
#define		PASSWD_LENGTH	13	/* UNIX password encoding length */

#define		MAX_PASSWORD	8	/* this is an arbitrary limit */

/*
** Global Variables
*/

/* passwd[] is the right-aligned password under consideration.
   pe points to its last character. tags make it easy to find in core */
char		tag1[]="PASSWORD CRACKER TAG1";
char 		passwd[MAX_PASSWORD+2];	/* add room for null & overflow */
char		tag2[]="PASSWORD CRACKER TAG2";
char		pw[PASSWD_LENGTH+1];	/* add room for null */
char		tag3[]="PASSWORD CRACKER TAG3";

/* pe points to the end (last character) of passwd[].
   pb points to the first character of the right-aligned passwd[] string. it
   is decrement to point further left whenever the string needs to grow. */
static char    	*pe = passwd+MAX_PASSWORD;
static char    	*pb = passwd+MAX_PASSWORD;

/* cp1 and cp2 are pointers used for fast integer comparison of the trial
   password encoding. cp2 assumes crypt() always returns the same buffer.
   These pointers probably fall on odd 16 bit boundaries, and may cause
   poor performance on machines which require 32 or 64 bit alignment
   for optimum performance. short is faster than char or long, on my PC. */
#define		C_TYPE		short
static C_TYPE	*cp1, *cp2;		/* WOW! static sure helps! ~10% */

int		check_file;		/* file descriptor */

/* char_tbl[256] is set up so each element indexed by an element of char_set[]
   contains the value of index, incremented. thus, incrementing a character
   is speeded up by merely looking in this table to find its next value! */
static int	char_tbl[256];

/* do_signal means a signal must be processed before performing another
   password check iteration. */
static bool	do_signal = FALSE;

/* do_checkpoint is TRUE when a checkpoint timer interval has expired */
bool		do_checkpoint=FALSE;

/* do_abort is TRUE when a process termination signal has been received */
bool		do_abort=FALSE;

/*
** function prototype declarations
*/
void initialize_char_table(const char *char_set);
void do_benchmark(void);
bool do_crack(char *pwd, char *seed);
__inline__ void increment_password(void);
__inline__ bool increment_char(char *p);
void set_timer(void);
void set_interrupt(void);


/*
** Program Begins...
*/

/************************************************************************
*									*
*			    CRACK ROUTINES				*
*									*
************************************************************************/

int main(int argc, char *argv[])
{
    fprintf(stderr,"\n%s\n%s\n", Version, Copyright);

    /* check for arguments */
    if(argc == 1 || strlen(argv[1]) != PASSWD_LENGTH) {
	printf("\nUsage -- %s <encoded password> [<seed>]\n\n"
    "\twhere <encoded password> is the 13 character string from /etc/passwd\n"
    "\tand <seed> is an optional password string to use as a starting point.\n"
    "\n<seed> is usually taken from the checkpoint file \"rjg-crack.chk\" to"
    "\nresume cracking after a crash or abort. Under 'sh' shell, for example:"
    "\n\t%s frxWbx4IRuBBA `cat rjg-crack.chk`\nshould resume a"
    " previous cracking session from the last checkpoint.\n\n",
	       argv[0], argv[0]);
	exit(1);
    }

    /* sanity check lifted from Alec David Edward Muffett's Crack 4.1 */
    if (strcmp (crypt ("fredfred", "fr"), "frxWbx4IRuBBA")) {
        printf("Version of crypt() is not compatible with standard.\n");
        exit(2);
    }

    /* get the collating string */
    char_set = getenv("CRACK_COLLATING_STRING");
    if(!char_set)
	char_set = default_char_set;
    
    /* setup char_tbl, cp1, cp2. do a quick benchmark */
    initialize_char_table(char_set);
    cp1 = (C_TYPE *)(pw+2);		  /* point past "salt" */
    cp2 = (C_TYPE *)(crypt("xY", "rg") +2); /* pointer to internal buffer */
    assert(cp2 == (C_TYPE *)(crypt("rg", "xY")+2));
    do_benchmark();

    /* open checkpoint file and set up signal handlers */
    if((check_file=open(CheckFilename,
			O_WRONLY|O_NONBLOCK|O_CREAT, 0600)) == -1) {
	perror(CheckFilename);
	exit(3);
    }
    set_timer();
    set_interrupt();

    /* find the password and print it */
    if(do_crack(argv[1], (argc > 2) ? argv[2] : 0)) {
	printf("Found it! Password for \"%s\" is:\n\"%s\"\n", argv[1], pb);
    }
    else {
	printf("Failed to find password for \"%s\"!\n", argv[1]);
	return 99;
    }

    return 0;
}



/*
** initialize_char_table() -- initializes the character table used to determine
** 			      which character comes next when incrementing
**			      the password guess string.
*/

void initialize_char_table(const char *char_set)
{
    int		i;

    memset(char_tbl, 0, sizeof(char_tbl));

    /* incrementing terminating '\0' results in looping to first char */
    char_tbl[0] = char_set[0];

    /* fill rest of char_tbl so that the character to be incremented
       used as an index into char_tbl returns the incremental result.
       increment_char and increment_passord check for NULL ('\0'), which
       causes it to perform carry to the next character position. */
    
    for(i=0; i < strlen(char_set); i++) {
	char_tbl[(int)char_set[i]] = char_set[i+1];
    }
    
}



/*
** increment_password() -- increments the password string and
** 			   index array. Perform carry and lengthen
**			   passwd[] when necessary.
*/

__inline__ void increment_password(void)
{
    char	*cp;

    cp = pe;
    while(increment_char(cp)) {
	cp--;
	if(cp < pb) {
	    /* carry into unused position: lengthen the password string */
	    pb--;
	}
    }		/* loop until no more carrys */
    
}



/*
** increment_char() -- increments one character, returning TRUE if overflow.
**
** returning TRUE means caller must perform a carry operation.
*/

__inline__ bool increment_char(char *cp)
{
    /* increment the character by looking up the next value in char_tbl[] */
    *cp = char_tbl[(int)(*cp)];
    
    if(*cp) {		/* if not null, no carry required */
	return FALSE;
    }
    else {		/* reset to first character and flag carry */
	*cp = char_tbl[0];
	return TRUE;
    }
}



/*
** do_benchmark() -- if this takes forever. something's very wrong!
*/

void do_benchmark(void)
{
    char	pw[13];
    float	bench_seconds;
    clock_t	t1, t2;
    
    strcpy(pw, crypt("xY", "rg"));

    t1 = clock();
    do_crack(pw, 0);
    t2 = clock();

    bench_seconds = ((float)(t2-t1)) / CLOCKS_PER_SEC;

    if(strcmp("xY", pb)) {
	fprintf(stderr, "Failed benchmark test to crack password \"xY\"\n");
	exit(9);
    }

    printf("Simple benchmark test passed in %.2f seconds.\n", bench_seconds);
    
}


/*
** do_crack(password, seed) -- cracks the password.
*/

bool do_crack(char *pwd, char *seed)
{
    if(seed) {
	pb = pe - strlen(seed)+1;
	strcpy(pb, seed);
    }
    else {
	pb = pe;
	pe[0] = char_set[0];
	pe[1] = '\0';
    }

    passwd[0] = '\0';
    strcpy(pw, pwd);
    
/* loop until password is found */
LOOP:
    /* test the password by calling crypt() and comparing result. */
    crypt(pb, pw);

    /* Hmm... This is slower (and expression vs. compound if):
       if(*cp1 == *cp2) && (strcmp(... */
    /* This way is faster: */
    if(*cp1 == *cp2) if (strcmp((char*)&cp1[1], (char*)&cp2[1]) == 0) 
    {
	return TRUE;
    }

    increment_password();

    /* for the vast majority of execution loops, this is the only conditional
       test, thus reducing per-loop execution overhead. */
    if(!do_signal)
	goto LOOP;

    if(do_checkpoint) {
	char		s[80];
	if(lseek(check_file, 0, SEEK_SET) == -1 ||
	   write(check_file, s, sprintf(s, "%s %s\n", pw, pb)) == -1) {
	    perror("writing checkpoint file");
	    exit(98);
	}
	sync();
	do_checkpoint = FALSE;

	/* test for overflow */
	if(passwd[0]) {
	    printf("All possible passwords were tried.\n");
	    return FALSE;
	}
	
    }
    
    if(do_abort) {
	printf("SIGINT received while about to test \"%s\"\n", pb);
    }
    else {
	goto LOOP;
    }
    
    return FALSE;	/* FAILED! */
}



/************************************************************************
*									*
*			    SIGNAL ROUTINES				*
*									*
************************************************************************/

void sig_handler(int);


/*
** set_timer() - sets up an alarm signal
*/

void set_timer(void) 
{
    if(signal(SIGALRM, sig_handler) == SIG_ERR) {
	perror("setting checkpoint timer");
	exit(97);
    }
    if(signal(SIGHUP, sig_handler) == SIG_ERR) {
	perror("setting hangup handler");
	exit(94);
    }
    alarm(13 * 60L);		/* 13 minutes */
}



/*
** set_interrupt() - sets termination signal handler
*/

void set_interrupt(void)
{
    if(signal(SIGINT, sig_handler) == SIG_ERR) {
	perror("setting SIGINT handler");
	exit(96);
    }
    if(signal(SIGQUIT, sig_handler) == SIG_ERR) {
	perror("setting SIGQUIT handler");
	exit(95);
    }
}




/*
** sig_handler() - handler for both SIGALRM/SIGHUP and
**		   SIGINT/SIGQUIT signals.
*/

void sig_handler(int signum)
{
    switch(signum){

    case SIGALRM:
    case SIGHUP:
	set_timer();
	do_checkpoint = TRUE;
	do_signal = TRUE;
	break;

    case SIGINT:
    case SIGQUIT:
	do_checkpoint = TRUE;
	do_abort = TRUE;
	do_signal = TRUE;
        break;
	
    }
}
