#include <config.h>
#include "basic.h"
#include "n_errno.h"
#include "jnetd.h"
#include "socketlb.h"
#include "parser.h"
#include "log.h"
#include "n_file.h"
#ifdef HAVE_PASSWD_H
# include <passwd.h>
#endif
#ifdef HAVE_PWD_H
# include <pwd.h>
#endif
#ifdef HAVE_GRP_H
# include <grp.h>
#endif
#ifdef HAVE_SHADOW_H
# include <shadow.h>
#endif
#include "tools.h"

/* Returns the name of a service according to a ServiceContext structure :
 * its description, the port name or the port number */

char *tools_servicenametxt(const ServiceContext * const servicecontext)
{
   static char buf[TOOLS_MAXPORTSTRINGLEN + 1];
   
   if (servicecontext->description != NULL) {
      return servicecontext->description;
   } else if (servicecontext->servicename.name != NULL) {
      return servicecontext->servicename.name;
   }
#ifdef HAVE_SNPRINTF
   snprintf(buf, TOOLS_MAXPORTSTRINGLEN, "%d", servicecontext->servicename.port);   
#else
   sprintf(buf, "%d", servicecontext->servicename.port);
#endif
   
   return buf;
}

struct env_table {
    const char *name;
    const int len;
};

static const struct env_table badenv_table[] = {
   { (const char *) "IFS=", 4 },
   { (const char *) "LD_", 3 },
   { (const char *) "_RLD_", 5 },
   { (const char *) "SHLIB_PATH=", 11 },
   { (const char *) "LIBPATH=", 8 },
   { (const char *) "KRB_CONF", 8 },
   { (const char *) "ENV=", 4 },
   { (const char *) "BASH_ENV=", 9 },
   { NULL, 0 }
};

/* Cleans some nasty environment variables */

extern char **environ;   

void tools_cleanenv(void)
{
   char **envp = environ;
   const struct env_table *bad;
   char **cur;
   
   for (cur = envp; *cur; cur++) {
      for (bad = badenv_table; bad->name; bad++) {
	 if (strncmp(*cur, bad->name, (size_t) bad->len) == 0) {
	    char **move;
	    
	    for (move = cur; *move; move++) {
	       *move = *(move + 1);
	    }	    
	    cur--;	    
	    break;
	 }
      }
   }
}

/* Change the current uid/gid of the current process */

int tools_changeid(const char * const user, const char * const group)
{
   struct passwd *pwd;
   struct group *grp;

   if (group != NULL) {
      grp = getgrnam(group);
   }
   if (user != NULL && geteuid() != 0) {
      log_log(LOG_ERR, NULL, 
	      _("Need supervisor priviledges to change uid/gid"));
      
      return -1;
   } else if (user == NULL) {
      return 0;
   }
   if ((pwd = getpwnam(user)) == NULL) {
      log_log(LOG_ERR, NULL,
	      _("Can't find any user called [%s] - Process not launched"),
	      user);
      
      return -1;
   }
   if (group != NULL && (grp = getgrnam(group)) != NULL) {
      if (setgid(grp->gr_gid) < 0) {
	 log_log(LOG_ERR, NULL, _("Unable to switch to group [%s] <%s>"),
		 group, newstrerror(errno));
	 
	 return -1;
      }
   } else {
      if (setgid(pwd->pw_gid) < 0) {
	 log_log(LOG_ERR, NULL, _("Can't set default user group <%s>"),
		 newstrerror(errno));
	 
	 return -1;
      }
   }
#ifdef HAVE_INITGROUPS
   if (group == NULL && initgroups(user, pwd->pw_gid) < 0) {
      log_log(LOG_ERR, NULL,
	      _("Can't set user groups for [%s] <%s>"), 
	      user, newstrerror(errno));
      
      return -1;
   }
#endif
#ifdef HAVE_SETRESUID
   if (setresuid(pwd->pw_uid, pwd->pw_uid, 0) < 0) {
      log_log(LOG_ERR, NULL, _("Unable to switch to user [%s] <%s>"), 
	      user, newstrerror(errno));
      
      return -1;
   }
#else
   if (setuid(pwd->pw_uid) < 0) {
      log_log(LOG_ERR, NULL, _("Unable to switch to user [%s] <%s>"),
	      user, newstrerror(errno));
      
      return -1;
   }
# ifdef HAVE_SETEUID
   if (seteuid(pwd->pw_uid) < 0) {
      log_log(LOG_ERR, NULL, _("Unable to switch to user [%s] <%s>"), 
	      user, newstrerror(errno));
      
      return -1;
   }
# endif
#endif
   
   return 0;
}

/* Return 1 if a name matches the regex */

int tools_namesmatchrx(const char **names, regex_t *rx)
{
   if (rx == NULL) {
      return -1;
   }
   while (*names != NULL) {
      if (regexec(rx, *names, 0, 0, 0) == 0) {
	 return 1;
      }
      names++;
   }
      
   return 0;
}

/* Needed for GNUregex */

void printchar(int c)
{
   putchar(c);
}

/* See, according to the IP options we found, if we should hangup or not */

const char *tools_seeipoptions(const SockEvent * const event)
{   
   SocketOptions options;
   
   if (socket == NULL) {
      return NULL;
   }
   options = event->options;
   
   if ((options & SOCKET_OPTION_SSRR)) {
      return _("strict source routing enabled");
   }
   if ((options & SOCKET_OPTION_LSRR)) {
      return _("loose source routing enabled");
   }
   
   return NULL;
}

/* The following function adapted from Stevens, "Advanced Programming in the
 * Unix Environment", p. 418,  initializes the the standalone daemon */

int tools_initdaemon(void)
{
   int i;
   
   pid_t pid, procgp;
   
   if ((pid = fork()) < 0) {
     return -1;
   } else if (pid != 0) {
      exit(EXIT_SUCCESS);
   }
   
#ifdef HAVE_SETSID
   if ((procgp = setsid()) == -1) {
      exit(EXIT_FAILURE);
   }
#else
# ifdef HAVE_SETPGRP
#  ifdef SETPGRP_VOID
   if ((procgp = setpgrp()) == -1) {
      exit(EXIT_FAILURE);
   }     
#  else
   if ((procgp = setpgrp(getpid(), 0)) == -1) {
      exit(EXIT_FAILURE);
   }
#  endif
# endif
#endif    
   umask(0);   
   for (i = 3; i < SOCKMAX_FD; i++) {
      close(i);
   }
   return 0;
}

