
/* (C)oderight 1997 by Jedi/Sector One <j@4u.net> */

#include <config.h>
#include "basic.h"
#include "n_errno.h"
#include "n_string.h"
#include "socket.h"
#include "socketlb.h"

#if MAXHOSTNAMELEN < 64
# undef MAXHOSTNAMELEN
#endif  
#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN  256             /* storage for host name */
#endif                 

extern int errno;

/* Just a stupid malloc() wrapper */

static void *socket_malloc(const size_t size)
{
#ifdef MALLOC_NEEDS_A_MINIMUM
   if (size < (size_t) MALLOC_NEEDS_A_MINIMUM) {
      size = (size_t) MALLOC_NEEDS_A_MINIMUM;
   }
#else
   if (size == 0) {
      return NULL;
   }
#endif
   return (void *) malloc(size);
}

/* Just a stupid free() wrapper */

static int socket_free(void * const pnt)
{
   if (pnt == NULL) {
      return -1;
   }
   free(pnt);
   
   return 0;
}

/* Endpoint name -> Endpoint constant */

static int parsendpoint(const char * const endpointstring)
{
   if (strcasecmp(endpointstring, "stream") == 0) {
      return SOCK_STREAM;      
   }
#ifdef SOCK_DGRAM
   else if (strcasecmp(endpointstring, "dgram") == 0) {
      return SOCK_DGRAM;
   } 
#endif
#ifdef SOCK_RDM   
   else if (strcasecmp(endpointstring, "rdm") == 0) {      
      return SOCK_RDM;
   } 
#endif
#ifdef SOCK_SEQPACKET
   else if (strcasecmp(endpointstring, "seqpacket") == 0) {
      return SOCK_SEQPACKET;
   } 
#endif
#ifdef SOCK_RAW
   else if (strcasecmp(endpointstring, "raw") == 0) {
      return SOCK_RAW;
   } 
#endif
   else if (strcasecmp(endpointstring, "tli") == 0) {
      abort();
   }
   
   return -1;
}

/* Find a host : either from the hostnameenv environment variable,
 * or from hostname_ . Return a hostent structure */

static struct hostent *socket_gethostent(const char *hostname_,
					 const char * const hostnameenv)
{
   const char *hostname;
   
   if (hostnameenv == NULL || 
       (hostname = (char *) getenv(hostnameenv)) == NULL) {
      hostname = hostname_;
   }
#ifdef IPV6_SUPPORT
   {
      struct hostent *ent;
      
      if ((ent = gethostbyname2(hostname, AF_INET6)) == NULL) {
	 return gethostbyname2(hostname, AF_INET);
      }
      return ent;
   }
#else
   return gethostbyname(hostname);
#endif
}

/* Find a port number */

static unsigned short socket_getport(const char * const protocol,
				     const char * const portname,
				     const unsigned short port)
{
   struct servent *service;

   if ((service = getservbyname(portname, protocol)) != NULL) {
      return service->s_port;
   }
   return htons(port);
}

/* Find a protocol (default = def) */

static int socket_getproto(const char * const protoname, const int def)
{
   struct protoent *proto;
   
   if (protoname == NULL || *protoname == 0 ||
       (proto = getprotobyname(protoname)) == NULL) {      
      return def;
   } 
   return proto->p_proto;
}

/* These options are systematically added to any socket */

static int socket_setopts(const int fd)
{
   socklen_t opt = 1;
   
   setsockopt(fd, socket_getproto("ip", IPPROTO_IP), IP_OPTIONS, NULL, 0);
   setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &opt, sizeof opt);
#ifdef SO_USELOOPBACK
   setsockopt(fd, SOL_SOCKET, SO_USELOOPBACK, (char *) &opt, sizeof opt);
#endif
   opt = 0;
#ifdef SO_BROADCAST
   setsockopt(fd, SOL_SOCKET, SO_BROADCAST, (char *) &opt, sizeof opt);
#endif
   
   return 0;
}

enum {
   MAX_IP_OPTIONS = 256
};

/* Get the options of a socket */

static int socket_fixopts(Socket * const socket, const int fd)
{
#if defined (IP_OPTIONS) && defined (IPOPT_OPTVAL)

# define FIRST_HOP_IN_GETSOCKOPT
# if defined (__linux__)
#  undef FIRST_HOP_IN_GETSOCKOPT
# endif
      
   static unsigned char optbuf[MAX_IP_OPTIONS];
   Sockets * const sockets = socket->sockets;
   socklen_t optbuflen = sizeof optbuf;
   size_t optlen = 0;
   unsigned char *optbufpnt = optbuf;

   if (getsockopt(fd, socket_getproto("ip", IPPROTO_IP), IP_OPTIONS,
		  (char *) optbuf, &optbuflen) == 0 &&
       optbuflen != 0) {
      
# ifdef FIRST_HOP_IN_GETSOCKOPT
      {
#  ifdef IPV6_SUPPORT
	 struct in6_addr foo;
#  else
	 struct in_addr foo;
#  endif
	 (void) foo;
	 optbufpnt += sizeof foo.s_addr;
	 optbuflen -= sizeof foo.s_addr;
      }
# endif
      while (optbuflen > 0) {
	 switch (optbuf[IPOPT_OPTVAL]) {

# ifdef IPOPT_RR	    
	  case IPOPT_RR :	    
	    sockets->sockevent.options |= SOCKET_OPTION_RR;
	    break;
# endif

# ifdef IPOPT_RA
	  case IPOPT_RA :
	    sockets->sockevent.options |= SOCKET_OPTION_RA;
	    break;
# endif	    
	    
# if defined (IPOPT_TS) || defined (IPOPT_TIMESTAMP)
#  ifdef IPOPT_TS
	  case IPOPT_TS :
#  else
	  case IPOPT_TIMESTAMP :
#  endif
	    sockets->sockevent.options |= SOCKET_OPTION_TS;	    
	    break;
# endif

# if defined (IPOPT_SATID) || defined (IPOPT_SID)
#  ifdef IPOPT_SATID
	  case IPOPT_SATID :
#  else
	  case IPOPT_SID :
#  endif
	    sockets->sockevent.options |= SOCKET_OPTION_SATID;	    
	    break;
# endif
	    
# ifdef IPOPT_SSRR
	  case IPOPT_SSRR :
	    sockets->sockevent.options |= SOCKET_OPTION_SSRR;	    
	    break;
# endif
	  	    
# ifdef IPOPT_LSRR	    
	  case IPOPT_LSRR :
	    sockets->sockevent.options |= SOCKET_OPTION_LSRR;	    
	    break;
# endif
	    
# if defined (IPOPT_SECURITY) || defined (IPOPT_SEC)
#  ifdef IPOPT_SECURITY	    
	  case IPOPT_SECURITY :
#  else
	  case IPOPT_SEC :
#  endif
	    sockets->sockevent.options |= SOCKET_OPTION_SECURITY;	    
	    break;
# endif
	    
# if defined (IPOPT_EOL) || defined (IPOPT_END)
#  ifdef IPOPT_EOL
	  case IPOPT_EOL :
#  else
	  case IPOPT_END :
#  endif	    
	    goto endopt;
# endif
	    
# if defined (IPOPT_NOP) || defined (IPOPT_NOOP)
#  ifdef IPOPT_NOP
	  case IPOPT_NOP :
#  else
	  case IPOPT_NOOP :
#  endif
	    optlen = 1;
	    break;
# endif
	  default :
	    if ((optlen = optbuf[IPOPT_OLEN]) == 0) {
	       goto endopt;
	    }
	 }	 
	 optbufpnt += optlen;
	 optbuflen -= optlen;
      }
   }
   endopt:   
   
#endif
   return 0;
}

/* Create a new basic socket (internal) */

static int socket_create(const int endpoint,
			 const struct protoent * const proto,
#ifdef IPV6_SUPPORT
			 struct sockaddr_in6 * const local
#else
			 struct sockaddr_in * const local
#endif			 
			 )
{
   int fd;

#ifdef IPV6_SUPPORT
   local->sin6_family = AF_INET6;
#else
   local->sin_family = AF_INET;
#endif
#ifdef IPV6_SUPPORT
   if ((fd = socket(PF_INET6, endpoint, proto->p_proto)) < 0 ||
       socket_setopts(fd) < 0 ||
       bind(fd, (struct sockaddr *) local, sizeof *local) < 0) {
      return -1;
   }
#else
   if ((fd = socket(PF_INET, endpoint, proto->p_proto)) < 0 ||   
          socket_setopts(fd) < 0 ||
              bind(fd, (struct sockaddr *) local, sizeof *local) < 0) {
            return -1; 
   }
#endif
      
   return fd;
}

/* Return 1 if fd points to an IPv6 socket */

int socket_isipv6(const int fd)
{
#ifdef IPV6_SUPPORT
   int  addrform;
   socklen_t len = sizeof(addrform);

   if (getsockopt(fd, IPPROTO_IPV6, IPV6_ADDRFORM,
		  (char *) &addrform, &len) == -1) {
      return -1;
   } else if (addrform == PF_INET) {
      return 0;
   } else if (addrform == PF_INET6) {
      return 1;
   } 
   return -2;
#else
   (void) fd;
   return 0;
#endif
}

/* Add the NODELAY option to an existing socket */

int socket_optionnodelay(Socket * const socket, const SocketOptionFlag on)
{
   socket->optionnodelay = on;
   return 0;
}

/* Add the KEEPALIVE option to an existing socket */

int socket_optionkeepalive(Socket * const socket, const SocketOptionFlag on)
{
   socket->optionkeepalive = on;
   return 0;
}

/* Let the user add any fd he wants to listen to.
 * This can be especially useful if you want to monitor the keyboard
 * while accepting network connexions */

int socket_fdlisten(Sockets * const sockets, const int fd)
{
   if (sockets == NULL) {
      return -1;
   }  
   if (fd >= sockets->lastfd) {
      if (fd < SOCKMAX_FD) {
	 sockets->lastfd = fd + 1;
      } else {
	 return -1;
      }
   } 
   FD_SET(fd, &sockets->rfdsbackup);   	 
   
   return 0;
}

/* Stop listening to a socket */

int socket_fdunlisten(Sockets * const sockets, const int fd)
{
   if (sockets == NULL) {
      return -1;
   }   
   FD_CLR(fd, &sockets->rfdsbackup);
   if (sockets->lastfd < (fd + 1) && sockets->lastfd > 1) {
      sockets->lastfd--;
   }
   
   return 0;
}

/* Does that socket use a stream endpoint ? */

int socket_isstream(const Socket * const socket)
{
   if (socket == NULL) {
      return -1;
   }
   return (socket->endpoint == SOCK_STREAM);
}

/* Add a socket to the global list */

static int socket_listadd(Sockets * const sockets, Socket * const socket)
{
   SocketListItem *socketlistitem;
   
   if (sockets == NULL || socket == NULL) {
      return -1;
   }
   if ((socketlistitem = (SocketListItem *)
	socket_malloc(sizeof *socketlistitem)) == NULL) {
      return -1;
   }
   socketlistitem->socket = socket;
   socketlistitem->next = NULL;
   if (sockets->listlastitem == NULL) {
      sockets->socketlist = socketlistitem;
   } else {
      sockets->listlastitem->next = socketlistitem;      
   }
   sockets->listlastitem = socketlistitem;            
   
   return 0;
}

/* Create a new client socket */

Socket * socket_clientcreate(Sockets * const sockets,			      
			     const char * const hostname, 
			     const char * const hostnameenv,
			     const char * const endpoint,			     
			     const char * const protocol,
			     const char * const portname, unsigned short port)
{
   Socket *socket;
   struct hostent *h;
   struct protoent *proto;
#ifdef IPV6_SUPPORT
   struct sockaddr_in6 server, local;
#else
   struct sockaddr_in server, local;
#endif
   char * const *addrpnt;
   int fd;   

   if ((socket = (Socket *) socket_malloc(sizeof *socket)) == NULL) {
      return NULL;
   }
   if ((h = socket_gethostent(hostname, hostnameenv)) == NULL) {
      return NULL;
   }
   if ((socket->endpoint = parsendpoint(endpoint)) == 0) {
      return NULL;
   }
   port = socket_getport(protocol, portname, port);
   socket->port = port;
   memset(&server, 0, sizeof server);
   memset(&local, 0, sizeof local);
#ifdef IPV6_SUPPORT
   server.sin6_family = local.sin6_family = AF_INET6;
   server.sin6_flowinfo = local.sin6_flowinfo = 0;
   local.sin6_addr = in6addr_any;
   server.sin6_port = port;
   local.sin6_port = htons(0);   
#else
   server.sin_family = local.sin_family = AF_INET;
   local.sin_addr.s_addr = INADDR_ANY;
   server.sin_port = port;
   local.sin_port = htons(0);   
#endif
   proto = getprotobyname(protocol);
   addrpnt = h->h_addr_list;
   while (*addrpnt != NULL) {
#ifdef IPV6_SUPPORT
      memcpy(&server.sin6_addr, *addrpnt, (size_t) h->h_length);      
#else
      memcpy(&server.sin_addr, *addrpnt, (size_t) h->h_length);
#endif
      if ((fd = socket_create(socket->endpoint, proto, &local)) < 0) {

	 return NULL;
      }
      socket->addr = local;
      if (connect(fd, (struct sockaddr *) &server, sizeof server) == 0) {
	 goto getdown;
      }
      close(fd);
      addrpnt++;
   }
   return NULL;
   
   getdown:

   socket->sockets = sockets;
   socket->cookie = socket->cookie2 = 0UL;
   socket->geek = socket->geek2 = NULL;
   socket->fd = fd;
   socket->optionnodelay = socket->optionkeepalive = SOCKET_OPTION_DEFAULT;

   sockets->fdtosocket[fd] = socket;
   if (fd >= sockets->lastfd) {
      if (fd < SOCKMAX_FD) {
	 sockets->lastfd = fd + 1;
      } else {
	 close(fd);
	 FD_CLR(fd, &sockets->rfdsbackup);
	 sockets->lastfd = SOCKMAX_FD;
	 sockets->sockevent.fdoverflow = 1;
	 sockets->sockevent.type = SOCKEVENT_TOOMANY;	 
	 return NULL;
      }
   }   
   FD_SET(fd, &sockets->rfdsbackup);
      
   return socket;
}

/* Initialize a Sockets structure : the parent container */

int socket_initsockets(Sockets * const sockets)
{
   register Socket **fdtosocketpnt;
   int t = SOCKMAX_FD + 1;
   
   if (sockets == NULL) {
      return -1;
   }   
   sockets->first = 1;
   sockets->socketlist = sockets->listlastitem = NULL;
   fdtosocketpnt = sockets->fdtosocket;
   while (t != 0) {
      *fdtosocketpnt++ = NULL;
      t--;
   }
   FD_ZERO(&sockets->lastrfds);      
   FD_ZERO(&sockets->rfdsbackup);
   sockets->sockevent.type = SOCKEVENT_NONE;
   sockets->pendingmessages = 0;
   sockets->pendingsync = SOCKET_PENDINGSYNC;
   sockets->cookie = sockets->cookie2 = 0UL;   
   sockets->geek = sockets->geek2 = NULL;
   
   return 0;
}

/* Close all existing sockets */

int socket_closeall(Sockets * const sockets)
{
   SocketListItem *socketlistpnt;
   SocketListItem *oldsocketlistpnt;
   
   if (sockets == NULL) {
      return -1;
   }
   socketlistpnt = sockets->socketlist;
   while (socketlistpnt != NULL) {
      socket_serverclose(socketlistpnt->socket);
      oldsocketlistpnt = socketlistpnt;
      socketlistpnt = socketlistpnt->next;
      socket_free(oldsocketlistpnt);            
   }
     
   return 0;
}

/* Remove a socket from the global list */

static int socket_listremove(Sockets * const sockets, Socket * const socket)
{
   SocketListItem *socketlistpnt;
   SocketListItem *oldsocketlistpnt = NULL;
   
   if (sockets == NULL || socket == NULL) {
      return -2;
   }
   socketlistpnt = sockets->socketlist;
   while (socketlistpnt != NULL) {
      if (socketlistpnt->socket == socket) {
	 if (oldsocketlistpnt != NULL) {
	    oldsocketlistpnt->next = socketlistpnt->next;
	 } else {
	    sockets->socketlist = socketlistpnt->next;
	 }
	 if (socketlistpnt->next == NULL) {
	    sockets->listlastitem = oldsocketlistpnt;
	 }
	 socket_free(socketlistpnt);
	 
	 return 0;
      }
      oldsocketlistpnt = socketlistpnt;
      socketlistpnt = socketlistpnt->next;
   }
   
   return -1;
}

/* Create a server */

Socket * socket_servercreate(Sockets * const sockets,
			     const char * const hostname,
			     const char * const hostnameenv,
			     const char * const endpoint,			     
			     const char * const protocol,
			     const char * const portname, unsigned short port)
{
   Socket *socket;
   struct hostent *h;
   struct protoent *proto;
#ifdef IPV6_SUPPORT
   struct sockaddr_in6 server, local;
#else
   struct sockaddr_in server, local;
#endif
   int fd;   
   
   if ((socket = (Socket *) socket_malloc(sizeof *socket)) == NULL) {
      return NULL;
   }
   socket->sockets = sockets;
   socket->cookie = socket->cookie2 = 0UL;
   socket->geek = socket->geek2 = NULL;
   socket->optionnodelay = socket->optionkeepalive = SOCKET_OPTION_DEFAULT;
   if ((socket->endpoint = parsendpoint(endpoint)) < 0) {
      return NULL;
   }
   port = socket_getport(protocol, portname, port);
   memset(&server, 0, sizeof server);
   memset(&local, 0, sizeof local);
#ifdef IPV6_SUPPORT
   local.sin6_port = port;   
#else
   local.sin_port = port;
#endif
   socket->port = port;
   proto = getprotobyname(protocol);
   if (hostname != NULL) {
      if ((h = socket_gethostent(hostname, hostnameenv)) == NULL) {
	 return NULL;
      }   
#ifdef IPV6_SUPPORT
      memcpy(&local.sin6_addr, *h->h_addr_list, (size_t) h->h_length);
#else
      memcpy(&local.sin_addr, *h->h_addr_list, (size_t) h->h_length);
#endif
   } else {
#ifdef IPV6_SUPPORT
      local.sin6_flowinfo = server.sin6_flowinfo = 0;
      local.sin6_addr = in6addr_any;
#else
      local.sin_addr.s_addr = INADDR_ANY;
#endif
   }
   socket->addr = local;
   if (socket->endpoint == SOCK_STREAM) {
      if ((fd = socket_create(SOCK_STREAM, proto, &local)) < 0 || 
	  listen(fd, SOCKET_BACKLOG) < 0) {
	 
	 return NULL;
      }   
   } else {
      if ((fd = socket_create(socket->endpoint, proto, &local)) < 0) {
	 
	 return NULL;
      }
   }
   socket->fd = fd;   
   socket_listadd(sockets, socket);
   sockets->fdtosocket[fd] = socket;
   if (fd >= sockets->lastfd) {
      if (fd < SOCKMAX_FD) {
	 sockets->lastfd = fd + 1;
      } else {
	 close(fd);
	 FD_CLR(fd, &sockets->rfdsbackup);
	 sockets->lastfd = SOCKMAX_FD;
	 sockets->sockevent.fdoverflow = 1;
	 sockets->sockevent.type = SOCKEVENT_TOOMANY;	 
	 return NULL;
      }
   }   
   FD_SET(fd, &sockets->rfdsbackup);

   return socket;
}

/* Close a server */

int socket_serverclose(Socket * const socket)
{
   Sockets * sockets;
   int fd;
   
   if (socket == NULL) {
      return -1;
   }
   if ((fd = socket->fd) < 0) {
      return -1;
   }
   close(fd);
   sockets = socket->sockets;   
   sockets->fdtosocket[fd] = NULL;
   socket_listremove(sockets, socket);   
   socket_free(socket);
   FD_CLR(fd, &sockets->rfdsbackup);
   if (sockets->lastfd < (fd + 1) && sockets->lastfd > 1) {
      sockets->lastfd--;
   }
   
   return 0;   
}

/* Close a client */

int socket_clientclose(Socket * const socket)
{
   return socket_serverclose(socket);
}

/* Hang up a file descriptor */

int socket_hangup(Socket * const socket, const int fd)
{
   if (socket == NULL) {
      return -1;
   }
   if (socket->endpoint != SOCK_STREAM) {
      return 0;
   }
   close(fd);
   FD_CLR(fd, &socket->sockets->rfdsbackup);
   if (fd == socket->sockets->lastfd - 1) {
      socket->sockets->lastfd = fd;
   }
   socket->sockets->sockevent.remotecnx[fd] = 0;
   socket->sockets->fdtosocket[fd] = NULL;
   
   return 0;   
}

/* Write a null-terminated string to a file descriptor */

int socket_print(const int fd, const char * const str)
{
   if (str == NULL) {
      return -1;
   }
   return write(fd, str, strlen(str));
}

/* Read pending data from a file descriptor */

char *socket_read(const int fd)
{
   static char line[SOCKLINE_MAX + 1];
   int len;

   if ((len = read(fd, line, SOCKLINE_MAX)) < 0 &&
       errno != EWOULDBLOCK) {
      return NULL;
   }
   line[len] = 0;
   
   return line;
}

/* Especially useful for UDP datagrams we want to discard */

int socket_flush(const int fd)
{
   char tmpbuf[SOCKLINE_MAX];
   recv(fd, tmpbuf, sizeof tmpbuf, 0);
   
   return 0;
}

#ifdef IPV6_SUPPORT
static int socket_addrcompare(const struct sockaddr_in6 * const addr1,
			      const struct sockaddr_in6 * const addr2)
{
   if (addr1->sin6_port == addr2->sin6_port &&
       addr1->sin6_addr == addr2->sin6_addr) {
      return 0;
   }
   return 1;   
}
#else
static int socket_addrcompare(const struct sockaddr_in * const addr1,
			      const struct sockaddr_in * const addr2)
#endif
{
   if (addr1->sin_port == addr2->sin_port &&
       addr1->sin_addr.s_addr == addr2->sin_addr.s_addr) {
      return 0;
   }
   return 1;
}

/* Accept a new connexion */

int socket_servercnxwait(Socket * const socket)
{
   int t;
   int fd;
#ifdef IPV6_SUPPORT
   struct sockaddr_in6 remote;
#else
   struct sockaddr_in remote;
#endif
   socklen_t addrlen = sizeof remote;
      
   if (socket == NULL) {
      return -1;
   }   
   fd = socket->fd;
   if (socket->endpoint != SOCK_STREAM) {
     return fd;
   }
   if ((t = accept(fd, (struct sockaddr *) &remote, &addrlen)) < 0) {

      return -1;
   }
   
   /* Avoid loop-spoofing attacks : if we're asked to reply on a non-user port
    * or the port we're connected to on the same host, we're the victim of
    * fake packets. Drop the connexion. */
   
#ifdef IPV6_SUPPORT
   if (remote.sin6_port < SOCKET_MINUSERPORT ||
       socket_addrcompare(&remote, &socket->addr) == 0) {
      close(t);
      
      return -2;
   }
#else
   if (remote.sin_port < SOCKET_MINUSERPORT ||
       socket_addrcompare(&remote, &socket->addr) == 0) {
      close(t);
      
      return -2;
   }
#endif

   socket_fixopts(socket, t);
   if (socket->optionnodelay != SOCKET_OPTION_DEFAULT) {
      int opt = socket->optionnodelay;
      setsockopt(t, socket_getproto("tcp", IPPROTO_TCP), TCP_NODELAY,
		 (char *) &opt, sizeof opt);
   }
   if (socket->optionkeepalive != SOCKET_OPTION_DEFAULT) {
      int opt = socket->optionkeepalive;
      setsockopt(t, SOL_SOCKET, SO_KEEPALIVE, (char *) &opt, sizeof opt);
   }
   
   return t;
}

/* Wait for an event : new connexion, socket reply, hang up, ... */

SockEvent *socket_serverwaitevent(Sockets * const sockets)
{
   SocketListItem *socketlistpnt;
   Socket *socket = NULL;
   fd_set rfds;
   int selectret;
   int newcnxfd;   
   register int scannedfd;
   
   if (sockets == NULL) {
      return NULL;
   }
   sockets->sockevent.options = SOCKET_OPTION_NONE;
   if (sockets->pendingmessages > 0) {
      rfds = sockets->lastrfds;
      if (sockets->pendingsync == 0) {
	 sockets->pendingsync = SOCKET_PENDINGSYNC;
	 scannedfd = 0;
	 while (scannedfd < sockets->lastfd) {
	    if (FD_ISSET(scannedfd, &rfds)) {
	       sockets->pendingmessages++;
	    }
	    scannedfd++;
	 }
      } else {
	 sockets->pendingsync--;
      }
      if (sockets->pendingmessages > 0) {
	 sockets->pendingmessages--;      
	 goto ugly;
      }
   } else if (sockets->pendingmessages < 0) {
      sockets->pendingmessages = 0;
   }
   rfds = sockets->rfdsbackup;
   if ((selectret = select(sockets->lastfd, &rfds, NULL, NULL, NULL)) <= 0) {
      if (errno == EINTR) {
	 sockets->sockevent.type = SOCKEVENT_INTERRUPTED;
	 return &sockets->sockevent;
      } 
      sockets->sockevent.type = SOCKEVENT_ERROR;
      return &sockets->sockevent;
   }
   if (selectret > 1) {
      if (sockets->pendingmessages < (INT_MAX - selectret)) {
	 sockets->pendingmessages += (selectret - 1);
      } else {
	 sockets->pendingmessages = INT_MAX;
      }
   }
   sockets->lastrfds = rfds;

   ugly:
   socketlistpnt = sockets->socketlist;
   while (socketlistpnt != NULL) {
      socket = socketlistpnt->socket;
      if (FD_ISSET(socket->fd, &rfds)) {
	 FD_CLR(socket->fd, &sockets->lastrfds);
	 break;
      }
      socketlistpnt = socketlistpnt->next;
   }
   if (socketlistpnt != NULL && socket != NULL) {
      sockets->sockevent.socket = socket;
      if ((newcnxfd = socket_servercnxwait(socket)) <= 0) {
	 sockets->sockevent.type = SOCKEVENT_PEERRESET;

	 return &sockets->sockevent;
      }   
      if (newcnxfd >= sockets->lastfd) {
	 if (newcnxfd < SOCKMAX_FD) {
	    sockets->lastfd = newcnxfd + 1;
	 } else {
	    close(newcnxfd);
	    FD_CLR(newcnxfd, &sockets->rfdsbackup);
	    sockets->lastfd = SOCKMAX_FD;
	    sockets->sockevent.fdoverflow = 1;
	    sockets->sockevent.type = SOCKEVENT_TOOMANY;
	    
	    return &sockets->sockevent;
	 }
      }
      sockets->sockevent.fd = newcnxfd;
      FD_SET(newcnxfd, &sockets->rfdsbackup);
      sockets->sockevent.remotecnx[newcnxfd] = 1;
      sockets->fdtosocket[newcnxfd] = socket;
      sockets->sockevent.type = SOCKEVENT_NEWCNX;
      
      return &sockets->sockevent;
   }
   scannedfd = 0;
   while (scannedfd < sockets->lastfd) {
      if (FD_ISSET(scannedfd, &rfds)) {
	 int len;

	 FD_CLR(scannedfd, &sockets->lastrfds);	 
	 sockets->sockevent.socket = sockets->fdtosocket[scannedfd];
	 sockets->sockevent.fd = scannedfd;
	 if ((len = read(scannedfd, sockets->sockevent.line, SOCKLINE_MAX)) < 0 &&
	     errno != EWOULDBLOCK) {
	    sockets->sockevent.type = SOCKEVENT_ERROR;
	    goto hanged;	    
	 } else if (len == 0) {
	    sockets->sockevent.type = SOCKEVENT_HANGUP;
	    hanged:	    	    
	    close(scannedfd);
	    FD_CLR(scannedfd, &sockets->rfdsbackup);
	    if (scannedfd == sockets->lastfd - 1) {
	       sockets->lastfd = scannedfd;
	    }
	    sockets->sockevent.remotecnx[scannedfd] = 0;
	    
	    return &sockets->sockevent;
	 }
	 sockets->sockevent.linelen = (size_t) len;
	 sockets->sockevent.line[len] = 0;
	 sockets->sockevent.type = SOCKEVENT_ANSWER;
	 
	 return &sockets->sockevent;
      }
      scannedfd++;
   }
   sockets->sockevent.type = SOCKEVENT_NONE;
   
   return &sockets->sockevent;
} 

/* Look up a remote host sockaddr info */

#ifdef IPV6_SUPPORT
static struct sockaddr_in6 *socket_getpeer(Socket * const socket,
					   const int fd)
#else
static struct sockaddr_in *socket_getpeer(Socket * const socket,
					  const int fd)
#endif
{
#ifdef IPV6_SUPPORT
   static struct sockaddr_in6 remote;
#else
   static struct sockaddr_in remote;
#endif
   socklen_t len = sizeof remote;
   
   if (!socket_isstream(socket) ||
       getpeername(fd, (struct sockaddr *) &remote, &len) < 0) {
      char tmpbuf[SOCKLINE_MAX + 1];
      
      len = sizeof remote;
      if (recvfrom(fd, tmpbuf, SOCKLINE_MAX, MSG_PEEK,
		   (struct sockaddr *) &remote, &len) < 0) {	   
	 return NULL;
      }
   }
   return &remote;
}

/* Look up a remote host address in dots notation */

const char *socket_getpeeraddr(Socket * const socket, const int fd)
{
#ifdef IPV6_SUPPORT
   struct sockaddr_in6 *remote;
#else
   struct sockaddr_in *remote;
#endif
   
#ifdef IPV6_SUPPORT
   if ((remote = socket_getpeer(fd)) == NULL ||       
       remote->sin6_addr == NULL) {
      return NULL;      
   }
   return inet_ntoa(remote->sin6_addr); /* FIXME */
#else
   if ((remote = socket_getpeer(socket, fd)) == NULL ||       
       remote->sin_addr.s_addr == 0) {
      return NULL;      
   }
   return inet_ntoa(remote->sin_addr);
#endif
}

/* Do the so-called "paranoid" DNS check and return 0 if
 * the host is not lying */

#ifdef IPV6_SUPPORT
static int socket_paranoidcheck(const struct sockaddr_in6 *host,
				const char * const name)
#else
static int socket_paranoidcheck(const struct sockaddr_in *host,
				const char * const name)
#endif
{
   const struct hostent *host2;
   char * const *addrpnt;
   
#ifdef IPV6_SUPPORT
   if ((host2 = gethostbyname2(name)) == NULL) {
      return -1;
   }
#else
   if ((host2 = gethostbyname(name)) == NULL) {
      return -1;
   }     
#endif

   /* SOCKET_CLEAN_DNS_CONFIGURATION supposes that no in-addr record
    * points to a CNAME */
   
#ifdef SOCKET_CLEAN_DNS_CONFIGURATION
   if (strcasecmp(name, host2->h_name) != 0 &&
       strcasecmp(name, "localhost") != 0) {
      return -1;
   }
#endif
   addrpnt = host2->h_addr_list;
   while (*addrpnt != NULL) {
      if (memcmp(*addrpnt, (const char *) &host->sin_addr,
		 sizeof host->sin_addr) == 0) {
	 return 0;
      }
      addrpnt++;
   }
   
   return -1;
}

/* Look up a remote host name. We systematically perform the double DNS
 * check in order to avoid spoofing. Anyway, we don't distinguish between
 * cases and also deal with aliases to avoid wrong alerts on common DNS
 * misconfigurations, moreover that won't increase the spoofing
 * possibility */

const char **socket_getpeername(Socket * const socket, const int fd)
{
   static const char *hostnames[SOCKET_MAXHOSTALIASES + 2];
   
   const char **hostnamespnt = hostnames;
   size_t hostnamescnt = SOCKET_MAXHOSTALIASES;
#ifdef IPV6_SUPPORT
   struct sockaddr_in6 *remote;
#else
   struct sockaddr_in *remote;
#endif
   struct hostent *host;
   char **aliases;

#ifdef IPV6_SUPPORT
   if ((remote = socket_getpeer(socket, fd)) == NULL ||
       remote->sin_6addr == 0) {
      return NULL;
   }
#else
   if ((remote = socket_getpeer(socket, fd)) == NULL ||
       remote->sin_addr.s_addr == 0) {
      return NULL;
   }
#endif
#ifdef IPV6_SUPPORT
   host = gethostbyaddr((char *) &remote->sin6_addr,
			sizeof remote->sin6_addr, AF_INET);
#else
   host = gethostbyaddr((char *) &remote->sin_addr,
			sizeof remote->sin_addr, AF_INET);
#endif
   if (host == NULL) {
      return NULL;
   }
   if (socket_paranoidcheck(remote, host->h_name) == 0) {
      *hostnamespnt++ = host->h_name;
      hostnamescnt--;
      if (hostnamescnt == 0) {
	 goto coolworld;
      }
   }
   aliases = host->h_aliases;
   while (*aliases != NULL) {
      if (socket_paranoidcheck(remote, *aliases) == 0) {
	 *hostnamespnt++ = *aliases;
	 hostnamescnt--;
	 if (hostnamescnt == 0) {
	    goto coolworld;
	 }
      }
      aliases++;
   }
   coolworld:
   *hostnamespnt = NULL;
   
   return hostnames;
}

/* Look up a remote host name. If we can't find it, return its IP address */

const char *socket_getpeernameoraddr(Socket * const socket, const int fd)
{
   const char **results;
   
   if ((results = socket_getpeername(socket, fd)) != NULL) {
      return *results;
   }
   return socket_getpeeraddr(socket, fd);
}

/* Renumber a file descriptor */

int fd_move(const int to, const int from)
{
   if (to == from) {
      return 0;
   }
   if (fd_copy(to, from) == -1) {
      return -1;
   }
   close(from);
   
   return 0;
}

/* Copy a file descriptor */

int fd_copy(const int to, const int from)
{
  if (to == from) {
     return 0;
  }
  if (fcntl(from, F_GETFL, 0) == -1) {
     return -1;
  }
  close(to);
  if (fcntl(from, F_DUPFD, to) == -1) {
     return -1;
  }
   
  return 0;
}

/* Maximum host name as known by the previous functions */

size_t maxhostnamelen(void)
{
   return MAXHOSTNAMELEN;
}

