/*
	vboxcountd rev.0.1 by Andreas H. Mueller 970425.
	Read COPYING for copyright information.
*/

#include <stdio.h>
#include <asm/io.h>
#include <unistd.h>
#include <utmp.h>
#include <dirent.h>
#include <sys/syslog.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>

#include "config.h"

#define PORTSTAT PORT+2
#define LEFT 0
#define RIGHT 6
#define FDELAY 1/FLASHFREQ*100000

#define REV "0.1"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

char pidfile[255], ownname[255], flash=TRUE;
unsigned int port=PORT, sleepdelay;
unsigned int portstat=PORTSTAT;
struct vboxdir {
	char dir[0xff];
	struct vboxdir *next;
};
struct vboxdir *vboxdir1;
char nodefault=FALSE;

#ifdef linux
extern char *basename __P((__const char *__name));
#endif

char getio()
{
	if (ioperm(port, 1, 1) | ioperm(portstat, 1, 1)) {
		syslog(LOG_ERR, "unable to get io-permissions for 0x%.4x/0x%.4x. Possibly not run as root?", port, portstat);
		fprintf(stderr, "%s: Error: unable to get io-permissions for 0x%.4x/0x%.4x.\n%s: Possibly not run as root?\n",
			ownname, port, portstat, ownname);
		exit(5);
	}
	return 0;
}

void cleanup()
{
	struct vboxdir *vboxdir, *vboxdir_tmp;

	outb(RIGHT, portstat);
	outb(192, port);
	unlink(pidfile);
	closelog();
	vboxdir=vboxdir1;
	while(vboxdir!=NULL) {
		vboxdir_tmp=vboxdir;
		vboxdir=vboxdir->next;
		free((void *)vboxdir_tmp);
	}
}

int scan_dir(char *path)
{
	int messages=0;
	DIR *dir;
	struct dirent *d;

	if((dir=opendir(path))==NULL) {
		syslog(LOG_ERR, "Unable to open directory \"%s\".", path);
		exit(6);
	}
	while((d=readdir(dir))!=NULL)
		messages++;
	messages-=2;
	closedir(dir);
	return(messages);
}

void sigtermhandler()
{
	syslog(LOG_INFO, "caught SIGTERM, cleaning up -- goodbye...");
	cleanup();
	exit(0);
}

void siginthandler()
{
	struct vboxdir *vboxdir;
	int cnt=0;

	syslog(LOG_INFO, "status requested via SIGINT:");
	vboxdir=vboxdir1;
	if(nodefault) {
		vboxdir=vboxdir->next;
		syslog(LOG_INFO, "running without default-directory (option \"-n\" was given on the command-line).");
	}
	else
		syslog(LOG_INFO, "the first directory is the default.");
	syslog(LOG_INFO, "directorys:");
	while(vboxdir!=NULL) {
		syslog(LOG_INFO, " \"%s\": %i.", vboxdir->dir, scan_dir(vboxdir->dir));
		vboxdir=vboxdir->next;
		cnt++;
	}
	syslog(LOG_INFO, "got %i directory%s.", cnt, cnt==1 ? "" : "s");
	syslog(LOG_INFO, "updating every %i seconds.", sleepdelay);
	syslog(LOG_INFO, "the flashing of the right point is %s.", flash==TRUE ? "enabled":"disabled");
	syslog(LOG_INFO, "displaying on port 0x%.4x.", port);
}

void disp_sleep(int seconds, char disp)
{
	int x, l;

	outb(LEFT, portstat);
	outb(64, port);
	usleep(10);
	outb(RIGHT, portstat);
	if(flash)
		for(x=0; x!=seconds; x++) {
			for(l=0; l!=9; l++) {
				outb(128|disp, port);
				usleep(20000);
				outb(disp, port);
				usleep(80000);
			}
		}
	else {
		outb(disp, port);
		sleep(seconds);
	}
}

void do_count()
{

	int messages;
	struct vboxdir *vboxdir;
	const unsigned char table[]={63,  6, 91, 79, 102, 109, 125, 7, 127, 111};
	FILE *fd;
	struct sigaction sa;

	openlog(ownname, LOG_PID | LOG_NDELAY, LOG_ERR);
	syslog(LOG_INFO, "restart");
	sa.sa_handler=sigtermhandler;
	sa.sa_mask=SIGTERM;
	sa.sa_flags=SA_ONESHOT;
	if(sigaction(SIGTERM, &sa, NULL)<0)
		syslog(LOG_WARNING, "unable to establish signal-handler for SIGTERM.\n"
			"Trying to continue without them.");
	sa.sa_handler=siginthandler;
	sa.sa_mask=SIGINT;
	sa.sa_flags=SA_RESTART;
	if(sigaction(SIGINT, &sa, NULL)<0)
		syslog(LOG_WARNING, "unable to establish signal-handler for SIGINT.\n"
			"Trying to continue without them.");
	fd=fopen(pidfile, "w");
	if(fd!=NULL) {
		fprintf(fd, "%d\n", getpid());
		fclose(fd);
	}
	for(;;) {
		messages=0;
		vboxdir=vboxdir1;
		if(nodefault)
			vboxdir=vboxdir->next;
		while(vboxdir!=NULL) {
			messages+=scan_dir(vboxdir->dir);
			if(messages>9)
				messages=9;
			vboxdir=vboxdir->next;
		}
		disp_sleep(sleepdelay, table[messages]);
	}
}

void help()
{
	fprintf(stderr, "\n  %s rev."REV" by Andreas H. Mueller\n", ownname);
	fprintf(stderr, "  (cameron@RadioFlora.apc.de) in 1998.\n");
	fprintf(stderr, "\n  Configured on %s,\n  %s by %s\n", C_HOST, C_TIME, C_USER);
	fprintf(stderr, "  for display-port 0x%.4x and an update-delay of %i second%s.\n", port, sleepdelay,
		sleepdelay==1 ? "" : "s");
	fprintf(stderr, "\n  This program is distributed under the terms of the GNU public license.\n\n");
	fprintf(stderr, "  Usage : %s < <-d SPOOLDIR> <-d...> <-p DISPLAY-PORT>\n", ownname);
	fprintf(stderr, "          <-s SLEEPDELAY-SECONDS> <-n> <-f> > | <--help> | <--version>\n\n");
	fprintf(stderr, "\n   %s returns the following error-codes:\n", ownname);
	fprintf(stderr, "\n  1 : Help called.\n");
	fprintf(stderr, "  2 : unknown option / parameter.\n");
	fprintf(stderr, "  3 : PID-file already exist.\n");
	fprintf(stderr, "  4 : version called.\n");
	fprintf(stderr, "  5 : unable to get io-permissions.\n");
	fprintf(stderr, "  6 : unable to detect at least one spool-directory (-d).\n");
	fprintf(stderr, "  7 : unable to fork to daemon. Happens usually only with low memory.\n\n");
}

int main(int argc, char *argv[])
{
        char **x=0;
	char portstr[8];
	int l=1;
	FILE *fd;
	struct vboxdir *vboxdir, *vboxdir_tmp;

	strncpy(portstr, PORTSTR, strlen(PORTSTR));
	sleepdelay=DELAY;
	vboxdir1=(struct vboxdir *)malloc(sizeof(struct vboxdir));
	strncpy(vboxdir1->dir, "/var/spool/vbox/", strlen("/var/spool/vbox/"));
	strncat(vboxdir1->dir, basename(getenv("HOME")), strlen(basename(getenv("HOME"))));
	strncat(vboxdir1->dir, "/incoming", strlen("/incoming"));
	vboxdir1->next=NULL;
	vboxdir_tmp=vboxdir1;
	strncpy(ownname, basename(argv[0]), strlen(basename(argv[0])));
	while(l!=argc) {
		if(argv[l][0]=='-')
			switch(argv[l][1]) {
				case 'd':	if(l<(argc-1)) {
							l++;
							vboxdir=(struct vboxdir *)malloc(sizeof(struct vboxdir));
							strncpy(vboxdir->dir, argv[l], strlen(argv[l]));
							vboxdir->next=NULL;
							vboxdir_tmp->next=vboxdir;
							vboxdir_tmp=vboxdir;
						}
						break;
				case 'p':	if(l<(argc-1)) {
							port=strtoul(argv[++l], x, 16);
							strncpy(portstr, argv[l], strlen(argv[l]));
							portstat=strtoul(argv[l], x, 16)+2;
						}
						break;
				case 'n':	nodefault=TRUE;
						break;
				case 's':	if(l<(argc-1)) {
							sleepdelay=atol(argv[++l]);
							break;
						}
				case 'R':	if(strcmp(argv[l], "-RN")==0) {
							fprintf(stderr, "  The original name of the tool is vboxcountd, the revision is 0.1.\n");
							return(0xff);
						}
				case '-':	if(strcmp(argv[l], "--help")==0) {
							help();
							return(1);
						}
						if(strcmp(argv[l], "--version")==0) {
							printf("  This is %s, revision %s.\n", ownname, REV);
							return(4);
						}
				case 'f':	flash=FALSE;
						break;
				default:	fprintf(stderr, "%s: Unknown option: %s.\n", ownname, argv[l]);
						fprintf(stderr, "%s: Try \"%s --help\" for usage information.\n", ownname, ownname);
						return(2);
			}
		else {
			fprintf(stderr, "%s: Unknown parameter: %s.\n", ownname, argv[l]);
			fprintf(stderr, "%s: Try \"%s --help\" for usage information.\n", ownname, ownname);
			return(2);
		}
		l++;
	}
	strncpy(pidfile, _PATH_VARRUN, strlen(_PATH_VARRUN));
	strncat(pidfile, ownname, strlen(ownname));
	strncat(pidfile, "-", 1);
	strncat(pidfile, portstr, strlen(portstr)-1);
	strncat(pidfile, ".pid", 4);
	fd=fopen(pidfile, "r");
	if(fd!=NULL) {
		fclose(fd);
		fprintf(stderr, "%s: Error: %s exist.\n%s: Error: seems there is running another %s on 0x%.4x.\n",
			ownname, pidfile, ownname, ownname, port);
		return(3);
	}
	fclose(fd);
	getio();
	vboxdir=vboxdir1;
	if(nodefault)
		vboxdir=vboxdir->next;
	while(vboxdir!=NULL) {
		scan_dir(vboxdir->dir);
		vboxdir=vboxdir->next;
	}
	switch(fork()) {
		case 0: /* I am the child */
			getio();
			do_count();
		case -1: /* fork failed */
			fprintf(stderr, "%s: can't fork.\n", argv[0]);
			exit(7);
		default: /* I am the parent */
			return(0);
	}
}

