/*
 * @(#)RGBImageFilter.java	1.8 95/09/08 Jim Graham
 *
 * Copyright (c) 1994 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

package java.awt.image;

import java.awt.image.ImageConsumer;
import java.awt.image.ColorModel;

/**
 * This class provides an easy way to create a ImageFilter which modifies
 * the pixels of an image in the default RGB ColorModel.  It is meant to
 * be used in conjunction with a FilteredImageSource object to produce
 * filtered versions of existing images.  It is an abstract class that
 * provides the calls needed to channel all of the pixel data through a
 * single method which converts pixels one at a time in the default RGB
 * ColorModel regardless of the ColorModel being used by the ImageProducer.
 * The only method which needs to be defined to create a useable image
 * filter is the filterRGB method.  Here is an example of a definition
 * of a filter which swaps the red and blue components of an image:
 * <pre>
 *
 *	class RedBlueSwapFilter extends RGBImageFilter {
 *	    public RedBlueSwapFilter() {
 *		// The filter's operation does not depend on the
 *		// pixel's location, so IndexColorModels can be
 *		// filtered directly.
 *		canFilterIndexColorModel = true;
 *	    }
 *
 *	    public int filterRGB(int x, int y, int rgb) {
 *		return ((rgb & 0xff00ff00)
 *			| ((rgb & 0xff0000) >> 16)
 *			| ((rgb & 0xff) << 16));
 *	    }
 *	}
 *
 * </pre>
 *
 * @see FilteredImageSource
 * @see ImageFilter
 * @see ColorModel#getRGBdefault
 *
 * @version	1.8 09/08/95
 * @author 	Jim Graham
 */
public abstract class RGBImageFilter extends ImageFilter {
    protected ColorModel origmodel;
    protected ColorModel newmodel;

    /**
     * This boolean indicates whether or not it is acceptable to apply
     * the color filtering of the filterRGB method to the color table
     * entries of an IndexColorModel object in lieu of pixel by pixel
     * filtering.  Subclasses should set this variable to true in their
     * constructor if their filterRGB method does not depend on the
     * coordinate of the pixel being filtered.
     * @see #substituteColorModel
     * @see #filterRGB
     * @see IndexColorModel
     */
    protected boolean canFilterIndexColorModel;

    /**
     * If the ColorModel is an IndexColorModel, and the subclass has
     * set the canFilterIndexColorModel flag to true, then we substitute
     * a filtered version of the color model here and whenever we see
     * that original ColorModel object in the setPixels methods, otherwise
     * we override the default ColorModel used by the ImageProducer and
     * specify the default RGB ColorModel instead.
     * @see ImageConsumer
     * @see ColorModel#getRGBdefault
     */
    public void setColorModel(ColorModel model) {
	if (canFilterIndexColorModel && (model instanceof IndexColorModel)) {
	    ColorModel newcm = filterIndexColorModel((IndexColorModel)model);
	    substituteColorModel(model, newcm);
	    consumer.setColorModel(newcm);
	} else {
	    consumer.setColorModel(ColorModel.getRGBdefault());
	}
    }

    /**
     * Register two ColorModel objects for substitution.  If the oldcm
     * is encountered during any of the setPixels methods, then the newcm
     * will be substituted for it and the pixels will be passed through
     * untouched (but with the new ColorModel object).
     * @param oldcm the ColorModel object to be replaced on the fly
     * @param newcm the ColorModel object to replace oldcm on the fly
     */
    public void substituteColorModel(ColorModel oldcm, ColorModel newcm) {
	origmodel = oldcm;
	newmodel = newcm;
    }

    /**
     * Filter an IndexColorModel object by running each entry in its
     * color tables through the filterRGB function that RGBImageFilter
     * subclasses must provide.  Use coordinates of -1 to indicate that
     * a color table entry is being filtered rather than an actual
     * pixel value.
     * @param icm the IndexColorModel object to be filtered
     * @return a new IndexColorModel representing the filtered colors
     */
    public IndexColorModel filterIndexColorModel(IndexColorModel icm) {
	byte r[] = new byte[256];
	byte g[] = new byte[256];
	byte b[] = new byte[256];
	byte a[] = new byte[256];
	icm.getReds(r);
	icm.getGreens(g);
	icm.getBlues(b);
	icm.getAlphas(a);
	int mapsize = icm.getMapSize();
	for (int i = 0; i < mapsize; i++) {
	    int rgb = filterRGB(-1, -1, icm.getRGB(i));
	    a[i] = (byte) (rgb >> 24);
	    r[i] = (byte) (rgb >> 16);
	    g[i] = (byte) (rgb >> 8);
	    b[i] = (byte) (rgb >> 0);
	}
	return new IndexColorModel(icm.getPixelSize(), mapsize, r, g, b, a);
    }

    /**
     * Filter a buffer of pixels in the default RGB ColorModel by passing
     * them one by one through the filterRGB method.
     * @see ColorModel#getRGBdefault
     * @see #filterRGB
     */
    public void filterRGBPixels(int x, int y, int w, int h,
				int pixels[], int off, int scansize) {
	int index = off;
	for (int cy = 0; cy < h; cy++) {
	    for (int cx = 0; cx < w; cx++) {
		pixels[index] = filterRGB(x + cx, y + cy, pixels[index]);
		index++;
	    }
	    index += scansize - w;
	}
	consumer.setPixels(x, y, w, h, ColorModel.getRGBdefault(),
			   pixels, off, scansize);
    }

    /**
     * If the ColorModel object is the same one that has already
     * been converted, then simply pass the pixels through with the
     * converted ColorModel, otherwise convert the buffer of byte
     * pixels to the default RGB ColorModel and pass the converted
     * buffer to the filterRGBPixels method to be converted one by one.
     * @see ColorModel#getRGBdefault
     * @see #filterRGBPixels
     */
    public void setPixels(int x, int y, int w, int h,
			  ColorModel model, byte pixels[], int off,
			  int scansize) {
	if (model == origmodel) {
	    consumer.setPixels(x, y, w, h, newmodel, pixels, off, scansize);
	} else {
	    int filteredpixels[] = new int[w];
	    int index = off;
	    for (int cy = 0; cy < h; cy++) {
		for (int cx = 0; cx < w; cx++) {
		    filteredpixels[cx] = model.getRGB((pixels[index] & 0xff));
		    index++;
		}
		index += scansize - w;
		filterRGBPixels(x, y + cy, w, 1, filteredpixels, 0, w);
	    }
	}
    }

    /**
     * If the ColorModel object is the same one that has already
     * been converted, then simply pass the pixels through with the
     * converted ColorModel, otherwise convert the buffer of integer
     * pixels to the default RGB ColorModel and pass the converted
     * buffer to the filterRGBPixels method to be converted one by one.
     * Convert a buffer of integer pixels to the default RGB ColorModel
     * and pass the converted buffer to the filterRGBPixels method.
     * @see ColorModel#getRGBdefault
     * @see #filterRGBPixels
     */
    public void setPixels(int x, int y, int w, int h,
			  ColorModel model, int pixels[], int off,
			  int scansize) {
	if (model == origmodel) {
	    consumer.setPixels(x, y, w, h, newmodel, pixels, off, scansize);
	} else {
	    int filteredpixels[] = new int[w];
	    int index = off;
	    for (int cy = 0; cy < h; cy++) {
		for (int cx = 0; cx < w; cx++) {
		    filteredpixels[cx] = model.getRGB(pixels[index]);
		    index++;
		}
		index += scansize - w;
		filterRGBPixels(x, y + cy, w, 1, filteredpixels, 0, w);
	    }
	}
    }

    /**
     * Subclasses must specify a method to convert a single input pixel
     * in the default RGB ColorModel to a single output pixel.
     * @see ColorModel#getRGBdefault
     * @see #filterRGBPixels
     */
    public abstract int filterRGB(int x, int y, int rgb);
}
