/* yellowfin-diag.c: EEPROM setup program for the Packet Engines Yellowfin.

   This is a diagnostic and EEPROM setup program for the Ethernet adapters
   based on the Packet Engines "Yellowfin" G-NIC chip, as used on Gigabit
   Ethernet boards.
   (Note: The EEPROM write code is not currently activated.)

   Copyright 1997 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771
*/

static char *version_msg =
"yellowfin-diag.c:v0.03 7/31/97 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: yellowfin-diag [-aEfFsvVw] [-p <IOport>].\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <linux/in.h>
#include <asm/io.h>

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"new-IOaddress",	1, 0, 'P'},	/* New base I/O address. */
	{"new-irq",	1, 0, 'Q'},		/* New interrupt number */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

/* Offsets to the various registers.
   All accesses need not be longword aligned. */
enum yellowfin_offsets {
	TxCtrl=0x00, TxStatus=0x04, TxPtr=0x0C,
	TxIntrSel=0x10, TxBranchSel=0x14, TxWaitSel=0x18,
	RxCtrl=0x40, RxStatus=0x44, RxPtr=0x4C,
	RxIntrSel=0x50, RxBranchSel=0x54, RxWaitSel=0x58,
	EventStatus=0x80, IntrEnb=0x82, IntrClear=0x84, IntrStatus=0x86,
	ChipRev=0x8C, DMACtrl=0x90, Cnfg=0xA0, RxDepth=0xB8, FlowCtrl=0xBC,
	AddrMode=0xD0, StnAddr=0xD2, HashTbl=0xD8,
	EEStatus=0xF0, EECtrl=0xF1, EEAddr=0xF2, EERead=0xF3, EEWrite=0xF4,
	EEFeature=0xF5,
};

#define EEPROM_SIZE 128

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_yellowfin_eeprom[16] = {
  /* Currently invalid! */ }; 

/* Values read from the EEPROM, and the new image. */
unsigned char eeprom_contents[128];
unsigned short new_ee_contents[64];

int do_write_eeprom = 0;
int ioaddr;
int verbose = 1;
int has_mii = 0;
int phy0 = -1;
int phy1 = -1;

static int read_eeprom(int ioaddr, int location);

static void write_eeprom(int ioaddr, int index, int value);
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value);
#ifdef notyet
#endif

static void parse_eeprom(unsigned char *ee_data);

/* The interrupt flags. */
const char *intr_names[16] ={
	"Rx DMA event", "Rx Illegal instruction",
	"PCI bus fault during receive", "PCI parity error during receive", 
	"Tx DMA event", "Tx Illegal instruction",
	"PCI bus fault during transmit", "PCI parity error during transmit", 
	"Early receive", "Carrier sense wakeup",
};
/* Non-interrupting events. */
const char *event_names[16] = {
	"Tx Abort", "Rx frame complete", "Transmit done",
};



int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1, new_irq = -1;
	int errflag = 0, opt_f = 0, show_version = 0;
	int show_eeprom = 0, show_mii = 0;
	int emergency_rewrite = 0;
	int dump_all_regs = 0;
	int c, longind, i, rev_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "aeEfF:i:mp:Q:svVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case 'a': dump_all_regs++;		 break;
		case 'e': show_eeprom++;	 break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++; break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'Q':
			new_irq = atoi(optarg);
			if (new_irq < 3 || new_irq > 15 || new_irq == 6 || new_irq == 8) {
				fprintf(stderr, "Invalid new IRQ %#x.  Valid values: "
						"3-5,7,9-15.\n", new_irq);
				errflag++;
			}
			break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': verbose--;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("yellowfin-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base == 0) {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int bus, device, function, device_id;
			int state = 0;
			if (verbose > 2) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (verbose > 2) printf("--scanning a line %s\n", buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &bus, &device, &function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					if (sscanf(buffer, " Ethernet controller: NCR Unknown"
							   " device (rev %d)", &rev_num) > 0) {
						state = 2;
						continue;
					}
					/* Handle older kernels that don't recognize the card. */
					if (sscanf(buffer, " Vendor id=1000. Device id=%d",
							   &device_id) > 0) {
						if (device_id == 702) {
							state = 2;
						} else
							fprintf(stderr, "Unknown PCI device ID %d.\n",
									device_id);
						continue;
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						printf("Found P-E Yellowfin card at %#x.\n",
							   port_base);
						state = 3;
						break;
					}
				}
			}
		}
	}

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find an Yellowfin card in /proc/pci.\n  If there is"
				" a G-NIC card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}

	ioaddr = port_base;

	/* It's mostly safe to examine the registers and EEPROM during
	   operation.  But warn the user, and make then pass '-f'. */
	if (!opt_f  && (inl(ioaddr + TxStatus) & 0x0400) != 0x0000) {
		printf("A potential Yellowfin chip has been found, but it appears to "
			   "be active.\nEither shutdown the network, or use the"
			   " '-f' flag.\n");
		return 1;
	}

	if (verbose || dump_all_regs) {
	  unsigned intr_status, event_status;

	  printf("Yellowfin chip registers at %#x", ioaddr);
	  for (i = 0; i < 0x100; i += 4) {
		  if ((i & 0x1f) == 0)
			  printf("\n 0x%3.3X:", i);
		  printf(" %8.8x", inl(ioaddr + i));
	  }
	  printf("\n");
	  intr_status = inw(ioaddr + IntrStatus);
	  printf("  %snterrupt sources are pending.\n",
			 (intr_status & 0x03ff) ? "I": "No i");
	  if (intr_status) {
		  for (i = 0; i < 16; i++)
			  if (intr_status & (1<<i))
				  printf("   %s indication.\n", intr_names[i]);
	  }
	  event_status = inw(ioaddr + EventStatus);
	  if (event_status) {
		  for (i = 0; i < 3; i++)
			  if (event_status & (1<<i))
				  printf("   %s event.\n", event_names[i]);
	  }
	}

	/* Read the EEPROM. */
	for (i = 0; i < EEPROM_SIZE; i++)
	  eeprom_contents[i] = read_eeprom(ioaddr, i);

	if (verbose > 1 || show_eeprom) {
	  unsigned short sum = 0;
	  printf("EEPROM contents:");
	  for (i = 0; i < EEPROM_SIZE; i++) {
		printf("%s %2.2x", (i & 7) == 0 ? "\n ":"",
			   eeprom_contents[i]);
		sum += eeprom_contents[i];
	  }
	  printf("\n The EEPROM checksum (should be 0xbaba) is %#2.2x.\n", sum);
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose) {
		parse_eeprom(eeprom_contents);
	}

	return 0;
}

/* Serial EEPROM section.
	Unlike many other boards the Yellowfin handles the EEPROM interactions
	itself, resulting in much simpler code.
	*/

static int read_eeprom(int ioaddr, int location)
{
	int bogus_cnt = 1000;

	outb(location, ioaddr + EEAddr);
	outb(0x30 | ((location >> 8) & 7), ioaddr + EECtrl);
	while (inb(ioaddr + EEStatus) < 0  && --bogus_cnt > 0)
		;
	return inb(ioaddr + EERead);
}

static void write_eeprom(int ioaddr, int index, int value)
{
	int bogus_cnt = 1000;

	outb(0x01, ioaddr + EEFeature); /* Enable EE write */
	outb(value, ioaddr + EEWrite);
	outb(index, ioaddr + EEAddr);
	outb(0x20 | ((index >> 8) & 7), ioaddr + EECtrl);
	while (inb(ioaddr + EEStatus) < 0  && --bogus_cnt > 0)
		;
	outb(0x00, ioaddr + EEFeature); /* Disable EE write */
	return;
  fprintf(stderr, "write_eeprom() not yet implemented.\n");
}

static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value)
{
	if (ee_values[index] != new_value) {
		if (do_write_eeprom) {
			printf("Writing new %s entry 0x%4.4x.\n",
				   field_name, new_value);
			write_eeprom(ioaddr, index, new_value);
		} else
			printf(" Would write new %s entry 0x%4.4x (old value 0x%4.4x).\n",
				   field_name, new_value, ee_values[index]);
		ee_values[index] = new_value;
		return 1;
	}
	return 0;
}


static void parse_eeprom(unsigned char *eeprom)
{
	int i;

	printf("Packet Engines Yellowfin at %#x, station address ", ioaddr);
	for (i = 0; i < 5; i++)
		printf("%2.2X:", eeprom[i]);
	printf("%2.2X.\n", eeprom[i]);

	return;
}


/*
 * Local variables:
 *  compile-command: "cc -N -O -Wall -o yellowfin-diag yellowfin-diag.c"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
