/* vortex-diag.c: Diagnostics/EEPROM setup for the 3Com Vortex series.

   This is a diagnostic and EEPROM setup program for Ethernet adapters
   based on the 3Com Vortex, Boomerang and Cyclone chips, as used on the
   3Com 3c590/595/900/905 PCI EtherLink XL adapters.

   Copyright 1997 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771

   References

   3Com Vortex Engineering Release Specification
   3Com Boomerang modifications (unreleased)
   http://cesdis.gsfc.nasa.gov/linux/misc/NWay.html
   http://www.national.com/pf/DP/DP83840.html
*/

static char *version_msg =
"vortex-diag.c:v1.05 5/22/98 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: vortex-diag [-aEfFsvVw] [-p <IOport>].\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <errno.h>
#include <linux/delay.h>
#include <linux/in.h>
#include <asm/io.h>
#ifdef LIBMII
extern show_mii_details(int ioaddr, int phy_id);
extern monitor_mii(int ioaddr, int phy_id);
#endif
#define udelay(microsecs)	do { int _i = microsecs<<1; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"Advertise", 1, 0, 'A'},
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"show-eeprom",  0, 0, 'e'}, /* Dump EEPROM contents */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"show-mii",  0, 0, 'e'},	/* Dump MII management registers. */
	{"new-hwaddr",  1, 0, 'H'},	/* Set a new hardware address. */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"test",	0, 0, 't'},		/* Do register and SRAM test. */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

/* Constants with useful names (you will still need the datasheet though). */
#ifndef PCI_VENDOR_ID_3COM
#define PCI_VENDOR_ID_3COM			0x10B7
#endif
/* Caution!  These entries must be consistent, with the EISA ones last. */
struct board_map {
	int product_id;
	const char *product_name;
	int flags;
} static card_id_map[] = {
	{ 0x5900, "3c590 Vortex 10Mbps", 0},
	{ 0x5950, 	"3c595 Vortex 100baseTX", 0},
	{ 0x5951,	"3c595 Vortex 100baseT4", 0},
	{ 0x5952,	"3c595 Vortex 100base-MII", 0},
	{ 0x9000,	"3c900 Boomerang 10baseT", 0},
	{ 0x9001,	"3c900 Boomerang 10Mbps/Combo", 0},
	{ 0x9050,	"3c905 Boomerang 100baseTx", 0},
	{ 0x9051,	"3c905 Boomerang 100baseT4", 0},
	{ 0x9055,	"3c905B Cyclone 100baseTx", 0},
#define GENERIC_VORTEX 9
	{ 0,		"3c590-series Vortex", 0},
#define GENERIC_CYCLONE 10
	{ 0,		"3c905-series Boomerang/Cyclone/Hurricane", 0},
	{ 0,		"3c592 EISA 10mbps Demon/Vortex", 0},
	{ 0,		"3c597 EISA Fast Demon/Vortex", 0},
#define UNKNOWN_PRODUCT 13
	{ 0,		"Unknown 3Com product", 0},
};

#define EL3WINDOW(win_num) outw(SelectWindow + (win_num), ioaddr + EL3_CMD)
#define EL3_CMD 0x0e

enum vortex_cmd {
	TotalReset = 0<<11, SelectWindow = 1<<11, StartCoax = 2<<11,
	RxDisable = 3<<11, RxEnable = 4<<11, RxReset = 5<<11,
	UpStall = 6<<11, UpUnstall = (6<<11)+1,
	DownStall = (6<<11)+2, DownUnstall = (6<<11)+3,
	RxDiscard = 8<<11, TxEnable = 9<<11, TxDisable = 10<<11, TxReset = 11<<11,
	FakeIntr = 12<<11, AckIntr = 13<<11, SetIntrEnb = 14<<11,
	SetStatusEnb = 15<<11, SetRxFilter = 16<<11, SetRxThreshold = 17<<11,
	SetTxThreshold = 18<<11, SetTxStart = 19<<11,
	StartDMAUp = 20<<11, StartDMADown = (20<<11)+1, StatsEnable = 21<<11,
	StatsDisable = 22<<11, StopCoax = 23<<11,};

/* Bits in the general status register. */
enum vortex_status {
	IntLatch = 0x0001, AdapterFailure = 0x0002, TxComplete = 0x0004,
	TxAvailable = 0x0008, RxComplete = 0x0010, RxEarly = 0x0020,
	IntReq = 0x0040, StatsFull = 0x0080,
	DMADone = 1<<8, DownComplete = 1<<9, UpComplete = 1<<10,
	DMAInProgress = 1<<11,			/* DMA controller is still busy.*/
	CmdInProgress = 1<<12,			/* EL3_CMD is still busy.*/
};
const char *intr_names[13] ={
	"Interrupt latch", "Adapter Failure", "Tx Complete", "Tx Available",
	"Rx Complete", "Rx Early Notice", "Driver Intr Request",
	"Statistics Full", "DMA Done", "Download Complete", "Upload Complete",
	"DMA in Progress", "Command in Progress",
};

enum Window0 {
	Wn0EepromCmd = 10,		/* Window 0: EEPROM command register. */
	Wn0EepromData = 12,		/* Window 0: EEPROM results register. */
	IntrStatus=0x0E,			/* Valid in all windows. */
};
enum Win0_EEPROM_bits {
	EEPROM_Read = 0x80, EEPROM_WRITE = 0x40, EEPROM_ERASE = 0xC0,
	EEPROM_EWENB = 0x30,		/* Enable erasing/writing for 10 msec. */
	EEPROM_EWDIS = 0x00,		/* Disable EWENB before 10 msec timeout. */
};

enum Window3 {			/* Window 3: MAC/config bits. */
	Wn3_Config=0, Wn3_MAC_Ctrl=6, Wn3_Options=8,
};
enum Window4 {		/* Window 4: Xcvr/media bits. */
	Wn4_FIFODiag = 4, Wn4_NetDiag = 6, Wn4_PhysicalMgmt=8, Wn4_Media = 10,
};

#define EEPROM_SIZE 64

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_vortex_eeprom[EEPROM_SIZE] = {
  /* Currently invalid! */ }; 

/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[EEPROM_SIZE];
unsigned short new_ee_contents[EEPROM_SIZE];

static int verbose = 1, opt_f = 0, debug = 0;
static int show_regs = 0, show_eeprom = 0, show_mii = 0;
static int do_write_eeprom = 0, do_test = 0;
unsigned int opt_a = 0,					/* Show-all-interfaces flag. */
	opt_restart = 0,
	opt_reset = 0,
	opt_watch = 0;
static int nway_advertise = -1;
static int fixed_speed = -1;
static unsigned char new_hwaddr[6], set_hwaddr = 0;

static int do_one_chip(int ioaddr, int part_num);
static int parse_advertise(const char *capabilities);
static int read_eeprom(int ioaddr, int location);
static void write_eeprom(int ioaddr, int index, int value);
static int do_update(int ioaddr, int eesize, 
					 unsigned short *ee_values, unsigned short *old_ee_values);
static void mdio_sync(int ioaddr);
int mdio_read(int ioaddr, int phy_id, int location);
void mdio_write(int ioaddr, int phy_id, int location, int value);

static void parse_eeprom(unsigned short *ee_data);


int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind, rev_num = 0;
	int card_cnt = 0, card_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "#:aA:DeEfF:H:i:mp:RrstvVwW",
							longopts, &longind))
		   != -1)
		switch (c) {
		case '#': card_num = atoi(optarg); break;
		case 'a': show_regs++;		break;
		case 'A': nway_advertise = parse_advertise(optarg); break;
		case 'D': debug++;			break;
		case 'e': show_eeprom++;	break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++;			break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			fixed_speed = parse_advertise(optarg); break;
			break;
		case 'H':
			{
				int hwaddr[6], i;
				if (sscanf(optarg, "%2x:%2x:%2x:%2x:%2x:%2x",
						   hwaddr, hwaddr + 1, hwaddr + 2,
						   hwaddr + 3, hwaddr + 4, hwaddr + 5) == 6) {
					for (i = 0; i < 6; i++)
						new_hwaddr[i] = hwaddr[i];
					set_hwaddr++;
				} else
					errflag++;
				break;
			}
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'r': opt_restart++;	break;
		case 'R': opt_reset++;		break;
		case 'q': verbose--;		 break;
		case 't': do_test--;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case 'W': opt_watch++;		break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose || show_version)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("vortex-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base)
		do_one_chip(port_base, 9);
	else {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int bus, device, function, vendor_id, device_id;
			int device_index = 0;
			int state = 0;
			if (debug) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (debug > 1)
					printf(" Parse state %d line -- %s", state, buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &bus, &device, &function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					if (sscanf(buffer, " Ethernet controller:  3Com Vortex (rev %d)", &rev_num) > 0) {
						device_index = 0;
						state = 2;
						continue;
					}
					if (sscanf(buffer, " Ethernet controller:  3Com 3C%x "
							   "%*[0-9A-Za-z] (rev %d)",
							   &device_id, &rev_num) > 0) {
						state = 2;
						if (device_id == 0x0905)
							device_index = 6;
						else
							device_index = 4;
						device_id <<= 4;
						device_id |= (rev_num & 15);
					}
					/* Handle a /proc/pci that does not recognize the card. */
					if (sscanf(buffer, " Vendor id=%4x. Device id=%4x",
							   &vendor_id, &device_id) > 0) {
						if (debug)
							printf("Found vendor 0x%4.4x device ID 0x%4.4x.\n",
								   vendor_id, device_id);
						if (vendor_id != PCI_VENDOR_ID_3COM)
							continue;
						state = 2;
						if ((device_id & 0xFF00) == 0x5900)
							device_index = GENERIC_VORTEX;
						else if ((device_id & 0xFF00) == 0x9000)
							device_index = GENERIC_CYCLONE;
						else {
							fprintf(stderr, "Unknown 3Com PCI device ID "
									"0x%4.4x.\n", device_id);
							if (opt_f)
								device_index = UNKNOWN_PRODUCT;
							else
								state = 0;
						}
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						card_cnt++;
						printf("Found a 3Com PCI Ethernet 3c%x rev %x at %#x.\n",
							   device_id>>4, device_id & 15, port_base);
						state = 3;
						if (card_num == 0 || card_num == card_cnt)
							do_one_chip(port_base, device_id);
						continue;
					}
				}
			}
		}
	}

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find a 3Com card in /proc/pci.\n  If there is"
				" a card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}
	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' or '-aa' to show device registers,\n"
			   "     '-e' to show EEPROM contents,\n"
			   "  or '-m' or '-mm' to show MII management registers.\n");

	return 0;
}


static int do_one_chip(int ioaddr, int part_num)
{
	int chip_active = 0;
	int saved_window = inw(ioaddr + EL3_CMD) >> 13;
	int card_idx;
	int i;

	/* Look up card ID. */
	if (part_num < 20)
		card_idx = part_num;
	else for (i = 0; card_id_map[i].product_id; i++)
		if (part_num == card_id_map[i].product_id) {
			card_idx = i;
			break;
		}

	/* It's mostly safe to examine the registers and EEPROM during
	   operation.  But warn the user, and make then pass '-f'. */
#ifdef notdef
	if ((inl(ioaddr + EL3_CMD) & 0xE000) == 0xE000)
		chip_active = 1;
#endif

	if (verbose > 1 || show_regs > 1) {
		int j;

		if (!opt_f) {
			printf("The Vortex chip may be active, so FIFO registers"
				   " will not be read.\n"
				   "To see all register values use the '-f' flag.\n");
		} else
			chip_active = 0;		/* Ignore the chip status with -f */

		printf("Initial window %d, registers values by window:\n",
			   saved_window);
		for (j = 0; j < 8; j++) {
			printf("  Window %d:", j);
			outw(SelectWindow + j, ioaddr + EL3_CMD);
			for (i = 0; i < 16; i+=2) {
				if (j == 1 && i < 4 && ! opt_f)
					printf(" FIFO");
				else
					printf(" %4.4x", inw(ioaddr + i));
			}
			printf(".\n");
		}
		printf("Vortex chip registers at %#x", ioaddr);
		for (i = 0x10; i < 0x40; i += 4) {
			if ((i & 0x0f) == 0)
				printf("\n  0x%3.3X:", ioaddr + i);
			if (i >= 0x10 && i <= 0x14 && ! opt_f)
				printf(" **FIFO**");
			else
				printf(" %8.8x", inl(ioaddr + i));
		}
		printf("\n");
	}
	if (verbose > 1 || show_regs) {
		unsigned intr_status = inw(ioaddr + IntrStatus);
		printf(" %snterrupt sources are pending.\n",
			   (intr_status & 0x03ff) ? "I": "No i");
		if (intr_status) {
			for (i = 0; i < 13; i++)
				if (intr_status & (1<<i))
					printf("   %s indication.\n", intr_names[i]);
		}
		{
			const char *medias[] = {"100baseT4", "100baseTx", "100baseFx",
			"10baseT", "10base2", "AUI", "MII", "", "10baseFL"};
			int MediaOptions, MacCtrl;
			EL3WINDOW(3);
			MediaOptions = inw(ioaddr + Wn3_Options);
			printf(" Transceiver/media interfaces available: ");
			for (i = 0; i < 8; i++)
				if (MediaOptions & 1<<i)
					printf(" %s", medias[i]);
			printf("%s.\n", (MediaOptions&0xE010)==0x0010 ? "10baseFL" : "");
			MacCtrl = inw(ioaddr + Wn3_MAC_Ctrl);
			printf(" MAC settings: %s-duplex%s%s.\n",
				   MacCtrl & 0x20 ? "full":"half",
				   MacCtrl & 0x100 ? ", 802.1Q flow control":"",
				   MacCtrl & 0x200 ? ", VLT VLAN enabled":"");
		}
	}
	/* Read the EEPROM. */
	EL3WINDOW(0);
	for (i = 0; i < EEPROM_SIZE; i++)
		eeprom_contents[i] = read_eeprom(ioaddr, i);

	if (set_hwaddr) {
		unsigned short sum = 0;
		memcpy(new_ee_contents, eeprom_contents,
			   EEPROM_SIZE*sizeof(eeprom_contents[0]));
		for (i = 0; i < 3; i++)
			new_ee_contents[i + 10] = (new_hwaddr[i*2]<<8) + new_hwaddr[i*2+1];
		/* Recalculate the checksum: Cyclone only! */
		for (i = 0; i < 0x1A; i++)
			sum ^= new_ee_contents[i];
		new_ee_contents[0x20] = (sum ^ (sum>>8)) & 0xff;
		do_update(ioaddr, EEPROM_SIZE, new_ee_contents, eeprom_contents);
		for (i = 0; i < EEPROM_SIZE; i++)
			eeprom_contents[i] = read_eeprom(ioaddr, i);
	}
	if (verbose > 1 || show_eeprom) {
		unsigned short sum = 0;
		printf("EEPROM contents:");
		for (i = 0; i < EEPROM_SIZE; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
			sum += eeprom_contents[i];
		}
		printf("\n The word-wide EEPROM checksum is %#4.4x.\n", sum);
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose > 1 || show_eeprom) {
		parse_eeprom(eeprom_contents);
	}

	/* Show up to four (not just the on-board) PHYs. */
	if (verbose > 1 || show_mii) {
		int phys[4], phy, phy_idx = 0;
		int mii_reg;
		int saved_media_options;
		phys[0] = 24;			/* Default for most 3Com products. */
		EL3WINDOW(3);
		/* Turn on the MII transceiver for some cards. */
		saved_media_options = inw(ioaddr + Wn3_Options);
		outw((saved_media_options & 0x1ff) | 0x8000, ioaddr + Wn3_Options);

		EL3WINDOW(4);
		mdio_sync(ioaddr);
		for (phy = 0; phy < 32 && phy_idx < 4; phy++) {
			int mii_status = mdio_read(ioaddr, phy, 1);
			if (mii_status != 0xffff  &&  mii_status != 0) {
				phys[phy_idx++] = phy;
				printf(" MII PHY found at address %d, status %4.4x.\n",
					   phy, mii_status);
			}
		}
		if (phy_idx == 0)
			printf(" ***WARNING***: No MII transceivers found!\n");
		for (phy = 0; phy < phy_idx; phy++) {
			printf(" MII PHY %d at #%d transceiver registers:",
				   phy, phys[phy]);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n  " : "",
					   mdio_read(ioaddr, phys[phy], mii_reg));
			printf(".\n");
		}
		if (opt_reset) {
			printf("Resetting the transceiver...\n");
			mdio_write(ioaddr, phys[phy], 0, 0x8000);
		}
		if (phy_idx  &&  nway_advertise > 0) {
			printf(" Setting the media capability advertisement register of "
				   "PHY #%d to 0x%4.4x.\n", phys[0], nway_advertise | 1);
			mdio_write(ioaddr, phys[0], 4, nway_advertise | 1);
		}
		if (opt_restart) {
			printf("Restarting negotiation...\n");
			mdio_write(ioaddr, phys[0], 0, 0x0000);
			mdio_write(ioaddr, phys[0], 0, 0x1200);
		}
		/* To force 100baseTx-HD do  mdio_write(ioaddr, phys[0], 0, 0x2000); */
		if (fixed_speed >= 0) {
			int reg0_val = 0;
			reg0_val |= (fixed_speed & 0x0180) ? 0x2000 : 0;
			reg0_val |= (fixed_speed & 0x0140) ? 0x0100 : 0;
			printf("Setting the speed to \"fixed\", %4.4x.\n", reg0_val);
			mdio_write(ioaddr, phys[0], 0, reg0_val);
		}

#ifdef LIBMII
		if (show_mii > 1)
			show_mii_details(ioaddr, phys[0]);
		if (opt_watch || show_mii > 2)
			monitor_mii(ioaddr, phys[0]);
#endif
		EL3WINDOW(3);
		outw(saved_media_options, ioaddr + Wn3_Options);
	}

	if (do_test) {
		int bogus_cnt = 1000;
		/* Run the built-in-self-test of the SRAM buffer.  This is
		   valid only on the boomerang, but we run it unconditionally. */
		EL3WINDOW(4);
		outw(0x0004, ioaddr + Wn4_FIFODiag);
		while ((inw(ioaddr + Wn4_FIFODiag) & 0x0020) == 0  &&  --bogus_cnt > 0)
			udelay(5);
		if (inw(ioaddr + Wn4_FIFODiag) & 0x0020)
			printf("SRAM buffer test %s\n",
				   inw(ioaddr + Wn4_FIFODiag) & 0x0010 ? "failed!" : "passed");
		else
			printf("INTERNAL FAILURE -- SRAM buffer test did not complete!\n");
	}
	EL3WINDOW(saved_window);
	return 0;
}

static int parse_advertise(const char *capabilities)
{
	const char *mtypes[] = {
		"100baseT4", "100baseTx", "100baseTx-FD", "100baseTx-HD",
		"10baseT", "10baseT-FD", "10baseT-HD", 0,
	};
	int cap_map[] = { 0x0200, 0x0180, 0x0100, 0x0080, 0x0060, 0x0040, 0x0020,};
	int i;
	if (debug)
		fprintf(stderr, "Advertise string is '%s'.\n", capabilities);
	for (i = 0; mtypes[i]; i++)
		if (strcmp(mtypes[i], capabilities) == 0)
			return cap_map[i];
	if ((i = strtol(capabilities, NULL, 16)) <= 0xffff)
		return i;
	fprintf(stderr, "Invalid media advertisement '%s'.\n", capabilities);
	return 0;
}


/* Serial EEPROM section. */
static int read_eeprom(int ioaddr, int location)
{
	int timer;
	outw(EEPROM_Read + location, ioaddr + Wn0EepromCmd);
   	/* Wait for the read to take place, worst-case 162 us. */
	for (timer = 1620; timer >= 0; timer--) {
		if ((inw(ioaddr + Wn0EepromCmd) & 0x8000) == 0)
			break;
	}
	return inw(ioaddr + Wn0EepromData);;
}

static void write_eeprom(int ioaddr, int index, int value)
{
	int timer;
	for (timer = 1620; inw(ioaddr + Wn0EepromCmd) & 0x8000;)
		if (--timer < 0)
			goto error_return;
	outw(EEPROM_EWENB, ioaddr + Wn0EepromCmd);
	for (timer = 400; inw(ioaddr + Wn0EepromCmd) & 0x8000;) {
		if (--timer < 0)
			goto error_return;
	}
	if (debug)
		fprintf(stderr, "EEPROM write enable took %d ticks!\n",
				16000-timer);
	outw(EEPROM_ERASE + index, ioaddr + Wn0EepromCmd);
	for (timer = 16000; inw(ioaddr + Wn0EepromCmd) & 0x8000;)
		if (--timer < 0) {
			fprintf(stderr, "EEPROM failed to erase index %d!\n", index);
			return;
		}
	if (debug)
		fprintf(stderr, "EEPROM erased index %d after %d ticks!\n",
				index, 16000-timer);
	outw(EEPROM_EWENB, ioaddr + Wn0EepromCmd);
	for (timer = 400; inw(ioaddr + Wn0EepromCmd) & 0x8000;) {
		if (--timer < 0)
			goto error_return;
	}
	if (debug)
		fprintf(stderr, "EEPROM write enable took %d ticks!\n",
				16000-timer);
	outw(value, ioaddr + Wn0EepromData);
	outw(EEPROM_WRITE + index, ioaddr + Wn0EepromCmd);
	for (timer = 16000; inw(ioaddr + Wn0EepromCmd) & 0x8000;)
		if (--timer < 0)
			goto error_return;
	if (debug)
		fprintf(stderr, "EEPROM wrote index %d with 0x%4.4x after %d ticks!\n",
				index, value, 16000-timer);

	return;
error_return:
	fprintf(stderr, "Failed to write EEPROM location %d with 0x%4.4x!\n",
			index, value);
}

static int do_update(int ioaddr, int eesize, 
					 unsigned short *ee_values, unsigned short *old_ee_values)
{
	int i;

	EL3WINDOW(0);
	for (i = 0; i < eesize; i++)
		if (ee_values[i] != old_ee_values[i]) {
			if (do_write_eeprom) {
				if (verbose)
					printf("Writing an EEPROM word offset %d value 0x%4.4x.\n",
						   i, ee_values[i]);
				write_eeprom(ioaddr, i, ee_values[i]);
			} else
				printf(" Would write new %d entry 0x%4.4x (old value 0x%4.4x).\n", 
					   i, ee_values[i], old_ee_values[i]);
	}
	outw(EEPROM_EWDIS, ioaddr + Wn0EepromCmd);
	for (i = 400; i > 0 && inw(ioaddr + Wn0EepromCmd) & 0x8000; i--)
		;
	return 0;
}


/* Read and write the MII registers using software-generated serial
   MDIO protocol.  It is just different enough from the EEPROM protocol
   to not share code.  The maxium data clock rate is 2.5 Mhz, which is met
   by the PCI I/O access timing. */
#if 1
#define mdio_delay()	do { int _i = 3; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)
#else
#define mdio_delay()	do {} while (0)
#endif

#define MDIO_SHIFT_CLK	0x01
#define MDIO_DIR_WRITE	0x04
#define MDIO_DATA_WRITE0 (0x00 | MDIO_DIR_WRITE)
#define MDIO_DATA_WRITE1 (0x02 | MDIO_DIR_WRITE)
#define MDIO_DATA_READ	0x02
#define MDIO_ENB_IN		0x00

static int mii_preamble_required = 1;
static void mdio_sync(int ioaddr)
{
	int mdio_addr = ioaddr + Wn4_PhysicalMgmt;
	int i;
	/* Establish sync by sending at least 32 logic ones. */ 
	for (i = 32; i >= 0; i--) {
		outw(MDIO_DATA_WRITE1, mdio_addr);
		mdio_delay();
		outw(MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
}
int mdio_read(int ioaddr, int phy_id, int location)
{
	int i;
	int read_cmd = (0xf6 << 10) | (phy_id << 5) | location;
	unsigned int retval = 0;
	int mdio_addr = ioaddr + Wn4_PhysicalMgmt;

	if (verbose > 2)		/* Debug: 5 */
		printf(" mdio_read(%#x, %d, %d)..", ioaddr, phy_id, location);
	if (mii_preamble_required)
		mdio_sync(ioaddr);
	/* Shift the read command bits out. */
	for (i = 14; i >= 0; i--) {
		int dataval = (read_cmd&(1<<i)) ? MDIO_DATA_WRITE1 : MDIO_DATA_WRITE0;
		if (verbose > 3)		/* Debug: 5 */
			printf("%d", (read_cmd & (1 << i)) ? 1 : 0);

		outw(dataval, mdio_addr);
		mdio_delay();
		outw(dataval | MDIO_SHIFT_CLK, mdio_addr);
		if (verbose > 3) printf(" %x", (inw(mdio_addr) >> 16) & 0x0f);
		mdio_delay();
	}
	if (verbose > 3) printf("-> %x", (inw(mdio_addr) >> 16) & 0x0f);

	if (verbose > 3)		/* Debug: 5 */
		printf(" \n");

	/* Read the two transition, 16 data, and wire-idle bits. */
	for (i = 19; i > 0; i--) {
		outw(MDIO_ENB_IN, mdio_addr);
		mdio_delay();
		retval = (retval << 1) | ((inw(mdio_addr) & MDIO_DATA_READ) ? 1 : 0);
		outw(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
		if (verbose > 3) printf(" %x", (inw(mdio_addr) >> 16) & 0x0f);
	}
	if (verbose > 2)		/* Debug: 5 */
		printf(" %4.4x.\n", retval);

	return retval>>1 & 0xffff;
}

void mdio_write(int ioaddr, int phy_id, int location, int value)
{
	int write_cmd = 0x50020000 | (phy_id << 23) | (location << 18) | value;
	int mdio_addr = ioaddr + Wn4_PhysicalMgmt;
	int i;

	if (verbose > 2)		/* Debug: 5 */
		printf(" mdio_write(%#x, %d, %d, %4.4x)..",
			   ioaddr, phy_id, location, value);
	if (mii_preamble_required)
		mdio_sync(ioaddr);

	/* Shift the command bits out. */
	for (i = 31; i >= 0; i--) {
		int dataval = (write_cmd&(1<<i)) ? MDIO_DATA_WRITE1 : MDIO_DATA_WRITE0;
		outw(dataval, mdio_addr);
		mdio_delay();
		outw(dataval | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	/* Leave the interface idle. */
	for (i = 1; i >= 0; i--) {
		outw(MDIO_ENB_IN, mdio_addr);
		mdio_delay();
		outw(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}

	return;
}


static void parse_eeprom(unsigned short *eeprom)
{
	unsigned char *p = (void *)eeprom;
	int i, sum = 0;

	printf("Parsing the EEPROM of a 3Com Vortex/Boomerang:\n");
	printf(" 3Com Node Address ");
	for (i = 0; i < 5; i++)
		printf("%2.2X:", p[i^1]);
	printf("%2.2X (used as a unique ID only).\n", p[i^1]);
	printf(" OEM Station address %2.2x", p[1 + 20]);
	for (i = 1; i < 6; i++)
		printf(":%2.2X", p[(i^1) + 20]);
	printf(" (used as the ethernet address).\n");
	{
		int builton = eeprom[4];
		printf(" Manufacture date (MM/DD/YY) %d/%d/%d, division %c,"
			   " product %c%c.\n", (builton>>5)&15, builton & 31,
			   builton>>9, p[10], p[12], p[13]);
	}
	printf("Options: %s.\n", eeprom[13] & 0x8000 ? "force full-duplex" : "");
	for (i = 0; i < 0x16; i++)
		sum ^= eeprom[i];
	printf("  Vortex checksum is %2.2x (vs. %2.2x),",
		   (sum ^ (sum>>8)) & 0xff, eeprom[0x17]);
	for ( ; i < 0x1A; i++)
		sum ^= eeprom[i];
	sum ^= sum>>8;
	printf("Cyclone checksum is %2.2x (vs. %2.2x).\n",
		   sum & 0xff, eeprom[0x20]);
	return;
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o vortex-diag vortex-diag.c -DLIBMII libmii.c"
 *  alt-compile-command: "cc -O -Wall -o vortex-diag vortex-diag.c"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
