/* tulip-diag.c: Diagnostic and setup for Digital DC21x4*-based ethercards.

   This is a diagnostic and EEPROM setup program for the Ethernet adapters
   based on the Digital "Tulip" series chips: 21040/21041/21140.
   (Note: The EEPROM setup code is not implemented.)

   Copyright 1996-1998 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771
*/

static char *version_msg =
"tulip-diag.c:v1.03 5/20/98 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: tulip-diag [-aEefFmqrRvVwW] [-p <IOport>] [-A <media>] [-F <media>]\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <errno.h>
#include <asm/io.h>
#ifdef LIBMII
extern show_mii_details(int ioaddr, int phy_id);
extern monitor_mii(int ioaddr, int phy_id);
#endif
/* Grrr, glibc fails the define this.. */
extern iopl(int level);

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"Advertise", 1, 0, 'A'},
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
    {"Reset",		0, 0, 'R'},		/* Reset the transceiver. */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

#define	PCI_VENDOR_ID_DEC		0x1011
#define	PCI_VENDOR_ID_LITEON	0x11AD
#define	PCI_VENDOR_ID_MXIC		0x10d9

/* Offsets to the Command and Status Registers, "CSRs".  All accesses
   must be longword instructions and quadword aligned. */
enum tulip_offsets {
	CSR0=0,    CSR1=0x08, CSR2=0x10, CSR3=0x18, CSR4=0x20, CSR5=0x28,
	CSR6=0x30, CSR7=0x38, CSR8=0x40, CSR9=0x48, CSR10=0x50, CSR11=0x58,
	CSR12=0x60, CSR13=0x68, CSR14=0x70, CSR15=0x78 };

#define EEPROM_SIZE 128

#define	 EEPROM_READ 0x80
#define	 EEPROM_WRITE 0x40
#define	 EEPROM_ERASE 0xC0
#define	 EEPROM_EWENB 0x30		/* Enable erasing/writing for 10 msec. */
#define	 EEPROM_EWDIS 0x00		/* Enable erasing/writing for 10 msec. */

/* EEPROM operation locations. */
enum eeprom_offset {
	PhysAddr01=0, PhysAddr23=1, PhysAddr45=2, ModelID=3,
	EtherLink3ID=7, IFXcvrIO=8, IRQLine=9,
	AltPhysAddr01=10, AltPhysAddr23=11, AltPhysAddr45=12,
	DriverTune=13, Checksum=15};

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_eeprom[16] = {
	0x0020, 0xaf0e, 0x3bc2, 0x9058, 0xbc4e, 0x0036, 0x4441, 0x6d50,
	0x0090, 0xaf00, 0x0020, 0xaf0e, 0x3bc2, 0x1310, 0x0000, 0x343c, }; 
/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[64];
unsigned short new_ee_contents[64];

int do_write_eeprom = 0;
int has_mii = 0;
static int verbose = 1, opt_f = 0, debug = 0;
static int show_regs = 0;
static int show_eeprom = 0, show_mii = 0;
unsigned int opt_a = 0,					/* Show-all-interfaces flag. */
	opt_restart = 0,
	opt_reset = 0,
	opt_watch = 0;
static int nway_advertise = 0;
static int fixed_speed = -1;
static int current_part_num = 0; /* Global, for mdio_{read,write,sync}() */

static int do_one_board(int ioaddr, int part_num);
static int parse_advertise(const char *capabilities);
static int read_eeprom(int ioaddr, int location);
static void write_eeprom(short ioaddr, int index, int value);
static void mdio_sync(int ioaddr);
int mdio_read(int ioaddr, int phy_id, int location);
void mdio_write(int ioaddr, int phy_id, int location, int value);

static unsigned int calculate_checksum1(unsigned short *eeprom_contents);
static unsigned int calculate_checksum2(unsigned short *eeprom_contents);
int do_update(int ioaddr, unsigned short *ee_values,
					 int index, char *field_name, int new_value);

static void parse_eeprom(unsigned char *ee_data, int chip_type);


int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind, part_num = 0;
	int card_cnt = 0, card_num = 0;
	int speed;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "#:aA:DeEfF:mp:srRvVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case '#': card_num = atoi(optarg); break;
		case 'a': show_regs++; opt_a++;	break;
		case 'A': nway_advertise = parse_advertise(optarg); break;
		case 'D': debug++;			break;
		case 'e': show_eeprom++;	 break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++; break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else if ((speed = parse_advertise(optarg)) > 0)
				fixed_speed = speed;
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': if (verbose) verbose--;		 break;
		case 'r': opt_restart++;	break;
		case 'R': opt_reset++;		break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case 'W': opt_watch++;		break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("tulip-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base)
		do_one_board(port_base, 0);
	else {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int pci_bus, pci_device, pci_function, vendor_id, device_id;
			int state = 0;
			if (debug) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (debug) fprintf(stderr, " Parse state %d line -- %s",
								   state, buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &pci_bus, &pci_device, &pci_function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					if (sscanf(buffer, " Ethernet controller: DEC DC%d",
							   &part_num) > 0) {
						state = 2;
						continue;
					}
					/* Handle a /proc/pci that does not recognize the card. */
					if (sscanf(buffer, " Vendor id=%x. Device id=%x",
							   &vendor_id, &device_id) > 0) {
						if (debug)
							printf("Found vendor 0x%4.4x device ID 0x%4.4x.\n",
								   vendor_id, device_id);
						if (vendor_id != PCI_VENDOR_ID_DEC
							&& vendor_id != PCI_VENDOR_ID_LITEON
							&& vendor_id != PCI_VENDOR_ID_MXIC)
							continue;
						printf(" Interface with Vendor ID %#x, Device ID %#x.\n",
							   vendor_id, device_id);
						if (vendor_id == PCI_VENDOR_ID_LITEON) {
							part_num = 82168;
							state = 2;
						} else if (vendor_id == PCI_VENDOR_ID_MXIC) {
							part_num = 98713;
							state = 2;
						} else if (device_id == 0x19) {
							part_num = 21142;
							state = 2;
						} else
							fprintf(stderr, "Unknown DEC PCI device ID %x.\n",
									device_id);
						continue;
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						card_cnt++;
						if (part_num == 82168)
							printf("Chip Index #%d: Found a LiteOn 82c168 "
								   "Tulip clone chip at %#x.\n",
								   card_cnt, port_base);
						else if (part_num == 98713)
							printf("Chip Index #%d: Found a Macronix MX98713 "
								   "Tulip clone chip at %#x.\n",
								   card_cnt, port_base);
						else
							printf("Chip Index #%d: Found a DC%d Tulip card at"
								   " PCI bus %d, device %d I/O %#x.\n",
								   card_cnt, part_num, pci_bus, pci_device, 
								   port_base);
						state = 3;
						if (card_num == 0 || card_num == card_cnt)
							if (port_base)
								do_one_board(port_base, part_num);
							else
								printf( "This chip has not been assigned a "
									   "valid I/O address, and will not "
									   "function.\n"" If you have warm-booted"
									   "from another operating system, a "
									   "complete shut-down and power cycle "
									   "may restore the card to normal "
									   "operation.\n");
					}
				}
			}
		}
	}

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find a Tulip card in /proc/pci.\n  If there is"
				" a Tulip card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}

	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' to show device registers,\n"
			   "     '-e' to show EEPROM contents,\n"
			   "  or '-m' to show MII management registers.\n");

	return 0;
}

static const char *tx_state[8] = {
  "Stopped", "Reading a Tx descriptor", "Waiting for Tx to finish",
  "Loading Tx FIFO", "<invalid Tx state>", "Processing setup information",
  "Idle", "Closing Tx descriptor" };
static const char *rx_state[8] = {
  "Stopped", "Reading a Rx descriptor", "Waiting for Rx to finish",
  "Waiting for packets", "Suspended -- no Rx buffers",
  "Closing Rx descriptor"
  "Unavailable Rx buffer -- Flushing Rx frame",
  "Transferring Rx frame into memory",  };
static const char *bus_error[8] = {
  "Parity Error", "Master Abort", "Target abort", "Unknown 3",
  "Unknown 4", "Unknown 5", "Unknown 6", "Unknown 7"};


static int do_one_board(int ioaddr, int part_num)
{
	char chipname[28];
	int i;
	/* It's mostly safe to use the Tulip EEPROM and MDIO register during
	   operation.  But warn the user, and make then pass '-f'. */
	if (!opt_f  && (inl(ioaddr + CSR6) & 0x2002) != 0) {
		printf("A potential Tulip chip has been found, but it appears to "
			   "be active.\nEither shutdown the network, or use the"
			   " '-f' flag.\n");
		return 1;
	}
	if (part_num == 82168)
		strncpy(chipname, "LinkSys/LiteOn 82c168/169", sizeof(chipname));
	else if (part_num == 98713)
		strncpy(chipname, "MX98713", sizeof(chipname));
	else
		snprintf(chipname, sizeof(chipname), "Digital DS%d", part_num);

	if (show_regs) {
		int csr5 = inl(ioaddr + CSR5);
		int csr6 = inl(ioaddr + CSR6);
		int num_regs = (part_num == 82168 || part_num == 21142) ? 0x100 : 0x80;

		printf("%s Tulip chip registers at %#x:", chipname, ioaddr);
		for (i = 0; i < num_regs; i += 8)
			printf("%s %8.8x", (i % 64) == 0 ? "\n " : "", inl(ioaddr + i));
		printf("\n");
		printf(" The Rx process state is '%s'.\n", 
			   rx_state[(csr5 >> 17) & 7]);
		printf(" The Tx process state is '%s'.\n", 
			   tx_state[(csr5 >> 20) & 7]);
		if (csr5 & 0x2000)
			printf("  PCI bus error!: %s.\n", 
				   bus_error[(csr5 >> 23) & 7]);
		printf("Transmit %s, Receive %s, %s-duplex.\n",
			   csr6 & 0x2000 ? "started" : "stopped",
			   csr6 & 0x0002 ? "started" : "stopped",
			   csr6 & 0x0200 ? "full" : "half");
		if (csr6 & 0x00200000)
			printf(" The transmit unit is set to store-and-forward.\n");
		else {
			const short tx_threshold[2][4] = {{ 72, 96, 128, 160 },
											  {128,256, 512, 1024}};
			printf(" The transmit threshold is %d.\n",
				tx_threshold[(csr6&0x00440000) == 0x00040000][(csr6>>14) & 3]);
		}
		printf(" Port selection is %s%s, %s-duplex.\n",
			   ! (csr6 & 0x00040000) ? "10mpbs-serial" :
			   (csr6 & 0x00800000 ? "100mbps-SYM/PCS" : "MII"),
			   csr6 & 0x01000000 ? " 100baseTx scrambler" : "",
			   csr6 & 0x0200 ? "full" : "half");
	}

	/* Read the EEPROM. */
	if (part_num == 21040) {
		outl(0, ioaddr + CSR9);		/* Reset the pointer with a dummy write. */
		for (i = 0; i < 128; i++) {
			int value, boguscnt = 100000, sum = 0;
			do
				value = inl(ioaddr + CSR9);
			while (value < 0  && --boguscnt > 0);
			((unsigned char *)eeprom_contents)[i] = value;
			if (i < 6)
				sum += value & 0xff;
		}
	} else if (part_num == 82168) {
		for (i = 0; i < EEPROM_SIZE/2; i++) {
			int value, boguscnt = 100000;
			unsigned short sum;
			outl(0x600 | i, ioaddr + 0x98);
			do
				value = inl(ioaddr + CSR9);
			while (value < 0  && --boguscnt > 0);
			((unsigned short *)eeprom_contents)[i] = value;
			sum += value & 0xffff;
		}
	} else
		for (i = 0; i < EEPROM_SIZE/2; i++)
			((unsigned short *)eeprom_contents)[i] = read_eeprom(ioaddr, i);

	if (show_eeprom > 1) {
		printf("EEPROM contents:");
		for (i = 0; i < 64; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
		}
		printf("\n ID CRC %#2.2x (vs. %#2.2x), complete CRC %4.4x.\n",
			   (calculate_checksum1(eeprom_contents) >> 8) & 0xff,
			   eeprom_contents[8] & 0xff,
			   calculate_checksum2(eeprom_contents));
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose  &&  part_num != 21040) {
		parse_eeprom((unsigned char *)eeprom_contents, part_num);
	}

	/* Show up to four (not just the on-board) PHYs. */
	if ((has_mii && verbose) || show_mii) {
		int phys[4], phy, phy_idx = 0;
		current_part_num = part_num; 		/* Hack, set a global */
		mdio_sync(ioaddr);
		for (phy = 0; phy < 32 && phy_idx < 4; phy++) {
			int mii_status = mdio_read(ioaddr, phy, 1);
			if (mii_status != 0xffff  && 
				mii_status != 0x0000) {
				phys[phy_idx++] = phy;
				printf(" MII PHY found at address %d, status 0x%4.4x.\n",
					   phy, mii_status);
			}
		}
		if (phy_idx) {
			if (nway_advertise > 0) {
				printf(" Setting the media capability advertisement register"
					   " of PHY #%d to 0x%4.4x.\n",
					   phys[0], nway_advertise | 1);
				mdio_write(ioaddr, phys[0], 4, nway_advertise | 1);
			}
			if (opt_restart) {
				printf("Restarting negotiation...\n");
				mdio_write(ioaddr, phys[0], 0, 0x0000);
				mdio_write(ioaddr, phys[0], 0, 0x1200);
			}
			/* Force 100baseTx-HD by mdio_write(ioaddr,phys[0], 0, 0x2000); */
			if (fixed_speed >= 0) {
				int reg0_val = 0;
				reg0_val |= (fixed_speed & 0x0180) ? 0x2000 : 0;
				reg0_val |= (fixed_speed & 0x0140) ? 0x0100 : 0;
				printf("Setting the speed to \"fixed\", %4.4x.\n", reg0_val);
				mdio_write(ioaddr, phys[0], 0, reg0_val);
			}
		}
		if (phy_idx == 0)
			printf(" ***WARNING***: No MII transceivers found!\n");
#ifdef LIBMII
		else {
			if (show_mii > 1)
				show_mii_details(ioaddr, phys[0]);
			if (opt_watch || show_mii > 2)
				monitor_mii(ioaddr, phys[0]);
		}
#else
		else for (phy = 0; phy < phy_idx; phy++) {
			int mii_reg;
			printf(" MII PHY #%d transceiver registers:", phys[phy]);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n  " : "",
					   mdio_read(ioaddr, phys[phy], mii_reg));
			printf(".\n");
		}
#endif
	}
	if (part_num == 21142 && show_mii) {
		int csr12 = inl(ioaddr + CSR12);
		const char *nway_state[] =
		{ "Autonegotiation disabled", "Transmit disabled", "Ability detect",
		  "Acknowledge detect", "Complete acknowledge", "Negotiation complete",
		  "Link check", "Invalid state"};
		printf("  Internal autonegotiation state is '%s'.\n",
			   nway_state[(csr12 >> 12) & 7]);
		if (opt_reset) {
			/* Restart NWay. */
			if (nway_advertise > 0) {
				int csr_setting = ((nway_advertise & 0x0380) << 9) |
					((nway_advertise & 0x0020) << 1);
				if (verbose)
					printf("Setting 21142 N-Way advertisement to %4.4x "
						   "(%x).\n", nway_advertise, csr_setting );
				outl(0x000FFBF | csr_setting, ioaddr + CSR14);
				outl(nway_advertise & 0x0040 ? 0x82420200 : 0x82420000,
					 ioaddr + CSR6);
			} else {
				outl(0x0003FFFF, ioaddr + CSR14);
				outl(0x82420200, ioaddr + CSR6);
			}
			outl(0x0000, ioaddr + CSR13);
			outl(0x0001, ioaddr + CSR13);
			outl(0x1000, ioaddr + CSR12); /* Start NWay. */
			printf("  Internal autonegotiation state is now '%s' CSR12 %x.\n",
				   nway_state[(inl(ioaddr + CSR12) >> 12) & 7],
				   inl(ioaddr + CSR12));
			sleep(2);
			printf("  Internal autonegotiation state is now '%s', "
				   "CSR12 %x CSR5 %x.\n",
				   nway_state[(inl(ioaddr + CSR12) >> 12) & 7],
				   inl(ioaddr + CSR12), inl(ioaddr + CSR5));
			/* We must explicitly switch to 100mbps mode. */
			if (((nway_advertise > 0 ? nway_advertise : 0x01e1) &
				 inl(ioaddr + CSR12) >> 16) & 0x0180)
				outl(0x83860200, ioaddr + CSR6);
		}
	}

	return 0;
}

static int parse_advertise(const char *capabilities)
{
	const char *mtypes[] = {
		"100baseT4", "100baseTx", "100baseTx-FD", "100baseTx-HD",
		"10baseT", "10baseT-FD", "10baseT-HD", 0,
	};
	int cap_map[] = { 0x0200, 0x0180, 0x0100, 0x0080, 0x0060, 0x0040, 0x0020,};
	int i;
	if (debug)
		fprintf(stderr, "Advertise string is '%s'.\n", capabilities);
	for (i = 0; mtypes[i]; i++)
		if (strcmp(mtypes[i], capabilities) == 0)
			return cap_map[i];
	if ((i = strtol(capabilities, NULL, 16)) <= 0xffff)
		return i;
	fprintf(stderr, "Invalid media advertisement '%s'.\n", capabilities);
	return 0;
}

/* Reading a serial EEPROM is a "bit" grungy, but we work our way through:->.*/
/* This code is a "nasty" timing loop, but PC compatible machines are
   *supposed* to delay an ISA-compatible period for the SLOW_DOWN_IO macro.  */
#define eeprom_delay()	do { int _i = 3; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)

/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x02	/* EEPROM shift clock. */
#define EE_CS			0x01	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x04	/* EEPROM chip data in. */
#define EE_WRITE_0		0x01
#define EE_WRITE_1		0x05
#define EE_DATA_READ	0x08	/* EEPROM chip data out. */
#define EE_ENB			(0x4800 | EE_CS)

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5 << 6)
#define EE_READ_CMD		(6 << 6)
#define EE_ERASE_CMD	(7 << 6)

static int read_eeprom(int ioaddr, int location)
{
	int i;
	unsigned short retval = 0;
	int ee_addr = ioaddr + CSR9;
	int read_cmd = location | EE_READ_CMD;
	
	outl(EE_ENB & ~EE_CS, ee_addr);
	outl(EE_ENB, ee_addr);
	
	/* Shift the read command bits out. */
	for (i = 10; i >= 0; i--) {
		short dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outl(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		outl(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		outl(EE_ENB | dataval, ee_addr);	/* Finish EEPROM a clock tick. */
		eeprom_delay();
	}
	outl(EE_ENB, ee_addr);
	
	for (i = 16; i > 0; i--) {
		outl(EE_ENB | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
		outl(EE_ENB, ee_addr);
		eeprom_delay();
	}

	/* Terminate the EEPROM access. */
	outl(EE_ENB & ~EE_CS, ee_addr);
	return retval;
}

static void write_eeprom(short ioaddr, int index, int value)
{
	int i;
	int ee_addr = ioaddr + CSR9;
	int cmd = ((index | EE_WRITE_CMD)<< 16) | value;
	
	outl(EE_ENB | EE_SHIFT_CLK, ee_addr);
	
	/* Shift the command bits out. */
	for (i = 26; i >= 0; i--) {
		short dataval = (cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outl(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		outl(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
	}
	outl(EE_ENB, ee_addr);
	
	/* Terminate the EEPROM access. */
	outl(EE_ENB & ~EE_CS, ee_addr);
}

/* Read and write the MII registers using software-generated serial
   MDIO protocol.  It is just different enough from the EEPROM protocol
   to not share code.  The maxium data clock rate is 2.5 Mhz. */
#ifdef _LINUX_DELAY_H
#define mdio_delay()	udelay(1) /* Min. 160, nominal 200ns. */
#else
#define mdio_delay()	__SLOW_DOWN_IO;
#endif

#define MDIO_SHIFT_CLK	0x10000
#define MDIO_DATA_WRITE0 0x00000
#define MDIO_DATA_WRITE1 0x20000
#define MDIO_ENB		0x00000		/* Ignore the 0x02000 databook setting. */
#define MDIO_ENB_IN		0x40000
#define MDIO_DATA_READ	0x80000

/* Syncronize the MII management interface by shifting 32 one bits out. */
static void mdio_sync(int ioaddr)
{
	int mdio_addr = ioaddr + CSR9;
	int i;

	if (current_part_num == 82168)
		return;
	for (i = 32; i >= 0; i--) {
		outl(MDIO_ENB | MDIO_DATA_WRITE1, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	return;
}

int mdio_read(int ioaddr, int phy_id, int location)
{
	int i;
	int read_cmd = (0xf6 << 10) | (phy_id << 5) | location;
	int retval = 0;
	int mdio_addr = ioaddr + CSR9;

	if (verbose > 2)		/* Debug: 5 */
		printf(" mdio_read(%#x, %d, %d)..", ioaddr, phy_id, location);
	if (current_part_num == 82168) {
		int i = 1000;
		outl(0x60020000 + (phy_id<<23) + (location<<18), ioaddr + 0xA0);
		while (--i > 0)
			if ( ! ((retval = inl(ioaddr + 0xA0)) & 0x80000000))
				break;
		if (debug)
			printf("Register at %#x is %#x (%#x).\n", ioaddr,
				   inl(ioaddr + 0xA0), retval & 0xffff);
		return retval & 0xffff;
	}
	/* Establish sync by sending at least 32 logic ones. */ 
	for (i = 32; i >= 0; i--) {
		outl(MDIO_ENB | MDIO_DATA_WRITE1, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	/* Shift the read command bits out. */
	for (i = 17; i >= 0; i--) {
		int dataval = (read_cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;
		if (verbose > 3)		/* Debug: 5 */
			printf("%d", (read_cmd & (1 << i)) ? 1 : 0);

		outl(MDIO_ENB | dataval, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB | dataval | MDIO_SHIFT_CLK, mdio_addr);
		if (verbose > 3) printf(" %x", (inl(mdio_addr) >> 16) & 0x0f);
		mdio_delay();
	}
	if (verbose > 3) printf("-> %x", (inl(mdio_addr) >> 16) & 0x0f);

	/* Read the two transition, 16 data, and wire-idle bits. */
	for (i = 19; i > 0; i--) {
		outl(MDIO_ENB_IN, mdio_addr);
		mdio_delay();
		retval = (retval << 1) | ((inl(mdio_addr) & MDIO_DATA_READ) ? 1 : 0);
		if (verbose > 3) printf(" %x", (inl(mdio_addr) >> 16) & 0x0f);
		outl(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	if (verbose > 3) printf(" == %4.4x.\n", retval);
	return (retval>>1) & 0xffff;
}

void mdio_write(int ioaddr, int phy_id, int location, int value)
{
	int i;
	int cmd = (0x5002 << 16) | (phy_id << 23) | (location<<18) | value;
	int mdio_addr = ioaddr + CSR9;

	/* Establish sync by sending 32 logic ones. */ 
	for (i = 32; i >= 0; i--) {
		outl(MDIO_ENB | MDIO_DATA_WRITE1, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB | MDIO_DATA_WRITE1 | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	/* Shift the command bits out. */
	for (i = 31; i >= 0; i--) {
		int dataval = (cmd & (1 << i)) ? MDIO_DATA_WRITE1 : 0;
		outl(MDIO_ENB | dataval, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB | dataval | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	/* Clear out extra bits. */
	for (i = 2; i > 0; i--) {
		outl(MDIO_ENB_IN, mdio_addr);
		mdio_delay();
		outl(MDIO_ENB_IN | MDIO_SHIFT_CLK, mdio_addr);
		mdio_delay();
	}
	return;
}

/* Calculate the EEPROM checksums. */

#define CRC1_POLYNOMIAL 0x07		/* x^8 + x^2 + x^1 + 1 */
static unsigned int
calculate_checksum1(unsigned short *eeprom)
{
	short crc = -1;
	int i, bit;

	for (i = 0; i <= 8; i++)				/* Note: loc. 18 is the sum. */
		for (bit = 15; bit >= 0; bit--) {
			/* Note: bits ordered as read from EEPROM */
			crc <<= 1;
			if (((eeprom[i]>>bit) ^ (crc >> 8)) & 1)
				crc ^= CRC1_POLYNOMIAL;
		}
	return crc;
}

#define CRC2_POLYNOMIAL 0x04c11db7
static unsigned int
calculate_checksum2(unsigned short *eeprom)
{
	int crc = -1;				/* NB really 32 bits, int32 or __s32.  */
	int i, bit;

	for (i = 0; i < 63; i++)				/* Note: loc. 63 is the CRC. */
		for (bit = 0; bit < 16; bit++) {
			int msb = crc < 0;
			crc <<= 1;
			if (((eeprom[i]>>bit) ^ msb) & 1)
				crc ^= CRC2_POLYNOMIAL;
		}
	return crc;
}

int do_update(int ioaddr, unsigned short *ee_values,
			  int index, char *field_name, int new_value)
{
	if (ee_values[index] != new_value) {
		if (do_write_eeprom) {
			printf("Writing new %s entry 0x%4.4x.\n",
				   field_name, new_value);
			write_eeprom(ioaddr, index, new_value);
		} else
			printf(" Would write new %s entry 0x%4.4x (old value 0x%4.4x).\n",
				   field_name, new_value, ee_values[index]);
		ee_values[index] = new_value;
		return 1;
	}
	return 0;
}


/* Parse and emit the information from the EEPROM table. */

static const char * const medianame[] = {
  "10baseT", "10base2", "AUI", "100baseTx",
  "10baseT-Full Duplex", "100baseTx Full Duplex", "100baseT4", "100baseFx",
  "100baseFx-Full Duplex", "MII 10baseT", "MII 10baseT-Full Duplex", "MII",
  "", "MII 100baseTx", "MII 100baseTx-Full Duplex", "MII 100baseT4",
};

struct mediainfo {
	struct mediainfo *next;
	int info_type;
	struct non_mii { char media, csr12val, bitnum, flags;}  non_mii;
	unsigned char *mii;
};

static void parse_eeprom(unsigned char *ee_data, int chip_type)
{
	unsigned char csr12;
	int i;
	unsigned char *p;
	short media;
	int count;

	if (ee_data[0] == 0xff) {
		printf(" This interface is missing the EEPROM.\n  This is likely the "
			   "non-primary interface on a multiport board.\n");
		return;
	}
	/* Detect an old-style (SA only) EEPROM layout. */
	if (memcmp(ee_data, ee_data + 16, 8) == 0) {
	  /* Should actually do a fix-up based on the vendor half of the station
		 address prefix here.  Or least use that information to report which
		 transceiver will work. */
		printf(" An old-style EEPROM layout was found.\n
 The old-style layout does not contain transceiver control information.\n
 This board may not work, or may work only with a subset of transceiver\n
 options or data rates.\n");
	  return;
	} else if (ee_data[27] == 0) {
		printf(" A simplifed EEPROM data table was found.\n"
			   " The EEPROM does not contain transceiver control information.\n");
		return;
	}
		

	printf("EEPROM transceiver/media description for the DC%d chip.\n",
		   chip_type);
	p = (void *)ee_data + ee_data[27];
	media = *((short *)p)++;
	printf("\nLeaf node at offset %d, default media type %4.4x (%s).\n",
		   ee_data[27], media,
		   media & 0x0800 ? "Autosense" : medianame[media & 15]);
	if (chip_type == 21140 || chip_type == 98713 || chip_type == 82168) {
		csr12 = *p++;
		printf(" CSR12 direction setting bits %2.2x.\n", csr12);
	}
	count = *p++;
	printf(" %d transceiver description blocks:\n", count);

	for (i = 0; i < count; i++) {
		if (chip_type == 21041) {
			printf("  21041 media index %2.2x (%s).\n",
				   p[0] & 0x3f, medianame[p[0] & 15]);
			if (p[0] & 0x40) {
				printf("   CSR Register override settings for this media:"
					   " %2.2x%2.2x %2.2x%2.2x %2.2x%2.2x.\n",
					   p[2], p[1], p[4], p[3], p[6], p[5]);
				p += 7;
			} else
				p += 1;
		} else if (p[0] & 0x80) { /* Extended block */
			int blk_type = p[1];
			if (show_eeprom)
				printf("  Media %s,  block type %d.\n",
					   p[1] & 1  ?  "MII"  :  medianame[p[2] & 15], blk_type);
			switch (blk_type) {
			case 0:
				printf("   AUI or SYM transceiver for %s (media type %d).\n"
					   "    CSR12 control port setting %#2.2x,"
					   " command %#2.2x %#2.2x.\n",
					   medianame[p[2] & 15], p[2], p[3], p[5], p[4]);
				if (p[5] & 0x80) {
					printf("    No media-active status available.\n");
					break;
				}
				printf("    Media detection by looking for a %d on bit %d of"
					   " the CSR12 control port.\n",
					   (p[4] & 0x80) ? 0 : 1, (p[4] >> 1) & 7);
				break;
			case 1:				/* 21140 MII PHY*/
			case 3:				/* 21142 MII PHY */
				printf("   MII interface PHY %d (media type 11).\n", p[2]);
				has_mii++;
				break;
			case 2:				/* 21142 SYM or AUI */
			case 4:
				printf("   %s transceiver for %s (media type %d).\n",
					   blk_type == 2 ? "Serial" : "SYM",
					   medianame[p[2] & 15], p[2]);
				if ( ! show_eeprom)
					break;
				if (p[2] & 0x40)
					printf("    CSR13 %2.2x%2.2x  CSR14 %2.2x%2.2x"
						   "  CSR15 %2.2x%2.2x.\n    GP pin direction "
						   "%2.2x%2.2x  GP pin data %2.2x%2.2x.\n",
						   p[4], p[3], p[6], p[5], p[8], p[7], p[10], p[9],
						   p[12], p[11]);
				else
					printf("    GP pin direction %2.2x%2.2x  "
						   "GP pin data %2.2x%2.2x.\n",
						   p[4], p[3], p[6], p[5]);
				if (blk_type == 4) {
					if (p[8] & 0x80)
						printf("    No media detection indication (command "
							   "%2.2x %2.2x).\n", p[8], p[7]);
					else
						printf("    Media detection by looking for a %d on "
							   "general purpose pin %d.\n",
							   (p[7] & 0x80) ? 0 : 1, (p[7] >> 1) & 7);
				}
				break;
			default:
				printf("   UNKNOW MEDIA DESCRIPTION BLOCK TYPE!.\n");
				break;
			}
			p += (p[0] & 0x3f) + 1;
		} else {				/* "Compact" blocks (aka design screw-up). */
			printf("  21140 Non-MII transceiver with media %d (%s).\n"
				   "   CSR12 control port setting %#2.2x,"
				   " command %#2.2x %#2.2x.\n",
				   p[0], medianame[p[0] & 15], p[1], p[3], p[2]);
			if (p[3] & 0x80) {
				printf("   No media-active status available.\n");
			} else
				printf("   Media detection by looking for a %d on bit %d of"
					   " the CSR12 control port.\n",
					   (p[2] & 0x80) ? 0 : 1, (p[2] >> 1) & 7);
			p += 4;
		}
	}
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o tulip-diag tulip-diag.c"
 *  alt-compile-command: "cc -O -Wall -o tulip-diag tulip-diag.c -DLIBMII libmii.c"
 *  tab-width: 4
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 */
