/* rtl8139-diag.c: Diagnostics/EEPROM setup for RealTek RTL8129/8139 chips.

   This is a diagnostic and EEPROM setup program for Ethernet adapters
   based on the RealTek RTL8129 and RTL8139 chips.

   Copyright 1997-1998 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771

   References

   http://www.realtek.com.tw/cn/cn.html
   http://cesdis.gsfc.nasa.gov/linux/misc/NWay.html
*/

static char *version_msg =
"rtl8139-diag.c:v0.06 2/4/98 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: rtl8139-diag [-aEfFsvVw] [-p <IOport>].\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <linux/delay.h>
#include <linux/in.h>
#include <asm/io.h>
#ifdef LIBMII
extern show_mii_details(int ioaddr, int phy_id);
extern monitor_mii(int ioaddr, int phy_id);
#endif

#define udelay(microsecs)	do { int _i = microsecs<<1; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"show-eeprom",  0, 0, 'e'}, /* Dump EEPROM contents */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"show-mii",  0, 0, 'e'},	/* Dump MII management registers. */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"test",	0, 0, 't'},		/* Do register and SRAM test. */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

/* Constants with useful names (you will still need the datasheet though). */
#ifndef PCI_VENDOR_ID_REALTEK
#define PCI_VENDOR_ID_REALTEK		0x10ec
#endif
#ifndef PCI_DEVICE_ID_REALTEK_8129
#define PCI_DEVICE_ID_REALTEK_8129	0x8129
#define PCI_DEVICE_ID_REALTEK_8139	0x8139
#endif
static const char *product_names[] = {
	"RealTek RTL8129",
	"RealTek RTL8139",
	"Unknown RealTek chip",
};
static int product_ids[] = {
	0x8129, 0x8139, 0, };
enum product_index { RTL8129, RTL8139, RTLUnknown };

const char *intr_names[16] ={
	"Rx Complete", "Rx Error", "Transmit OK", "Transmit Error",
	"Rx Buffer Overflow", "Rx Buffer Underrun", "Rx FIFO Overflow",
	"unknown-0080",
	"unknown-0100", "unknown-0200", "PCS timeout - packet too long",
	"PCI System Error",
};

/* Symbolic offsets to registers. */
enum RTL8129_registers {
	MAC0=0,						/* Ethernet hardware address. */
	MAR0=8,						/* Multicast filter. */
	TxStat0=0x10,				/* Transmit status (Four 32bit registers). */
	TxAddr0=0x20,				/* Tx descriptors (also four 32bit). */
	RxBuf=0x30, RxEarlyCnt=0x34, RxEarlyStatus=0x36,
	ChipCmd=0x37, RxBufPtr=0x38, RxBufAddr=0x3A,
	IntrMask=0x3C, IntrStatus=0x3E,
	TxConfig=0x40, RxConfig=0x44, /* Must enable Tx/Rx before writing. */
	Timer=0x48,					/* A general-purpose counter. */
	RxMissed=0x4C,				/* 24 bits valid, write clears. */
	Cfg9346=0x50, Config0=0x51, Config1=0x52,
	FlashReg=0x54, GPPinData=0x58, GPPinDir=0x59, MII_SMI=0x5A, HltClk=0x5B,
	MultiIntr=0x5C, TxSummary=0x60,
	MII_BMCR=0x62, MII_BMSR=0x64, NWayAdvert=0x66, NWayLPAR=0x68,
	NWayExpansion=0x6A,
};

#define EEPROM_SIZE 64

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_8129_eeprom[EEPROM_SIZE] = {
	0x8129, 0x10ec, 0x8129, 0x10ec, 0x00d8, 0x4020, 0xd000, 0x4f00,
	0x004d, 0xa537, 0x0c14, 0x0000, 0xffff, 0xffff, 0xffff, 0xffff,
}; 

/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[EEPROM_SIZE];
unsigned short new_ee_contents[EEPROM_SIZE];

static int verbose = 0, opt_f = 0, debug = 0;
static int show_regs = 0, show_eeprom = 0, show_mii = 0;
static int do_write_eeprom = 0, do_test = 0;

static int do_one_chip(int ioaddr, int part_num);
static int read_eeprom(int ioaddr, int location);
#ifdef notyet
static void write_eeprom(int ioaddr, int index, int value);
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value);
#endif
static void mdio_sync(int ioaddr);
int mdio_read(int ioaddr, int phy_id, int location);
static void parse_eeprom(unsigned short *ee_data);


int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind, rev_num = 0;
	int card_cnt = 0, card_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "#:adDeEfF:i:mp:stvVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case '#': card_num = atoi(optarg); break;
		case 'a': show_regs++;		break;
		case 'D': case 'd': debug++;		break;
		case 'e': show_eeprom++;	break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++;			break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': verbose--;		 break;
		case 't': do_test--;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose || show_version)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("rtl8139-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base == 0) {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int bus, device, function, vendor_id, device_id;
			int device_index = 0;
			int state = 0;
			if (debug) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (debug) printf(" Parse state %d line -- %s", state, buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &bus, &device, &function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					if (sscanf(buffer, " Ethernet controller:  Realtek %d (rev %d)", &device_id, &rev_num) > 0) {
						vendor_id = PCI_VENDOR_ID_REALTEK;
						state = 2;
						continue;
					}
					/* Handle a /proc/pci that does not recognize the card. */
					if (sscanf(buffer, " Vendor id=%4x. Device id=%4x",
							   &vendor_id, &device_id) > 0) {
						int id;
						if (debug)
							printf("Found vendor 0x%4.4x device ID 0x%4.4x.\n",
								   vendor_id, device_id);
						if (vendor_id != PCI_VENDOR_ID_REALTEK)
							continue;
						for (id = 0; product_ids[id]; id++)
							if (device_id == product_ids[id]) {
								device_index = id;
								state = 2;
								break;
							}
						if (product_ids[id] == 0)
							fprintf(stderr, "Unknown PCI device ID 0x%4.4x.\n",
									device_id);
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						card_cnt++;
						printf("Found a RealTek PCI Ethernet %s card at %#x.\n",
							   product_names[device_index], port_base);
						state = 3;
						if (card_num == 0 || card_num == card_cnt)
							do_one_chip(port_base, device_index);
						continue;
					}
				}
			}
		}
	} else
		do_one_chip(port_base, RTLUnknown);

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find an RealTek card in /proc/pci.\n  If there is"
				" a card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}
	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' to show device registers,\n"
			   "     '-e' to show EEPROM contents,\n"
			   "  or '-m' to show MII management registers.\n");

	return 0;
}


static int do_one_chip(int ioaddr, int part_version)
{
	int chip_active = 0;
	int i;

	/* It's mostly safe to examine the registers and EEPROM during
	   operation.  But warn the user, and make then pass '-f'. */
	if ((inb(ioaddr + ChipCmd) & 0x000C) != 0x0000)
		chip_active = 1;

	if (show_regs) {
		unsigned intr_status;

		if (chip_active && !opt_f) {
			printf("The RealTek chip appears to be active, so some registers"
				   " will not be read.\n"
				   "To see all register values use the '-f' flag.\n");
		} else
			chip_active = 0;		/* Ignore the chip status with -f */

		printf("RealTek chip registers at %#x", ioaddr);
		for (i = 0; i < 0x80; i += 4) {
			if ((i & 0x1f) == 0)
				printf("\n 0x%3.3X:", i);
			printf(" %8.8x", inl(ioaddr + i));
		}
		printf(".\n");

		intr_status = inw(ioaddr + IntrStatus);
		printf("  %snterrupt sources are pending.\n",
			   (intr_status & 0x03ff) ? "I": "No i");
		if (intr_status) {
			for (i = 0; i < 16; i++)
				if (intr_status & (1<<i))
					printf("   %s indication.\n", intr_names[i]);
		}
		{
			unsigned char cfg0 = inb(ioaddr + Config0);
			unsigned char cfg1 = inb(ioaddr + Config1);
			const char *xcvr_mode[] = {
				"MII", "an invalid transceiver", "MII/symbol",
				"4B/5B scambler"};
			printf(" The chip configuration is 0x%2.2x 0x%2.2x, %s mode.\n",
				   cfg0, cfg1,
				   cfg0 & 0x20 ? "10baseT" : xcvr_mode[cfg0>>6]);
		}
	}

	/* Read the EEPROM. */
	for (i = 0; i < EEPROM_SIZE; i++)
		eeprom_contents[i] = read_eeprom(ioaddr, i);

	if (show_eeprom) {
		unsigned short sum = 0;
		printf("EEPROM contents:");
		for (i = 0; i < EEPROM_SIZE; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
			sum += eeprom_contents[i];
		}
		printf("\n The word-wide EEPROM checksum is %#4.4x.\n", sum);
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (show_eeprom) {
		parse_eeprom(eeprom_contents);
	}

	/* Show up to four (not just the on-board) PHYs. */
	if (show_mii  &&  part_version == RTL8129) {
		int phys[4], phy, phy_idx = 0;
		mdio_sync(ioaddr);
		for (phy = 0; phy < 32 && phy_idx < 4; phy++) {
			int mii_status = mdio_read(ioaddr, phy, 0);
			if (mii_status != 0xffff) {
				phys[phy_idx++] = phy;
				printf(" MII PHY found at address %d (%4.4x).\n",
					   phy, mii_status);
			}
		}
		if (phy_idx == 0)
			printf(" ***WARNING***: No MII transceivers found!\n");
#ifdef LIBMII
		else if (show_mii > 1) {
			show_mii_details(ioaddr, phys[0]);
			if (show_mii > 2) monitor_mii(ioaddr, phys[0]);
		}
#else
		else for (phy = 0; phy < phy_idx; phy++) {
			int mii_reg;
			printf(" MII PHY #%d transceiver registers:", phys[phy]);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n  " : "",
					   mdio_read(ioaddr, phys[phy], mii_reg));
			printf(".\n");
		}
#endif
	}
	if (show_mii  &&  part_version == RTL8139) {
		printf(" The RTL8139 does not use a MII transceiver.\n"
			   " It does have internal MII-compatible registers:\n"
			   "   Basic mode control register   0x%4.4x.\n"
			   "   Basic mode status register    0x%4.4x.\n"
			   "   Autonegotiation Advertisement 0x%4.4x.\n"
			   "   Link Partner Ability register 0x%4.4x.\n"
			   "   Autonegotiation expansion     0x%4.4x.\n"
			   "   Disconnects                   0x%4.4x.\n"
			   "   False carrier sense counter   0x%4.4x.\n"
			   "   NWay test register            0x%4.4x.\n"
			   "   Receive frame error count     0x%4.4x.\n",
			   inw(ioaddr + MII_BMSR), inw(ioaddr + MII_BMCR),
			   inw(ioaddr + NWayAdvert), inw(ioaddr + NWayLPAR),
			   inw(ioaddr + NWayExpansion), inw(ioaddr + 0x6C),
			   inw(ioaddr + 0x6E), inw(ioaddr + 0x70), inw(ioaddr + 0x72));
#ifdef LIBMII
		if (show_mii > 1) {
			show_mii_details(ioaddr, -1);
			if (show_mii > 2) monitor_mii(ioaddr, 32);
		}
#endif
	}

	if (do_test) {
		printf("FIFO buffer test not yet available.\n");
	}
	return 0;
}


/* Serial EEPROM section. */

/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x04	/* EEPROM shift clock. */
#define EE_CS			0x08	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x02	/* EEPROM chip data in. */
#define EE_WRITE_0		0x00
#define EE_WRITE_1		0x02
#define EE_DATA_READ	0x01	/* EEPROM chip data out. */
#define EE_ENB			(0x80 | EE_CS)

/* Delay between EEPROM clock transitions.
   The 1.2 code is a "nasty" timing loop, but PC compatible machines are
   *supposed* to delay an ISA-compatible period for the SLOW_DOWN_IO macro.  */
#ifdef _LINUX_DELAY_H
#define eeprom_delay(nanosec)	udelay((nanosec + 999)/1000)
#else
#define eeprom_delay(nanosec)	do { int _i = 3; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)
#endif

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5 << 6)
#define EE_READ_CMD		(6 << 6)
#define EE_ERASE_CMD	(7 << 6)

static int read_eeprom(int ioaddr, int location)
{
	int i;
	unsigned short retval = 0;
	short ee_addr = ioaddr + Cfg9346;
	int read_cmd = location | EE_READ_CMD;

	outb(EE_ENB & ~EE_CS, ee_addr);
	outb(EE_ENB, ee_addr);

	/* Shift the read command bits out. */
	for (i = 10; i >= 0; i--) {
		short dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outb(EE_ENB | dataval, ee_addr);
		eeprom_delay(100);
		outb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(150);
		outb(EE_ENB | dataval, ee_addr);	/* Finish EEPROM a clock tick. */
		eeprom_delay(250);
	}
	outb(EE_ENB, ee_addr);

	for (i = 16; i > 0; i--) {
		outb(EE_ENB | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(100);
		retval = (retval << 1) | ((inb(ee_addr) & EE_DATA_READ) ? 1 : 0);
		outb(EE_ENB, ee_addr);
		eeprom_delay(100);
	}

	/* Terminate the EEPROM access. */
	outb(~EE_CS, ee_addr);
	return retval;
}

#ifdef not_yet
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value)
{
	if (ee_values[index] != new_value) {
		if (do_write_eeprom) {
			printf("Writing new %s entry 0x%4.4x.\n",
				   field_name, new_value);
			write_eeprom(ioaddr, index, new_value);
		} else
			printf(" Would write new %s entry 0x%4.4x (old value 0x%4.4x).\n",
				   field_name, new_value, ee_values[index]);
		ee_values[index] = new_value;
		return 1;
	}
	return 0;
}
#endif /* notyet */


/* MII serial management: mostly bogus for now. */
/* Read and write the MII management registers using software-generated
   serial MDIO protocol.  The maxium data clock rate is 2.5 Mhz. */
#define MDIO_DIR		0x80
#define MDIO_DATA_OUT	0x04
#define MDIO_DATA_IN	0x02
#define MDIO_CLK		0x01
#ifdef _LINUX_DELAY_H
#define mdio_delay()	udelay(1) /* Really 400ns. */
#else
#define mdio_delay()	__SLOW_DOWN_IO;
#endif

/* Syncronize the MII management interface by shifting 32 one bits out. */
static void mdio_sync(int ioaddr)
{
	int i;
	int mdio_addr = ioaddr + MII_SMI;

	for (i = 32; i >= 0; i--) {
		outb(MDIO_DIR | MDIO_DATA_OUT, mdio_addr);
		mdio_delay();
		outb(MDIO_DIR | MDIO_DATA_OUT | MDIO_CLK, mdio_addr);
		mdio_delay();
	}
	return;
}
int mdio_read(int ioaddr, int phy_id, int location)
{
	int i;
	int read_cmd = (0xf6 << 10) | (phy_id << 5) | location;
	int retval = 0;
	int mdio_addr = ioaddr + MII_SMI;

	if (phy_id == 32) {			/* Really a 8139.  Use internal registers. */
		char mii_2_8139_map[8] = {MII_BMCR, MII_BMSR, 0, 0, NWayAdvert,
								  NWayLPAR, NWayExpansion, 0 };
		return location < 8 && mii_2_8139_map[location] ?
			inw(ioaddr + mii_2_8139_map[location]) : 0;
	}
	mdio_sync(ioaddr);
	/* Shift the read command bits out. */
	for (i = 15; i >= 0; i--) {
		int dataval =
		  (read_cmd & (1 << i)) ? MDIO_DATA_OUT : 0;

		if (verbose > 3)		/* Debug: 5 */
			printf("%d", (read_cmd & (1 << i)) ? 1 : 0);

		outb(MDIO_DIR | dataval, mdio_addr);
		mdio_delay();
		outb(MDIO_DIR | dataval | MDIO_CLK, mdio_addr);
		if (verbose > 4) printf(" %x ", (inb(mdio_addr) & 0x0f));
		mdio_delay();
	}
	if (verbose > 3) printf("-> %x", inb(mdio_addr) & 0x0f);

	/* Read the two transition, 16 data, and wire-idle bits. */
	for (i = 19; i > 0; i--) {
		outb(0, mdio_addr);
		mdio_delay();
		retval = (retval << 1) | ((inb(mdio_addr) & MDIO_DATA_IN) ? 1 : 0);
		outb(MDIO_CLK, mdio_addr);
		mdio_delay();
		if (verbose > 3) printf("%x", inb(mdio_addr) & 0x0f);
	}
	if (verbose > 2)
		printf("  MII read of %d:%d -> %4.4x.\n", phy_id, location, retval);
	return (retval>>1) & 0xffff;
}

static void parse_eeprom(unsigned short *eeprom)
{
	unsigned char *p = (void *)eeprom;
	int i, sum = 0;

	printf("Parsing the EEPROM of a RealTek chip:\n"
		   "  PCI IDs -- Vendor %#4.4x, Device %#4.4x, Subsystem %#4.4x.\n"
		   "  PCI timer settings -- minimum grant %d, maximum latency %d.\n"
		   "  General purpose pins --  direction 0x%2.2x  value 0x%2.2x.\n"
		   "  Station Address ",
		   eeprom[1], eeprom[2], eeprom[3], p[10], p[11], p[13], p[12] );
	for (i = 14; i < 19; i++)
		printf("%2.2X:", p[i]);
	printf("%2.2X.\n"
		   "  Configuration register 0/1 -- 0x%2.2x / 0x%2.2x.\n",
		   p[i], p[21], p[22]);
	for (i = 0; i < 24; i++)
		sum += p[i];
	printf(" EEPROM active region checksum is %4.4x.\n", sum);
	return;
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o rtl8139-diag rtl8139-diag.c"
 *  alt-compile-command: "cc -O -Wall -o rtl8139-diag rtl8139-diag.c -DLIBMII libmii.o"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
