/* ne2k-pci-diag.c: Diagnostics/EEPROM setup for PCI NE2000 clones.

   This is a diagnostic and EEPROM setup program for PCI NE2000 Ethernet
   adapters based on the RealTek and WinBond chips.

   Copyright 1998 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771

   References
   RealTek RTL8029AS data sheet, available from www.RealTek.com.tw.
*/

static char *version_msg =
"ne2k-pci-diag.c:v1.01 1/27/97 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: ne2k-pci-diag [-aEfFsvVw].\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <errno.h>
#include <linux/in.h>
#include <asm/io.h>
/* Grrr, glibc fails the define this.. */
extern iopl(int level);

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"show-eeprom",  0, 0, 'e'}, /* Dump EEPROM contents */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"test",	0, 0, 't'},		/* Do register and SRAM test. */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

static struct {
	unsigned short vendor, dev_id;
	char *name, *p_name;		/* /proc/pci name as well. */
} pci_clone_list[] = {
	{0xffff, 0x2000, "Generic PCI NE2000 clone"},
	{0x10ec, 0x8029, "RealTek RTL8029", "Realtek 8029"},
	{0x1050, 0x0940, "Winbond 89C940"},
	{0x11f6, 0x1401, "Compex RL2000"},
	{0x8e2e, 0x3000, "KTI ET32P2"},
	{0x4a14, 0x5000, "NetVin NV5000"},
	{0x1106, 0x0926, "Via 82C926"},
	{0,}
};

/* Constants with useful names (you will still need the datasheet though). */
#define EEPROM_SIZE 64

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_rtl8029_eeprom[EEPROM_SIZE] = {
  /* Currently invalid! */ }; 

/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[EEPROM_SIZE];
unsigned short new_ee_contents[EEPROM_SIZE];

/* Values from 8390.h, here to avoid version skew and bogus email. */
#define NS_CMD	 (dev->base_addr)
#define NS_BASE	 (dev->base_addr)
#define NS_DATAPORT		0x10	/* NatSemi-defined port window offset. */
#define NE_DATAPORT		0x10	/* NatSemi-defined port window offset. */
#define NS_RESET		0x1f	/* Issue a read to reset, a write to clear. */

#define NE1SM_START_PG	0x20	/* First page of TX buffer */
#define NE1SM_STOP_PG	0x40	/* Last page +1 of RX ring */
#define NESM_START_PG	0x40	/* First page of TX buffer */
#define NESM_STOP_PG	0x80	/* Last page +1 of RX ring */

#define E8390_CMD		0x00	/* The command register (for all pages) */
#define E8390_STOP		0x01	/* Stop and reset the chip */
#define E8390_START		0x02	/* Start the chip, clear reset */
#define E8390_RREAD		0x08	/* Remote read */
#define E8390_NODMA		0x20	/* Remote DMA */
#define E8390_PAGE0		0x00	/* Select page chip registers */
#define E8390_PAGE1		0x40	/* using the two high-order bits */
#define E8390_PAGE2		0x80	/* Page 2 is has only a few defined. */
#define E8390_PAGE3		0xC0	/* Page 3 is propriatary data. */

#define E8390_RXOFF 0x20		/* EN0_RXCR: Accept no packets */
#define E8390_TXOFF 0x02		/* EN0_TXCR: Transmitter off */

/* Page 0 register offsets. */
#define EN0_CLDALO		0x01	/* Low byte of current local dma addr  RD */
#define EN0_STARTPG		0x01	/* Starting page of ring bfr WR */
#define EN0_CLDAHI		0x02	/* High byte of current local dma addr	RD */
#define EN0_STOPPG		0x02	/* Ending page +1 of ring bfr WR */
#define EN0_BOUNDARY	0x03	/* Boundary page of ring bfr RD WR */
#define EN0_TSR			0x04	/* Transmit status reg RD */
#define EN0_TPSR		0x04	/* Transmit starting page WR */
#define EN0_NCR			0x05	/* Number of collision reg RD */
#define EN0_TCNTLO		0x05	/* Low	byte of tx byte count WR */
#define EN0_FIFO		0x06	/* FIFO RD */
#define EN0_TCNTHI		0x06	/* High byte of tx byte count WR */
#define EN0_ISR			0x07	/* Interrupt status reg RD WR */
#define EN0_CRDALO		0x08	/* low byte of current remote dma address RD */
#define EN0_RSARLO		0x08	/* Remote start address reg 0 */
#define EN0_CRDAHI		0x09	/* high byte, current remote dma address RD */
#define EN0_RSARHI		0x09	/* Remote start address reg 1 */
#define EN0_RCNTLO		0x0a	/* Remote byte count reg WR */
#define EN0_RCNTHI		0x0b	/* Remote byte count reg WR */
#define EN0_RSR			0x0c	/* rx status reg RD */
#define EN0_RXCR		0x0c	/* RX configuration reg WR */
#define EN0_TXCR		0x0d	/* TX configuration reg WR */
#define EN0_COUNTER0	0x0d	/* Rcv alignment error counter RD */
#define EN0_DCFG		0x0e	/* Data configuration reg WR */
#define EN0_COUNTER1	0x0e	/* Rcv CRC error counter RD */
#define EN0_IMR			0x0f	/* Interrupt mask reg WR */
#define EN0_COUNTER2	0x0f	/* Rcv missed frame error counter RD */


static int verbose = 0, opt_f = 0, debug = 0;
static int show_regs = 0, show_eeprom = 0, show_mii = 0;
static int do_write_eeprom = 0, do_test = 0;

static int do_one_chip(int ioaddr, int part_num);
static int read_eeprom(int ioaddr, int location);
#ifdef notyet
static void write_eeprom(int ioaddr, int index, int value);
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value);
#endif

static void parse_eeprom(unsigned short *ee_data);


int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind;
	int card_cnt = 0, card_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "#:aA:DeEfF:i:mp:stvVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case '#': card_num = atoi(optarg); break;
		case 'a': show_regs++;		break;
		case 'D': debug++;			break;
		case 'e': show_eeprom++;	break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++;			break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': verbose--;		 break;
		case 't': do_test++;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose || show_version)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("ne2k-pci-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base == 0) {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int bus, device, function, vendor_id, device_id;
			int device_index = 0;
			int state = 0;
			if (debug) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (debug) printf(" Parse state %d line -- %s", state, buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &bus, &device, &function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					char pci_name[80];
					int i;
					if (sscanf(buffer, " Ethernet controller: %[^\n]",
							   pci_name) > 0) {
						for (i = 0; pci_clone_list[i].name; i++) {
							char *cmp_name = pci_clone_list[i].p_name ?
								pci_clone_list[i].p_name :
								pci_clone_list[i].name;
							if (debug)
								printf("Comparing %s to %s.\n",
									   cmp_name, pci_name);
							if (strncmp(pci_name, cmp_name,
										strlen(cmp_name)-1)== 0){
								device_index = i;
								state = 2;
								break;
							}
						}
						if (state != 2)
							continue;
					}
					/* Handle a /proc/pci that does not recognize the card. */
					if (sscanf(buffer, " Vendor id=%4x. Device id=%4x",
							   &vendor_id, &device_id) > 0) {
						if (debug)
							printf("Found vendor 0x%4.4x device ID 0x%4.4x.\n",
								   vendor_id, device_id);
						for (i = 0; pci_clone_list[i].name; i++) {
							if (vendor_id == pci_clone_list[i].vendor
								&& device_id == pci_clone_list[i].dev_id) {
								device_index = i;
								state = 2;
							}
						}
						if (state != 2)
							continue;
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						card_cnt++;
						printf("Found a PCI NE2000 clone %s at %#x.\n",
							   pci_clone_list[device_index].name, port_base);
						state = 3;
						if (card_num == 0 || card_num == card_cnt)
							do_one_chip(port_base, device_index);
						continue;
					}
				}
			}
		}
	} else
		do_one_chip(port_base, 0);

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find an NE2000 clone in /proc/pci.\n  If there is"
				" a card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}
	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' to show device registers,\n"
			   "     '-e' to show EEPROM contents,\n"
			   "  or '-m' to show MII management registers.\n");

	return 0;
}


static int do_one_chip(int ioaddr, int part_version)
{
	int chip_active = 0;
	int saved_window = inb(E8390_CMD);
	int i;

	/* It's mostly safe to examine the registers and EEPROM during
	   operation.  But warn the user, and make then pass '-f'. */
	if ((inb(ioaddr + E8390_CMD) & 0xC0) == 0x00)
		chip_active = 1;

	if (verbose || show_regs) {
		unsigned intr_status;
		int j;

		if (chip_active && ! opt_f) {
			printf("The 8390 core appears to be active, so some registers"
				   " will not be read.\n"
				   "To see all register values use the '-f' flag.\n");
		}

		printf("Initial window %d, registers values by window:\n",
			   saved_window >> 6);
		for (j = 0; j < 0x100; j+=0x40) {
			printf("  Window %d:", j>>6);
			outw(j, ioaddr + E8390_CMD);
			for (i = 0; i < 16; i++) {
				printf(" %2.2x", inb(ioaddr + i));
			}
			printf(".\n");
		}
		outb(0, ioaddr + E8390_CMD);
		intr_status = inb(ioaddr + EN0_ISR);
		printf("\n %snterrupt sources are pending.\n",
			   (intr_status & 0xff) ? "I": "No i");
		if (intr_status) {
			const char *intr_names[] = {
				"Rx packet", "Tx done", "Rx error", "Tx error", "Rx overflow",
				"Counter full", "Transfer done", "Resetting" };
			for (i = 0; i < 13; i++)
				if (intr_status & (1<<i))
					printf("   %s indication.\n", intr_names[i]);
		}
	}

	/* Read the EEPROM. */
	outb(E8390_PAGE3, ioaddr + E8390_CMD);
	for (i = 0; i < EEPROM_SIZE; i++)
		eeprom_contents[i] = read_eeprom(ioaddr, i);

	if (show_eeprom) {
		unsigned short sum = 0;
		printf("EEPROM contents:");
		for (i = 0; i < EEPROM_SIZE; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
			sum += eeprom_contents[i];
		}
		printf("\n The word-wide EEPROM checksum is %#4.4x.\n", sum);
	}
	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose || show_eeprom) {
		parse_eeprom(eeprom_contents);
	}

	if (do_test) {
		/* Run a loopback test. */
	}
	outb(saved_window, ioaddr + E8390_CMD);
	return 0;
}


/* Serial EEPROM section. */

/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x04	/* EEPROM shift clock. */
#define EE_CS			0x08	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x02	/* EEPROM chip data in. */
#define EE_WRITE_0		0x00
#define EE_WRITE_1		0x02
#define EE_DATA_READ	0x01	/* EEPROM chip data out. */
#define EE_ENB			(0x80 | EE_CS)

#define WN0_EECTL	1

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5 << 6)
#define EE_READ_CMD		(6 << 6)
#define EE_ERASE_CMD	(7 << 6)

static int read_eeprom(int ioaddr, int location)
{
	int i;
	unsigned retval = 0;
	int ee_addr = ioaddr + WN0_EECTL;
	int read_cmd = location | EE_READ_CMD;

	outb(EE_ENB & ~EE_CS, ee_addr);
	outb(EE_ENB, ee_addr);

	if (debug)
		printf(".\n read_eeprom(%#x, %d): register is %#2.2x, cmd is %x: ",
			   ioaddr, location, inb(ee_addr), read_cmd);
	
	/* Shift the read command bits out. */
	for (i = 10; i >= 0; i--) {
		int dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outb(EE_ENB | dataval, ee_addr);
		outb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		inb(ee_addr);			/* Delay a PCI cycle. */
	}
	outb(EE_ENB, ee_addr);

	for (i = 16; i > 0; i--) {
		outb(EE_ENB | EE_SHIFT_CLK, ee_addr);
		retval = (retval << 1) | ((inb(ee_addr) & EE_DATA_READ) ? 1 : 0);
		outb(EE_ENB, ee_addr);
	}

	/* Terminate the EEPROM access. */
	outb(EE_ENB & ~EE_CS, ee_addr);
	outb(0, ee_addr);
	return retval;
}

static void write_eeprom(int ioaddr, int location, int value)
{
	int ee_addr = ioaddr + WN0_EECTL;
	int write_cmd = location | EE_WRITE_CMD;
	int i;

	write_cmd <<= 16;
	write_cmd |= value;

	outb(EE_ENB & ~EE_CS, ee_addr);
	outb(EE_ENB, ee_addr);

	/* Shift the command bits out. */
	for (i = 10; i >= 0; i--) {
		int dataval = (write_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outb(EE_ENB | dataval, ee_addr);
		outb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		inb(ee_addr);
	}
}

#ifdef notdef
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value)
{
	if (ee_values[index] != new_value) {
		if (do_write_eeprom) {
			printf("Writing new %s entry 0x%4.4x.\n",
				   field_name, new_value);
			write_eeprom(ioaddr, index, new_value);
		} else
			printf(" Would write new %s entry 0x%4.4x (old value 0x%4.4x).\n",
				   field_name, new_value, ee_values[index]);
		ee_values[index] = new_value;
		return 1;
	}
	return 0;
}
#endif

static void parse_eeprom(unsigned short *eeprom)
{
	unsigned char *p = (void *)eeprom;
	int i;

	printf("Parsing the EEPROM of a RTL8029:\n");
	printf(" Station Address ");
	for (i = 2; i < 7; i++)
		printf("%2.2X:", p[i]);
	printf("%2.2X (used as the ethernet address).\n", p[i]);
	return;
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o ne2k-pci-diag ne2k-pci-diag.c"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
