/* Mode: C;
 * mii_look.c: Examine the MII registers of a network interfaces.
 *
 * Usage:	mii_look [-v] interface.
 *
 * Author:	Donald Becker <becker@cesdis.gsfc.nasa.gov>
 *		Copyright 1997 Donald Becker
 *
 *		This program is free software; you can redistribute it
 *		and/or modify it under the terms of the GNU General Public
 *		License as published by the Free Software Foundation.
 *
 *	The author may be reached as becker@CESDIS.gsfc.nasa.gov, or C/O
 *	Center of Excellence in Space Data and Information Sciences
 *	   Code 930.5, Goddard Space Flight Center, Greenbelt MD 20771
 */

static char *version =
"mii_look.c:v0.01 9/6/97  Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static const char *usage_msg =
"Usage: mii_look [-afvV] <interface>.\n";

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/if_ether.h>

typedef unsigned short u16;

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
    {"all-interfaces", 0, 0, 'a'},	/* Show all interfaces. */
    {"force",       0, 0, 'f'},		/* Force the operation. */
    {"help", 		0, 0, '?'},		/* Give help */
    {"verbose", 	0, 0, 'v'},		/* Report each action taken.  */
    {"version", 	0, 0, 'V'},		/* Emit version information.  */
    { 0, 0, 0, 0 }
};

/* Command-line flags. */
unsigned int
opt_a = 0,					/* Show-all-interfaces flag. */
opt_f = 0,					/* Force the operation. */
verbose = 0,					/* Verbose flag. */
opt_version;
int skfd = -1;					/* AF_INET socket for ioctl() calls.	*/

int
main(int argc, char **argv)
{
	struct ifreq  ifr;
	int c, errflag = 0;
	char **spp, *ifname;

	while ((c = getopt_long(argc, argv, "afvV?", longopts, 0)) != EOF)
		switch (c) {
		case 'a': opt_a++; break;
		case 'f': opt_f++; break;
		case 'v': verbose++;		break;
		case 'V': opt_version++;	break;
		case '?': errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 2;
	}

	if (verbose || opt_version)
		printf(version);

	/* Open a basic socket. */
	if ((skfd = socket(AF_INET, SOCK_DGRAM,0)) < 0) {
		perror("socket");
		exit(-1);
	}

	if (verbose > 1)
		fprintf(stderr, "DEBUG: argc=%d, optind=%d and argv[optind] is %s.\n",
				argc, optind, argv[optind]);

	/* No remaining args means show all interfaces. */
	if (optind == argc) {
		/* if_print((char *)NULL); */
		(void) close(skfd);
		exit(0);
	}

	/* Copy the interface name. */
	spp = argv + optind;
	ifname = *spp++;

	/* A single arg means show the configuration for this interface. */
	if (ifname == NULL) {
		/* if_print(ifname); */
		(void) close(skfd);
		exit(0);
	}

	/* Get the vitals from the interface. */
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
	if (ioctl(skfd, SIOCDEVPRIVATE, &ifr) < 0) {
		fprintf(stderr, "SIOCGPHYNUM on %s failed: %s\n", ifname,
				strerror(errno));
		(void) close(skfd);
		return 1;
	} else {
		u16 *data = (u16 *)(&ifr.ifr_data);
		int i;

		printf("MII PHY in use is %d.\n", data[0]);
		for (i = 0; i < 32; i++) {
			data[1] = i;
			if (ioctl(skfd, SIOCDEVPRIVATE+1, &ifr) < 0) {
				fprintf(stderr, "SIOCGMIIREG on %s failed: %s\n", ifname,
						strerror(errno));
			} else
				printf(" %4.4x", data[3]);
		}
	}

	(void) close(skfd);
	return 0;
}


/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  compile-command: "gcc -Wall -Wstrict-prototypes -O mii_look.c -o mii_look"
 * End:
 */
