/* Mode: C;
 * mii-diag.c: Examine and set the MII registers of a network interfaces.

	Usage:	mii-diag [-v] interface.

	Notes:
	The compile-command is at the end of this source file.
	This program only works with drivers that implement MII ioctl() calls.

	Written/copyright 1997-1998 by Donald Becker <becker@cesdis.gsfc.nasa.gov>

	This program is free software; you can redistribute it
	and/or modify it under the terms of the GNU General Public
	License as published by the Free Software Foundation.

	The author may be reached as becker@CESDIS.gsfc.nasa.gov, or C/O
	Center of Excellence in Space Data and Information Sciences
	Code 930.5, Goddard Space Flight Center, Greenbelt MD 20771

	References
	http://cesdis.gsfc.nasa.gov/linux/misc/NWay.html
	http://www.national.com/pf/DP/DP83840.html
*/

static char version[] =
"mii-diag.c:v1.02 3/3/98  Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";

static const char usage_msg[] =
"Usage: mii-diag [-afvV] <interface>.\n";

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#ifdef use_linux_libc5
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#endif

typedef unsigned short u16;

#define SIOCGMIIPHY (SIOCDEVPRIVATE)		/* Get the PHY in use. */
#define SIOCGMIIREG (SIOCDEVPRIVATE+1) 		/* Read a PHY register. */
#define SIOCSMIIREG (SIOCDEVPRIVATE+2) 		/* Write a PHY register. */

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
    {"all-interfaces", 0, 0, 'a'},	/* Show all interfaces. */
	{"Advertise",	1, 0, 'A'},		/* Change the capabilities advertised. */
    {"force",       0, 0, 'f'},		/* Force the operation. */
    {"restart",		0, 0, 'r'},		/* Restart the link negotiation */
    {"Reset",		0, 0, 'R'},		/* Reset the transceiver. */
    {"help", 		0, 0, '?'},		/* Give help */
    {"verbose", 	0, 0, 'v'},		/* Report each action taken.  */
    {"version", 	0, 0, 'V'},		/* Emit version information.  */
    { 0, 0, 0, 0 }
};

#ifdef LIBMII
extern show_mii_details(int ioaddr, int phy_id);
extern monitor_mii(int ioaddr, int phy_id);
#endif

/* Command-line flags. */
unsigned int opt_a = 0,					/* Show-all-interfaces flag. */
	opt_f = 0,					/* Force the operation. */
	verbose = 0,				/* Verbose flag. */
	debug = 0,
	opt_version = 0,
	opt_restart = 0,
	opt_reset = 0,
	opt_watch = 0;
static int nway_advertise = -1;
static int fixed_speed = -1;
int skfd = -1;					/* AF_INET socket for ioctl() calls.	*/
struct ifreq ifr;

int mdio_read(int skfd, int phy_id, int location);
void mdio_write(int skfd, int phy_id, int location, int value);
static int parse_advertise(const char *capabilities);


int
main(int argc, char **argv)
{
	int c, errflag = 0;
	char **spp, *ifname;

	while ((c = getopt_long(argc, argv, "aA:DfF:rRvVw?", longopts, 0)) != EOF)
		switch (c) {
		case 'a': opt_a++; break;
		case 'A': nway_advertise = parse_advertise(optarg); break;
		case 'D': debug++;			break;
		case 'f': opt_f++; break;
		case 'F': fixed_speed = parse_advertise(optarg); break;
		case 'r': opt_restart++;	break;
		case 'R': opt_reset++;		break;
		case 'v': verbose++;		break;
		case 'V': opt_version++;	break;
		case 'w': opt_watch++;		break;
		case '?': errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 2;
	}

	if (verbose || opt_version)
		printf(version);

	/* Open a basic socket. */
	if ((skfd = socket(AF_INET, SOCK_DGRAM,0)) < 0) {
		perror("socket");
		exit(-1);
	}

	if (debug)
		fprintf(stderr, "DEBUG: argc=%d, optind=%d and argv[optind] is %s.\n",
				argc, optind, argv[optind]);

	/* No remaining args means show all interfaces. */
	if (optind == argc) {
		ifname = "eth0";
		fprintf(stderr, "Using the default interface 'eth0'.\n");
	}

	/* Copy the interface name. */
	spp = argv + optind;
	ifname = *spp++;

	if (ifname == NULL) {
		ifname = "eth0";
		fprintf(stderr, "Using the default interface 'eth0'.\n");
#ifdef notdef
		if_print(ifname); */
		(void) close(skfd);
		exit(0);
#endif
	}

	/* Get the vitals from the interface. */
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
	if (ioctl(skfd, SIOCGMIIPHY, &ifr) < 0) {
		fprintf(stderr, "SIOCGMIIPHY on %s failed: %s\n", ifname,
				strerror(errno));
		(void) close(skfd);
		return 1;
	}
	{
		u16 *data = (u16 *)(&ifr.ifr_data);
		unsigned phy_id = data[0];
		int i;

		printf("MII PHY in use is %d.\n", phy_id);
		for (i = 0; i < 32; i++)
			printf(" %4.4x", mdio_read(skfd, phy_id, i));

		if (opt_reset) {
			printf("Resetting the transceiver...\n");
			mdio_write(skfd, phy_id, 0, 0x8000);
		}
		if (phy_id < 32  &&  nway_advertise >= 0) {
			printf(" Setting the media capability advertisement register of "
				   "PHY #%d to 0x%4.4x.\n", phy_id, nway_advertise | 1);
			mdio_write(skfd, phy_id, 4, nway_advertise | 1);
		}

		if (opt_restart) {
			printf("Restarting negotiation...\n");
			mdio_write(skfd, phy_id, 0, 0x0000);
			mdio_write(skfd, phy_id, 0, 0x1200);
		}
		/* To force 100baseTx-HD do  mdio_write(skfd, phy_id, 0, 0x2000); */
		if (fixed_speed >= 0) {
			int reg0_val = 0;
			reg0_val |= (fixed_speed & 0x0180) ? 0x2000 : 0;
			reg0_val |= (fixed_speed & 0x0140) ? 0x0100 : 0;
			printf("Setting the speed to \"fixed\", %4.4x.\n", reg0_val);
			mdio_write(skfd, phy_id, 0, reg0_val);
		}

#ifdef LIBMII
		show_mii_details(skfd, phy_id);
		if (opt_watch)
			monitor_mii(skfd, phy_id);
#endif
	}

	(void) close(skfd);
	return 0;
}

int mdio_read(int skfd, int phy_id, int location)
{
	u16 *data = (u16 *)(&ifr.ifr_data);

	data[0] = phy_id;
	data[1] = location;

	if (ioctl(skfd, SIOCGMIIREG, &ifr) < 0) {
		fprintf(stderr, "SIOCGMIIREG on %s failed: %s\n", ifr.ifr_name,
				strerror(errno));
		return -1;
	}
	return data[3];
}

void mdio_write(int skfd, int phy_id, int location, int value)
{
	u16 *data = (u16 *)(&ifr.ifr_data);

	data[0] = phy_id;
	data[1] = location;
	data[2] = value;

	if (ioctl(skfd, SIOCSMIIREG, &ifr) < 0) {
		fprintf(stderr, "SIOCSMIIREG on %s failed: %s\n", ifr.ifr_name,
				strerror(errno));
	}
}

/* Parse the command line argument for advertised capabilities. */
static int parse_advertise(const char *capabilities)
{
	const char *mtypes[] = {
		"100baseT4", "100baseTx", "100baseTx-FD", "100baseTx-HD",
		"10baseT", "10baseT-FD", "10baseT-HD", 0,
	};
	int cap_map[] = { 0x0200, 0x0180, 0x0100, 0x0080, 0x0060, 0x0040, 0x0020,};
	int i;
	if ( ! capabilities) {
		fprintf(stderr, "You passed -A 'NULL'.  You must provide a media"
				" list to advertise!\n");
		return -1;
	}
	if (debug)
		fprintf(stderr, "Advertise string is '%s'.\n", capabilities);
	for (i = 0; mtypes[i]; i++)
		if (strcmp(mtypes[i], capabilities) == 0)
			return cap_map[i];
	if ((i = strtol(capabilities, NULL, 16)) <= 0xffff)
		return i;
	fprintf(stderr, "Invalid media advertisement '%s'.\n", capabilities);
	return -1;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  compile-command: "gcc -Wall -Wstrict-prototypes -O mii-diag.c -o mii-diag"
 *  compile-command: "gcc -Wall -Wstrict-prototypes -O -DLIBMII mii-diag.c libmii.c -o mii-diag"
 * End:
 */
