/* epic-diag.c: Diagnostics and EEPROM setup program for the SMC EPIC-100 chip.

   This is a diagnostic and EEPROM setup program for Ethernet adapters
   based on the SMC83C170 EPIC/100 chip, as used on the SMC EtherPowerII
   boards.

   Copyright 1997 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771

   References

   http://www.smc.com/components/catalog/smc83c170.html
   http://cesdis.gsfc.nasa.gov/linux/misc/NWay.html
   http://www.national.com/pf/DP/DP83840.html
*/

static char *version_msg =
"epic-diag.c:v0.06 3/26/98 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: epic-diag [-aeEfFmsvVw] [-p <IOport>].\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <linux/in.h>
#include <asm/io.h>

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"show-eeprom",  0, 0, 'e'}, /* Dump EEPROM contents */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"show-mii",  0, 0, 'e'},	/* Dump MII management registers. */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

/* Constants with useful names (you will still need the datasheet though). */
#ifndef PCI_VENDOR_ID_SMC
#define PCI_VENDOR_ID_SMC			0x10B8
#define PCI_DEVICE_ID_SMC_EPIC100	0x0005
#endif

/* The rest of these values should never change. */
/* Offsets to registers, using the (ugh) SMC names. */
enum epic_registers {
  COMMAND=0, IntrStatus=4, INTMASK=8, GENCTL=12, NVCTL=16, EECTL=20,
  CRCCNT=0x20, ALICNT=0x24, MPCNT=0x28,	/* Rx error counters. */
  MIICtrl=0x30, MIIData=0x34, MIICfg=0x38,
  LAN0=64,						/* MAC address. */
  MC0=80,						/* Multicast filter table. */
  RxCtrl=96, TxCtrl=112, TxSTAT=0x74,
  PRxCDAR=0x84, RxSTAT=0xA4, PTxCDAR=0xC4, TxThresh=0xDC,
};

/* Interrupt register bits, using my own meaningful names. */
enum IntrStatus {
  TxIdle=0x40000, RxIdle=0x20000,
  CntFull=0x0200, TxUnderrun=0x0100,
  TxEmpty=0x0080, TxDone=0x0020, RxError=0x0010,
  RxOverflow=0x0008, RxFull=0x0004, RxHeader=0x0002, RxDone=0x0001,
};


#define EEPROM_SIZE 64

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...). */
unsigned short djb_epic_eeprom[EEPROM_SIZE] = {
  /* Currently invalid! */ }; 

/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[EEPROM_SIZE];
unsigned short new_ee_contents[EEPROM_SIZE];

static int verbose = 0, opt_f = 0, debug = 0;
static int show_regs = 0, show_eeprom = 0, show_mii = 0;
static int do_write_eeprom = 0, do_test = 0;

static int do_one_chip(int ioaddr, int part_num);
static int read_eeprom(int ioaddr, int location);
#ifdef notyet
static void write_eeprom(int ioaddr, int index, int value);
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value);
#endif
int mdio_read(int ioaddr, int phy_id, int location);
void mdio_write(int ioaddr, int phy_id, int location, int value);
static void parse_eeprom(unsigned short *ee_data);


/* The interrupt flags. */
const char *intr_names[24] ={
	"Rx Copy Done", "Rx Header Done", "Rx Queue Empty", "Rx Buffer Overflow",
	"Rx CRC error", "Tx done", "Tx chain done", "Tx Queue empty",
	"Tx underrun", "Counter overflow", "Rx threshold crossed",
	"PCI data parity error", "PCI address parity error", "PCI master abort",
	"PCI target abort",
	"Rx status valid", "Interrupt active", "Rx idle", "Tx idle",
	"Rx copy in progress", "Tx copy in progress", "Rx buffers empty",
	"Early Rx threshold passed", "***",
};
/* Non-interrupting events. */
const char *event_names[16] = {
	"Tx Abort", "Rx frame complete", "Transmit done",
};



int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind, rev_num = 0;
	int card_cnt = 0, card_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "#:adDeEfF:i:mp:svVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case '#': card_num = atoi(optarg); break;
		case 'a': show_regs++;		break;
		case 'D': case 'd': debug++;			break;
		case 'e': show_eeprom++;	break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++;			break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': verbose--;		 break;
		case 't': do_test--;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose || show_version)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("epic-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base == 0) {
		FILE *fp;
		fp = fopen("/proc/pci", "r");
		if (fp) {
			char buffer[514];
			int bus, device, function, vendor_id, device_id;
			int state = 0;
			if (debug) printf("Done open of /proc/pci.\n");
			while (fgets(buffer, sizeof(buffer), fp)) {
				if (debug) printf(" Parse state %d line -- %s", state, buffer);
				if (sscanf(buffer, " Bus %d, device %d, function %d",
						   &bus, &device, &function) > 0) {
					state = 1;
					continue;
				}
				if (state == 1) {
					if (sscanf(buffer, " Ethernet controller:  EPIC/100 (rev %d)", &rev_num) > 0) {
						state = 2;
						continue;
					}
					/* Handle a /proc/pci that does not recognize the card. */
					if (sscanf(buffer, " Vendor id=%4x. Device id=%4x",
							   &vendor_id, &device_id) > 0) {
						if (debug)
							printf("Found vendor 0x%4.4x device ID 0x%4.4x.\n",
								   vendor_id, device_id);
						if (vendor_id == PCI_VENDOR_ID_SMC
							&& device_id == PCI_DEVICE_ID_SMC_EPIC100) {
							state = 2;
						} else
							fprintf(stderr, "Unknown PCI device ID %#4.4x.\n",
									device_id);
						continue;
					}
				}
				if (state == 2) {
					if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
						card_cnt++;
						printf("Found SMC83C170 EPIC/100 card #%d at %#x.\n",
							   card_cnt, port_base);
						state = 3;
						if (card_num == 0 || card_num == card_cnt)
							do_one_chip(port_base, 0);
						continue;
					}
				}
			}
		}
	} else
		do_one_chip(port_base, 0);


	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find an EPIC/100 card in /proc/pci.\n  If there is"
				" a card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}
	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' to show device registers,\n"
			   "     '-e' to show EEPROM contents,\n"
			   "  or '-m' to show MII management registers.\n");

	return 0;
}


static int do_one_chip(int ioaddr, int part_version)
{
	int chip_active = 0, chip_lowpower = 0;
	int i;

	/* It's mostly safe to examine the registers and EEPROM during
	   operation.  But warn the user, and make then pass '-f'. */
	if (inl(ioaddr + GENCTL) & 0x0008) {
		chip_lowpower = 1;
		outl(0x0200, ioaddr + GENCTL);		 /* Wake up the chip. */
	} else if ((inl(ioaddr + RxCtrl) & 0x003F) != 0x0000)
		chip_active = 1;

	/* Undocumented bit that must be set for the chip to work. */
	outl(0x8, ioaddr + 0x1c);

	if (verbose || show_regs) {
		unsigned intr_status;

		if (!opt_f) {
		  printf("The EPIC/100 chip appears to be active, so some registers"
				 " will not be read.\n"
				 "To see all register values use the '-f' flag.\n");
		} else
			chip_active = 0;		/* Ignore the chip status with -f */

		printf("EPIC chip registers at %#x", ioaddr);
		for (i = 0; i < 0x100; i += 4) {
		  /* Reading some registers hoses the chip operation. */
		  char dont_read[8] = {0x00, 0x00, 0x00, 0xce, 0xfd, 0xed, 0x7d, 0xff};
		  if ((i & 0x1f) == 0)
			  printf("\n 0x%3.3X:", i);
		  if (chip_active && (dont_read[i>>5]) & (1<<((i>>2) & 7)))
			  printf(" ********");
		  else
			  printf(" %8.8x", inl(ioaddr + i));
		}
		printf("\n");
		intr_status = inw(ioaddr + IntrStatus);
		printf("  %snterrupt sources are pending.\n",
			   (intr_status & 0x03ff) ? "I": "No i");
		if (intr_status) {
		  for (i = 0; i < 24; i++)
			  if (intr_status & (1<<i))
				  printf("   %s indication.\n", intr_names[i]);
		}
	}

	/* Read the EEPROM. */
	for (i = 0; i < EEPROM_SIZE; i++)
		eeprom_contents[i] = read_eeprom(ioaddr, i);

	if (verbose || show_eeprom) {
		unsigned short sum = 0;
		printf("EEPROM contents:");
		for (i = 0; i < EEPROM_SIZE; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
			sum += eeprom_contents[i];
		}
		printf("\n The word-wide EEPROM checksum is %#4.4x.\n", sum);
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose || show_eeprom) {
		parse_eeprom(eeprom_contents);
	}

	/* Show up to four (not just the on-board) PHYs. */
	if (verbose || show_mii) {
		int phys[4], phy, phy_idx = 0;
		int mii_reg;
		for (phy = 0; phy < 32 && phy_idx < 4; phy++) {
			int mii_status = mdio_read(ioaddr, phy, 0);
			if (mii_status != 0xffff  && mii_status != 0x0000) {
				phys[phy_idx++] = phy;
				printf(" MII PHY found at address %d.\n", phy);
			}
		}
		if (phy_idx == 0)
			printf(" ***WARNING***: No MII transceivers found!\n");
		for (phy = 0; phy < phy_idx; phy++) {
			printf(" MII PHY #%d transceiver registers:", phys[phy]);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n  " : "",
					   mdio_read(ioaddr, phys[phy], mii_reg));
			printf(".\n");
		}
#ifdef LIBMII
		show_mii_details(ioaddr, phys[0]);
		if (show_mii > 1)
			monitor_mii(ioaddr, phys[0]);
#endif
	}

	if (chip_lowpower)
		outl(0x0008, ioaddr + GENCTL);
	return 0;
}


/* Serial EEPROM section. */

/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x04	/* EEPROM shift clock. */
#define EE_CS			0x02	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x08	/* EEPROM chip data in. */
#define EE_WRITE_0		0x01
#define EE_WRITE_1		0x09
#define EE_DATA_READ	0x10	/* EEPROM chip data out. */
#define EE_ENB			(0x0001 | EE_CS)

/* Delay between EEPROM clock transitions.
   The 1.2 code is a "nasty" timing loop, but PC compatible machines are
   *supposed* to delay an ISA-compatible period for the SLOW_DOWN_IO macro.  */
#ifdef _LINUX_DELAY_H
#define eeprom_delay(nanosec)	udelay((nanosec + 999)/1000)
#else
#define eeprom_delay(nanosec)	do { int _i = 3; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)
#endif

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5 << 6)
#define EE_READ_CMD		(6 << 6)
#define EE_ERASE_CMD	(7 << 6)

static int read_eeprom(int ioaddr, int location)
{
	int i;
	unsigned short retval = 0;
	short ee_addr = ioaddr + EECTL;
	int read_cmd = location | EE_READ_CMD;
	
	outl(EE_ENB & ~EE_CS, ee_addr);
	outl(EE_ENB, ee_addr);
	
	/* Shift the read command bits out. */
	for (i = 10; i >= 0; i--) {
		short dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outl(EE_ENB | dataval, ee_addr);
		eeprom_delay(100);
		outl(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(150);
		outl(EE_ENB | dataval, ee_addr);	/* Finish EEPROM a clock tick. */
		eeprom_delay(250);
	}
	outl(EE_ENB, ee_addr);
	
	for (i = 16; i > 0; i--) {
		outl(EE_ENB | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(100);
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
		outl(EE_ENB, ee_addr);
		eeprom_delay(100);
	}

	/* Terminate the EEPROM access. */
	outl(EE_ENB & ~EE_CS, ee_addr);
	return retval;
}

#ifdef notyet
static void write_eeprom(int ioaddr, int index, int value)
{
	int bogus_cnt = 1000;

	outb(0x01, ioaddr + EEFeature); /* Enable EE write */
	outb(value, ioaddr + EEWrite);
	outb(index, ioaddr + EEAddr);
	outb(0x20 | ((index >> 8) & 7), ioaddr + EECtrl);
	while (inb(ioaddr + EEStatus) < 0  && --bogus_cnt > 0)
		;
	outb(0x00, ioaddr + EEFeature); /* Disable EE write */
	return;
  fprintf(stderr, "write_eeprom() not yet implemented.\n");
}

static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value)
{
	if (ee_values[index] != new_value) {
		if (do_write_eeprom) {
			printf("Writing new %s entry 0x%4.4x.\n",
				   field_name, new_value);
			write_eeprom(ioaddr, index, new_value);
		} else
			printf(" Would write new %s entry 0x%4.4x (old value 0x%4.4x).\n",
				   field_name, new_value, ee_values[index]);
		ee_values[index] = new_value;
		return 1;
	}
	return 0;
}
#endif /* notyet */


/* Read and write the MII registers using the hardware support.
   This may also be done with a software-generated serial bit stream, but
   the hardware method should be more reliable.
   */
#define MDIO_SHIFT_CLK	0x20
#define MDIO_DATA_WRITE 0x40
#define MDIO_ENB		0x80
#define MDIO_DATA_READ	0x40
#define MII_READOP		1
#define MII_WRITEOP		2
int mdio_read(int ioaddr, int phy_id, int location)
{
	int i;

	outl((phy_id << 9) | (location << 4) | MII_READOP, ioaddr + MIICtrl);
	for (i = 10000; i > 0; i--) {
		int ctrl = inl(ioaddr + MIICtrl);
		if ((ctrl & 0x08) && debug)
			printf("MII control register returned %8.8x at tick %d.\n",
				   ctrl, 10000 - i);
		if ((ctrl & MII_READOP) == 0)
			break;
	}
	if (debug)
		printf("MII register %d:%d took %d ticks to read: %8.8x -> %8.8x.\n",
			   phy_id, location, 10000 - i, inl(ioaddr + MIICtrl),
			   inl(ioaddr + MIIData));
	return inw(ioaddr + MIIData);
}

void mdio_write(int ioaddr, int phy_id, int location, int value)
{
	int i;

	outw(value, ioaddr + MIIData);
	outl((phy_id << 9) | (location << 4) | MII_WRITEOP, ioaddr + MIICtrl);
	for (i = 10000; i > 0; i--) {
		int ctrl = inl(ioaddr + MIICtrl);
		if ((ctrl & 0x08) && debug)
			printf("MII control register returned %8.8x at tick %d.\n",
				   ctrl, 10000 - i);
		if ((ctrl & MII_WRITEOP) == 0)
			break;
	}
	if (debug)
		printf("MII register %d:%d took %d ticks to write: %8.8x -> %8.8x.\n",
			   phy_id, location, 10000 - i, inl(ioaddr + MIICtrl),
			   inl(ioaddr + MIIData));
	return;
}


static void parse_eeprom(unsigned short *eeprom)
{
	unsigned char *p = (void *)eeprom;
	int i;
	unsigned char sum = 0;

	printf("Parsing the EEPROM of a EPIC/100:\n Station Address ");
	for (i = 0; i < 5; i++)
		printf("%2.2X:", p[i]);
	printf("%2.2X.\n Board name '%.20s', revision %d.\n", p[5], p + 88, p[i]);
	for (i = 0; i < 8; i++)
		sum += p[i];
	printf(" Calculated checksum is %2.2x.\n", sum);

	return;
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o epic-diag epic-diag.c"
 *  alt-compile-command: "cc -O -Wall -o epic-diag epic-diag.c -DLIBMII libmii.c"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
