/* eepro100-diag.c: EEPROM setup program for ethercards using Digital DC21x4*.

   This is a diagnostic and EEPROM setup program for the Ethernet adapters
   based on the Intel "Speedo3" i82557 chip: EtherExpress Pro100B, and
   EEPro PCI 10+.
   (Note: The EEPROM write code is not currently functional.)

   Copyright 1996-1998 by Donald Becker.
   This version released under the Gnu Public Lincese, incorporated herein
   by reference.  Contact the author for use under other terms.

   This program must be compiled with "-O"!  See the bottom of this file
   for the suggested compile-command.

   The author may be reached as becker@cesdis.gsfc.nasa.gov.
   C/O USRA-CESDIS, Code 930.5 Bldg. 28, Nimbus Rd., Greenbelt MD 20771
*/

static char *version_msg =
"eepro100-diag.c:v0.07 2/25/98 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
static char *usage_msg =
"Usage: eepro100-diag [-aEfFsvVw] [-p <IOport>] [-A 10baseT|100baseTx[-FD]>]\n";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <getopt.h>
#include <strings.h>
#include <linux/in.h>
#include <errno.h>
#include <asm/io.h>
#ifdef LIBMII
extern show_mii_details(int ioaddr, int phy_id);
extern monitor_mii(int ioaddr, int phy_id);
#endif

/* Grrr, glibc fails the define this.. */
extern iopl(int level);

struct option longopts[] = {
 /* { name  has_arg  *flag  val } */
	{"Advertise", 1, 0, 'A'},
	{"base-address", 1, 0, 'p'},
	{"show_all_registers",	0, 0, 'a'},	/* Print all registers. */
	{"help",	0, 0, 'h'},	/* Give help */
	{"emergency-rewrite",  0, 0, 'E'}, /* Re-write a corrupted EEPROM.  */
	{"force-detection",  0, 0, 'f'},
	{"new-interface",  1, 0, 'F'},	/* New interface (built-in, AUI, etc.) */
	{"new-IOaddress",	1, 0, 'P'},	/* New base I/O address. */
	{"new-irq",	1, 0, 'Q'},		/* New interrupt number */
	{"quiet",	0, 0, 'q'},		/* Decrease verbosity */
	{"verbose",	0, 0, 'v'},		/* Verbose mode */
	{"version", 0, 0, 'V'},		/* Display version number */
	{"write-EEPROM", 1, 0, 'w'},/* Actually write the EEPROM with new vals */
	{ 0, 0, 0, 0 }
};

/* Offsets to the various registers.
   All accesses need not be longword aligned. */
enum speedo_offsets {
	SCBStatus = 0, SCBCmd = 2,	/* Rx/Command Unit command and status. */
	SCBPointer = 4,				/* General purpose pointer. */
	SCBPort = 8,				/* Misc. commands and operands.  */
	SCBflash = 12, SCBeeprom = 14, /* EEPROM and flash memory control. */
	SCBCtrlMDI = 16,			/* MDI interface control. */
	SCBEarlyRx = 20,			/* Early receive byte count. */
};

#define EEPROM_SIZE 128

/* Last-hope recovery major boo-boos: rewrite the EEPROM with the values
   from my card (and hope I don't met you on the net...).
   Only experts should use this! */
unsigned short djb_eepro100_eeprom[64] = {
	0xa000, 0x49c9, 0x87ab, 0x0000, 0x0000, 0x0101, 0x4401, 0x0000,
	0x6455, 0x2022, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0x7fcd, }; 

/* Values read from the EEPROM, and the new image. */
unsigned short eeprom_contents[64];
unsigned short new_ee_contents[64];

static int verbose = 1, opt_f = 0;
static int show_regs = 0, show_eeprom = 0, show_mii = 0;
static int nway_advertise = 0;

int do_write_eeprom = 0;
int has_mii = 0;
int phy0 = -1;
int phy1 = -1;

static int read_eeprom(int ioaddr, int location);
int mdio_read(int ioaddr, int phy_id, int location);
void mdio_write(int ioaddr, int phy_id, int location, int value);
#ifndef LIBMII
static int monitor_mii(int ioaddr, int phy_id);
#endif

#ifdef notyet
static void write_eeprom(int ioaddr, int index, int value);
static int do_update(unsigned short *ee_values,
					 int index, char *field_name, int new_value);
#endif

static int do_one_chip(int ioaddr, int part_num);
static int parse_advertise(const char *capabilities);
static void parse_eeprom(unsigned short *ee_data);


int
main(int argc, char **argv)
{
	int port_base = 0;
	int new_interface = -1;
	int errflag = 0, show_version = 0;
	int emergency_rewrite = 0;
	int c, longind, rev_num = 0;
	extern char *optarg;

	while ((c = getopt_long(argc, argv, "aeEfF:i:mp:Q:svVw",
							longopts, &longind))
		   != -1)
		switch (c) {
		case 'a': show_regs++;		 break;
		case 'A': nway_advertise = parse_advertise(optarg); break;
		case 'e': show_eeprom++;	 break;
		case 'E': emergency_rewrite++;	 break;
		case 'f': opt_f++; break;
		case 'F':
			if (strncmp(optarg, "10base", 6) == 0) {
				switch (optarg[6]) {
				case 'T':  new_interface = 0; break;
				case '2':  new_interface = 3; break;
				case '5':  new_interface = 1; break;
				default: errflag++;
				}
			} else if (strcmp(optarg, "AUI") == 0)
				new_interface = 1;
			else if (optarg[0] >= '0' &&  optarg[0] <= '3'
					   &&  optarg[1] == 0)
				new_interface = optarg[0] - '0';
			else {
				fprintf(stderr, "Invalid interface specified: it must be"
						" 0..3, '10base{T,2,5}' or 'AUI'.\n");
				errflag++;
			}
			break;
		case 'm': show_mii++;	 break;
		case 'p':
			port_base = strtol(optarg, NULL, 16);
			break;
		case 'q': verbose--;		 break;
		case 'v': verbose++;		 break;
		case 'V': show_version++;		 break;
		case 'w': do_write_eeprom++;	 break;
		case '?':
			errflag++;
		}
	if (errflag) {
		fprintf(stderr, usage_msg);
		return 3;
	}

	if (verbose)
		printf(version_msg);

	/* Get access to all of I/O space. */
	if (iopl(3) < 0) {
		perror("eepro100-diag: iopl()");
		fprintf(stderr, "This program must be run as root.\n");
		return 2;
	}

	/* Try to read a likely port_base value from /proc/pci. */
	if (port_base)
		do_one_chip(port_base, 0);
	else {
		FILE *fp;
		char buffer[514];
		int bus, device, function, device_id;
		int state = 0;

		fp = fopen("/proc/pci", "r");
		if (fp == NULL) {
			fprintf(stderr, "Unable to open /proc/pci to search for PCI"
					" devices.\n");
			return 2;
		}
		if (verbose > 2) printf("Done open of /proc/pci.\n");
		while (fgets(buffer, sizeof(buffer), fp)) {
			if (verbose > 2) printf("--scanning a line %s\n", buffer);
			if (sscanf(buffer, " Bus %d, device %d, function %d",
					   &bus, &device, &function) > 0) {
				state = 1;
				continue;
			}
			if (state == 1) {
				if (sscanf(buffer,
						   " Ethernet controller: Intel 82557 (rev %d)",
						   &rev_num) > 0) {
					state = 2;
					continue;
				}
				/* Handle older kernels that don't recognize the card. */
				if (sscanf(buffer,
						   " Vendor id=8086. Device id=%x", &device_id) > 0) {
					if (device_id == 0x1229) {
						state = 2;
					} else
						fprintf(stderr, "Unknown Intel PCI device ID %d.\n",
								device_id);
					continue;
				}
			}
			if (state == 2) {
				if (sscanf(buffer, "  I/O at %x", &port_base) > 0) {
					printf("Found Intel i82557/i82558 Speedo3 chip at %#x.\n",
						   port_base);
					state = 3;
					do_one_chip(port_base, rev_num);
					break;
				}
			}
		}
	}

	if (port_base == 0) {
		fprintf(stderr,
				"Unable to find an EEPro100 card in /proc/pci.\n  If there is"
				" a i82557 card in the machine, explicitly set the I/O port"
				" address\n  using '-p <ioaddr>\n");
		return ENODEV;
	}

	if (show_regs == 0  &&  show_eeprom == 0  &&  show_mii == 0)
		printf(" Use '-a' to show device registers,\n"
			   "     '-e' to show parsed EEPROM contents, -e -e to show more,\n"
			   "  or '-m' to show MII management registers.\n");

	return 0;
}

static int do_one_chip(int ioaddr, int part_num)
{
	int i;

	/* It's mostly safe to use the EEPro100 EEPROM and MDIO register during
	   operation.  But warn the user, and make then pass '-f'. */
	if (!opt_f  && (inw(ioaddr + SCBStatus) & 0x00fc) != 0x0000) {
		printf("A potential i82557 chip has been found, but it appears to "
			   "be active.\nEither shutdown the network, or use the"
			   " '-f' flag.\n");
		return 1;
	}

	if (verbose > 1 || show_regs) {
		static const char *tx_state[] = {
			"Idle", "Suspended", "Active", "Unknown"};
		static const char *rx_state[] = {
			"Idle", "Idle/Suspended", "No Resources", "Unknown",
			"Ready", "Broken-5",  "Broken-6",  "Broken-7", "Broken-8",
			"Suspended - no buffers", "No resources - no buffers", "Broken-11",
			"Ready, but no buffers",  "Broken-13", "Broken-14", "Broken-15",
		};
		unsigned short status;

		printf("i82557 chip registers at %#x:\n ", ioaddr);
		for (i = 0; i < 0x18; i += 4)
			printf(" %8.8x", inl(ioaddr + i));
		printf("\n");
		status = inw(ioaddr + SCBStatus);
		printf("  %snterrupt sources are pending.\n",
			   (status & 0xff00) ? "I": "No i");
		printf("   The transmit unit state is '%s'.\n",
			   tx_state[(status>>6) & 3]);
		printf("   The receive unit state is '%s'.\n",
			   rx_state[(status>>2) & 15]);
		if (inw(ioaddr + SCBCmd) != 0)
			printf(" The Command register has an unprocessed command "
				   "%4.4x(?!).\n",
				   inw(ioaddr + SCBCmd));
	}

	/* Read the EEPROM. */
	for (i = 0; i < EEPROM_SIZE/2; i++)
		eeprom_contents[i] = read_eeprom(ioaddr, i);
	phy0 = eeprom_contents[6] & 0x8000 ? -1 : eeprom_contents[6] & 0x1f;
	phy1 = eeprom_contents[7] & 0x8000 ? -1 : eeprom_contents[7] & 0x1f;

	if (verbose > 2 || show_eeprom > 1) {
		unsigned short sum = 0;
		printf("EEPROM contents:");
		for (i = 0; i < EEPROM_SIZE/2; i++) {
			printf("%s %4.4x", (i & 7) == 0 ? "\n ":"",
				   eeprom_contents[i]);
			sum += eeprom_contents[i];
		}
		printf("\n The EEPROM checksum (should be 0xbaba) is %#2.2x.\n", sum);
	}

	/* The user will usually want to see the interpreted EEPROM contents. */
	if (verbose > 1 || show_eeprom) {
		parse_eeprom(eeprom_contents);
	}

	/* Grrr, it turns out they the PHY is not always #0. */
	if (verbose > 1 || show_mii) {
		if (phy0 > 0 &&
			mdio_read(ioaddr, phy0, 0) != 0xffff  && 
			mdio_read(ioaddr, phy0, 0) != 0x0000) {
			int mii_reg;
			printf(" MII PHY #%d transceiver registers:", phy0);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n " : "",
					   mdio_read(ioaddr, phy0, mii_reg));
			printf(".\n");
#ifdef LIBMII
			show_mii_details(ioaddr, phy0);
#endif
		}
		if (phy1 > 0 &&
			mdio_read(ioaddr, phy1, 0) != 0xffff  && 
			mdio_read(ioaddr, phy1, 0) != 0x0000) {
			int mii_reg;
			printf(" Alternate MII PHY (#%d) transceiver registers:", phy1);
			for (mii_reg = 0; mii_reg < 32; mii_reg++)
				printf("%s %4.4x", (mii_reg % 8) == 0 ? "\n " : "",
					   mdio_read(ioaddr, phy1, mii_reg));
			printf(".\n");
#ifdef LIBMII
			show_mii_details(ioaddr, phy1);
#endif
		}
		if (phy0 > 0 &&  nway_advertise > 0) {
			printf(" Setting the media capability advertisement register of "
				   "PHY #%d to 0x%4.4x.\n", phy0, nway_advertise | 1);
			mdio_write(ioaddr, phy0, 4, nway_advertise | 1);
		}
	}
	if (show_mii > 1) {			/* Monitor MII status */
		monitor_mii(ioaddr, phy0);
	}
	return 0;
}

static int parse_advertise(const char *capabilities)
{
	const char *mtypes[] = {
		"100baseT4", "100baseTx", "100baseTx-FD", "100baseTx-HD",
		"10baseT", "10baseT-FD", "10baseT-HD", 0,
	};
	int cap_map[] = { 0x0200, 0x0180, 0x0100, 0x0080, 0x0060, 0x0040, 0x0020,};
	int i;
	if (verbose > 2)
		fprintf(stderr, "Advertise string is '%s'.\n", capabilities);
	for (i = 0; mtypes[i]; i++)
		if (strcmp(mtypes[i], capabilities) == 0)
			return cap_map[i];
	if ((i = strtol(capabilities, NULL, 16)) <= 0xffff)
		return i;
	fprintf(stderr, "Invalid media advertisement '%s'.\n", capabilities);
	return 0;
}


/* Serial EEPROM section.
   A "bit" grungy, but we work our way through bit-by-bit :->. */
/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x01	/* EEPROM shift clock. */
#define EE_CS			0x02	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x04	/* EEPROM chip data in. */
#define EE_WRITE_0		0x01
#define EE_WRITE_1		0x05
#define EE_DATA_READ	0x08	/* EEPROM chip data out. */
#define EE_ENB			(0x4800 | EE_CS)

/* Delay between EEPROM clock transitions.
   This is a "nasty" timing loop, but PC compatible machines are defined
   to delay an ISA compatible period for the SLOW_DOWN_IO macro.  */
#define eeprom_delay(nanosec)	do { int _i = 3; while (--_i > 0) { __SLOW_DOWN_IO; }} while (0)

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5 << 6)
#define EE_READ_CMD		(6 << 6)
#define EE_ERASE_CMD	(7 << 6)

static int read_eeprom(int ioaddr, int location)
{
	int i;
	unsigned short retval = 0;
	int ee_addr = ioaddr + SCBeeprom;
	int read_cmd = location | EE_READ_CMD;
	
	outw(EE_ENB & ~EE_CS, ee_addr);
	outw(EE_ENB, ee_addr);
	
	/* Shift the read command bits out. */
	for (i = 10; i >= 0; i--) {
		short dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		outw(EE_ENB | dataval, ee_addr);
		eeprom_delay(100);
		outw(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(150);
		outw(EE_ENB | dataval, ee_addr);	/* Finish EEPROM a clock tick. */
		eeprom_delay(250);
	}
	outw(EE_ENB, ee_addr);
	
	for (i = 15; i >= 0; i--) {
		outw(EE_ENB | EE_SHIFT_CLK, ee_addr);
		eeprom_delay(100);
		retval = (retval << 1) | ((inw(ee_addr) & EE_DATA_READ) ? 1 : 0);
		outw(EE_ENB, ee_addr);
		eeprom_delay(100);
	}

	/* Terminate the EEPROM access. */
	outw(EE_ENB & ~EE_CS, ee_addr);
	return retval;
}

int mdio_read(int ioaddr, int phy_id, int location)
{
	int val, boguscnt = 64*4;		/* <64 usec. to complete, typ 27 ticks */
	outl(0x08000000 | (location<<16) | (phy_id<<21), ioaddr + SCBCtrlMDI);
	do {
		SLOW_DOWN_IO;
		val = inl(ioaddr + SCBCtrlMDI);
		if (--boguscnt < 0) {
			fprintf(stderr, " mdio_read() timed out with val = %8.8x.\n", val);
		}
	} while (! (val & 0x10000000));
	return val & 0xffff;
}

void mdio_write(int ioaddr, int phy_id, int location, int value)
{
	int val, boguscnt = 64*4;		/* <64 usec. to complete, typ 27 ticks */
	outl(0x04000000 | (location<<16) | (phy_id<<21) | value,
		 ioaddr + SCBCtrlMDI);
	do {
		SLOW_DOWN_IO;
		val = inl(ioaddr + SCBCtrlMDI);
		if (--boguscnt < 0) {
			fprintf(stderr, " mdio_write() timed out with val = %8.8x.\n",
					val);
		}
	} while (! (val & 0x10000000));
	return;
}

#ifndef LIBMII
static int monitor_mii(int ioaddr, int phy_id)
{
	int i;
	unsigned short new_1, baseline_1 = mdio_read(ioaddr, phy_id, 1);
	if (baseline_1 == 0xffff) {
		fprintf(stderr, "No MII transceiver present to monitor.\n");
		return -1;
	}
	printf("  Baseline value of MII status register is %4.4x.\n",
		   baseline_1);
	for (i = 0; i < 60; i++) {
		new_1 = mdio_read(ioaddr, phy_id, 1);
		if (new_1 != baseline_1) {
			printf("  MII status register changed to %4.4x.\n", new_1);
			baseline_1 = new_1;
		}
		sleep(1);
	}
	return 0;
}
#endif  /* not LIBMII */

#ifdef notdef					/* Code blurbs for future reference. */
	phys_addr = (unsigned char *)(eeprom_contents + PhysAddr01);
	printf("Primary physical address is %2.2x", phys_addr[1]);
	for (i = 1; i < 6; i++)
		printf(":%2.2x", phys_addr[i^1]);
#endif

#ifdef notdef
static void write_eeprom(int ioaddr, int index, int value)
{
  fprintf(stderr, "write_eeprom() not yet implemented.\n");
}
#endif


/* PHY media interface chips. */
static const char *phys[] = {
	"None", "i82553-A/B", "i82553-C", "i82503",
	"DP83840", "80c240", "80c24", "i82555",
	"unknown-8", "unknown-9", "DP83840A", "unknown-11",
	"unknown-12", "unknown-13", "unknown-14", "unknown-15", };
enum phy_chips { NonSuchPhy=0, I82553AB, I82553C, I82503, DP83840, S80C240,
					 S80C24, I82555, DP83840A=10, };
static const char is_mii[] = { 0, 1, 1, 0, 1, 1, 0, 1 };
const char *connectors[] = {" RJ45", " BNC", " AUI", " MII"};


static void parse_eeprom(unsigned short *eeprom)
{
	unsigned char dev_addr[6];
	int i, j;

	printf("Intel EtherExpress Pro 10/100 EEPROM contents:\n"
		   "  Station address ");
	for (j = 0, i = 0; i < 3; i++) {
		dev_addr[j++] = eeprom[i];
		dev_addr[j++] = eeprom[i] >> 8;
	}
	for (i = 0; i < 5; i++)
		printf("%2.2X:", dev_addr[i]);
	printf("%2.2X.\n", dev_addr[i]);
	if ((eeprom[3] & 0x03) != 3)
		printf("  Receiver lock-up bug exists. (The driver work-around *is* "
			   "implemented.)\n");
	printf("  Board assembly %4.4x%2.2x-%3.3d, Physical connectors present:",
		   eeprom[8], eeprom[9]>>8, eeprom[9] & 0xff);
	for (i = 0; i < 4; i++)
		if (eeprom[5] & (1<<i))
			printf(connectors[i]);
	printf("\n  Primary interface chip %s PHY #%d.\n",
		   phys[(eeprom[6]>>8)&7], phy0);
	if (((eeprom[6]>>8) & 0x3f) == DP83840)
		printf("  Transceiver-specific setup is required for the DP83840"
			   " transceiver.\n");
	if (eeprom[7] & 0x0700)
		printf("    Secondary interface chip %s.\n",
			   phys[(eeprom[7]>>8)&7]);

	return;
}


/*
 * Local variables:
 *  compile-command: "cc -O -Wall -o eepro-diag eepro100-diag.c"
 *  alt-compile-command: "cc -O -Wall -o eepro-diag eepro100-diag.c -DLIBMII libmii.c"
 *  tab-width: 4
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 */
