/*****************************************************************************/

/*
 *	stlstty.c  -- stallion intelligent multiport special options.
 *
 *	Copyright (C) 1996  Greg Ungerer (gerg@stallion.oz.au).
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>

#include "cdk.h"

/*****************************************************************************/

char	*version = "1.1.2";
char	*progname;

/*
 *	Define some marker flags (to append to pflag values).
 */
#define	PFLAG_ON	0x40000000
#define	PFLAG_OFF	0x80000000

/*
 *	List of all options used. Use the option structure even though we
 *	don't use getopt!
 */
struct option longops[] = {
	{ "-V", 0, 0, 'V' },
	{ "--version", 0, 0, 'V' },
	{ "-?", 0, 0, 'h' },
	{ "-h", 0, 0, 'h' },
	{ "--help", 0, 0, 'h' },
	{ "maprts", 0, 0, (PFLAG_ON | P_MAPRTS) },
	{ "-maprts", 0, 0, (PFLAG_OFF | P_MAPRTS) },
	{ "mapcts", 0, 0, (PFLAG_ON | P_MAPCTS) },
	{ "-mapcts", 0, 0, (PFLAG_OFF | P_MAPCTS) },
	{ "rtslock", 0, 0, (PFLAG_ON | P_RTSLOCK) },
	{ "-rtslock", 0, 0, (PFLAG_OFF | P_RTSLOCK) },
	{ "ctslock", 0, 0, (PFLAG_ON | P_CTSLOCK) },
	{ "-ctslock", 0, 0, (PFLAG_OFF | P_CTSLOCK) },
	{ "loopback", 0, 0, (PFLAG_ON | P_LOOPBACK) },
	{ "-loopback", 0, 0, (PFLAG_OFF | P_LOOPBACK) },
	{ "fakedcd", 0, 0, (PFLAG_ON | P_FAKEDCD) },
	{ "-fakedcd", 0, 0, (PFLAG_OFF | P_FAKEDCD) },
	{ "dtrfollow", 0, 0, (PFLAG_ON | P_DTRFOLLOW) },
	{ "-dtrfollow", 0, 0, (PFLAG_OFF | P_DTRFOLLOW) },
	{ 0, 0, 0, 0 }
};

/*****************************************************************************/

/*
 *	Declare internal function prototypes here.
 */
void	usage(void);

/*****************************************************************************/

void usage()
{
	fprintf(stderr, "Usage: %s [OPTION] [ARGS]\n\n", progname);
	fprintf(stderr, "  -h, --help            print this information\n");
	fprintf(stderr, "  -V, --version         show version information and exit\n");
	fprintf(stderr, "  maprts, -maprts       set RTS mapping to DTR on or off\n");
	fprintf(stderr, "  mapcts, -mapcts       set CTS mapping to DCD on or off\n");
	fprintf(stderr, "  rtslock, -rtslock     set RTS hardware flow on or off\n");
	fprintf(stderr, "  ctslock, -ctslock     set CTS hardware flow on or off\n");
	fprintf(stderr, "  fakedcd, -fakedcd     set fake DCD on or off\n");
	fprintf(stderr, "  dtrfollow, -dtrfollow set DTR data follow on or off\n");
	fprintf(stderr, "  loopback, -loopback   set port internal loop back on or off\n");
	exit(0);
}

/*****************************************************************************/

void getpflags()
{
	unsigned long	pflags;

	if (ioctl(0, STL_GETPFLAG, &pflags) < 0) {
		fprintf(stderr, "ERROR: not a Stallion serialport\n");
		exit(1);
	}

	if (pflags & P_MAPRTS)
		printf("maprts ");
	else
		printf("-maprts ");
	if (pflags & P_MAPCTS)
		printf("mapcts ");
	else
		printf("-mapcts ");

	if (pflags & P_RTSLOCK)
		printf("rtslock ");
	else
		printf("-rtslock ");
	if (pflags & P_CTSLOCK)
		printf("ctslock ");
	else
		printf("-ctslock ");

	if (pflags & P_FAKEDCD)
		printf("fakedcd ");
	else
		printf("-fakedcd ");
	if (pflags & P_DTRFOLLOW)
		printf("dtrfollow ");
	else
		printf("-dtrfollow ");
	if (pflags & P_LOOPBACK)
		printf("loopback ");
	else
		printf("-loopback ");
	printf("\n");
}

/*****************************************************************************/

void setpflags(unsigned long pflagin, unsigned long pflagout)
{
	unsigned long	pflags;

	if (ioctl(0, STL_GETPFLAG, &pflags) < 0) {
		fprintf(stderr, "ERROR: not a Stallion serial port\n");
		exit(1);
	}

	pflags &= ~(pflagout & ~PFLAG_OFF);
	pflags |= (pflagin & ~PFLAG_ON);

	if (ioctl(0, STL_SETPFLAG, &pflags) < 0) {
		fprintf(stderr, "ERROR: ioctl(SET_SETPFLAGS) failed, errno=%d\n", errno);
		exit(1);
	}
}

/*****************************************************************************/

void main(int argc, char *argv[])
{
	unsigned long	pflagin, pflagout;
	int		optind, optfound;
	int		i, val;

	progname = argv[0];
	pflagin = 0;
	pflagout = 0;

	for (optind = 1; (optind < argc); optind++) {
		optfound = 0;
		for (i = 0; (longops[i].name[0] != 0) ; i++) {
			if (strcmp(argv[optind], &(longops[i].name[0])) == 0) {
				val = longops[i].val;
				optfound++;
				break;
			}
		}
		if (optfound == 0) {
			fprintf(stderr, "ERROR: invalid option '%s'\n", argv[optind]);
			usage();
			exit(1);
		}

		switch (val) {
		case 'V':
			printf("%s version %s\n", progname, version);
			exit(0);
			break;
		case 'h':
			usage();
			break;
		default:
			if (val & PFLAG_ON) {
				pflagin |= val;
			} else if (val & PFLAG_OFF) {
				pflagout |= val;
			} else {
				fprintf(stderr, "ERROR: unknown option found, val=%x!\n", val);
				exit(1);
			}
		}
	}

	if (pflagin | pflagout)
		setpflags(pflagin, pflagout);
	else
		getpflags();

	exit(0);
}

/*****************************************************************************/
