/*
 * linux/driver/char/gs4500.c
 *
 *	Scanner driver for GS4500: version 1.6
 *	16.4.96 Jan Willamowius (jan@janhh.shnet.org)
 *
 *	Changes:
 *	- added interrupt handling (thanks to Rick Lyons for
 *	  the patch)
 *	- updated for kernel 1.3.88
 *
 *	Scanner driver for GS4500: version 1.5
 *	3.1.96 Jan Willamowius (jan@janhh.shnet.org)
 *
 *	Changes:
 *      - moved to device major 16
 *      - updated for kernel 1.3.45 (and up) and ELF
 *      - made gcc 2.7.2 happy
 *
 *	Scanner driver for GS4500: version 1.4
 *	19.5.95 Jan Willamowius (jan@janhh.shnet.org)
 *
 *	Changes:
 *	- modified for kernels > 1.1.80 and modutils-1.1.87
 *      - safe allocation of DMA buffer (finally)
 *      - added request_region()
 *
 *	Scanner driver for GS4500: version 1.3
 *	14.12.94 Jan Willamowius
 *
 *	Changes:
 *	- Check if system has more than 8 MB ram on module load
 *	  and give a warning.
 *	- updated request_dma for newer kernels
 *	- fixed cli()/sti()
 *
 *	Scanner driver for GS4500: version 1.2
 *	5.10.94	Jan Willamowius
 *
 *	Changes:
 *	- changed buffer allocation to make
 *	  loadable module work
 *	- changed ioctls to be compatible with M105, AC4096 and
 * 	  Logitech driver
 *
 *
 *	Scanner driver for GS4500: version 1.1
 *	1.10.94	Jan Willamowius
 *
 *	Changes:
 *	- added automatic detection what io adresses and
 *	  dma channel is set on the interface board
 *	- fixed a bug with dma channel selection
 *	- scanner dosn't flash when driver loads
 *	- changed device major to 26 like the other handheld
 *	  scanner drivers use (avoids conflict with ibcs and
 * 	  sound driver)
 *	- added fsync to file_operations table (= NULL)
 *	- adapted scandemo from logitech driver for gs4500
 *	- added the option to create a loadable kernel module
 *	- small changes here and there...
 *
 *	Supported Scanner:
 *	- Genius GeniScan GS4500
 *	? Genius GeniScan GS4000 (DOS drivers for the GS4000 work
 *	  with my GS4500 so I'd assume this driver works with a
 *	  GS4000)
 *	? Genius GeniScan GS4500A (again, just a guess)
 *
 *	
 *	Scanner driver for GS4500: version 1.0
 *
 *	Depending on how much scanner support we want, the
 *	following should be done:
 *
 *	    Split this file into device independent code (scanner.c)
 *	    and device dependent stuff (gs4500.c, artec.c, etc).
 *
 *	    Get an official major # allocated by RICK@EE.UWM.EDU.
 *
 *	    Create include/scanner.h with major/minors and
 *	    struct scan_info.
 *
 *	    Add configuration to toplevel makefile.
 *
 *	(C) Copyright 1992,1993 Richard Lyons
 *	      pclink@qus102.qld.tne.oz.au
 *	      rick@razorback.brisnet.org.au
 */

#ifdef MODULE
#include <linux/module.h>
#else	/* provide dummies */
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif
#include "scanner.h"
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/wait.h>
#include <linux/timer.h>
#include <linux/fcntl.h>

#include <asm/dma.h>
#include <asm/segment.h>

#define SCANNER_NAME    "GeniScan GS4500"

#ifndef SCANNER_MAJOR
#define	SCANNER_MAJOR	16	/* major device */
#endif

#define BUFFER_SIZE	1024	/* size of buffer for scanned data */

static struct {
	int data;
	int status;
	int control;
	int dmahold;
} gs_port[] = { {0x272, 0x273, 0x27a, 0x27b},
		{0x2e1, 0x2e2, 0x2e3, 0x2e4},
		{0x371, 0x372, 0x373, 0x374},
		{0x3e1, 0x3e2, 0x3e3, 0x3e4} };

static unsigned int gs_dma, gs_irq, gs_data, gs_status, gs_control, gs_dmahold;

static	int	isopen = 0;
static	int	resolution = -1;
static	int	dpi[] = { 100, 200, 300, 400 };
static	int	bytesperline[] = { 52, 104, 157, 206 };
static	int	gs_mode[] = { 0xF0, 0xA0, 0x80, 0xE0 };

static struct modeinfo_struct scan_info;

static struct scanner_capabilities scc_gs4500 = {

        SCC_HARDSWITCH,   		/* O_monochrome mode */
        SCC_HARDSWITCH,   		/* O_greyscale  mode */
        0,				/* O_true-color mode */

        SCC_HARDSWITCH | SCC_SOFTDETECT,/* Options for x resolution */
        SCC_HARDSWITCH | SCC_SOFTDETECT,/* Options for y resolution */

        0,                 		/* Options for brightness */
        {0,0},
        0,                              /* Options for contrast */
        {0,0},
        0,                              /* Options for hue */
        {0,0},
        0,                              /* Options for saturation */
        {0,0},

        105,                            /* Width in mm */
        0,                              /* Length in mm (unlimited) */
        0                               /* No options */
};

static struct scanner_type sctxt = { "Genius", "GS4500" };

static	char	*startbuf, *endbuf, *head, *tail;
static	int	buflen, linecnt;
static	int	scanning = 0, looping;
static	struct	wait_queue *gs4500_wait = NULL;

static void gs4500_timer(unsigned long ignored);

static struct timer_list gs4500_timer_list = {NULL, NULL, 0, 0, gs4500_timer };

static void
startscan(void)
{
	unsigned long flags;

	if (scanning)
		return;
	if (gs_dma) {
		save_flags(flags);      /* Save the current state here... */
		cli();
		disable_dma(gs_dma);
		clear_dma_ff(gs_dma);
		set_dma_mode(gs_dma, DMA_MODE_READ);
		set_dma_addr(gs_dma, (unsigned int)head);
		set_dma_count(gs_dma, scan_info.bpl);
		enable_dma(gs_dma);
		restore_flags(flags);  /* Use this instead of sti() */
	} else
		linecnt = 0;
	outb(0, gs_dmahold);
	scanning = 1;
}

static void linecomplete(void)
{
	int	count;

	if ((count = head - tail) < 0)
		count += buflen;
	if (buflen - count >= scan_info.bpl) {
		if ((head += scan_info.bpl) >= endbuf)
			head -= buflen;
	} else
		printk("GS4500: scan overrun\n");
	scanning = 0;
	startscan();
	wake_up_interruptible(&gs4500_wait);
}

static void gs4500_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	if (scanning) {
		if (gs_dma) {
			if ((inb(DMA1_STAT_REG) & (1 << gs_dma)))
				linecomplete();
		} else {
/*
 *	This is not right.  There should be a bit somewhere that indicates
 *	that there is data still to be read from gs_data, but I can't
 *	figure out what it is.  I've tried a number of solutions,
 *	but bytes are still falling down the cracks, so I leave it
 *	to somebody with more time/better documentation to fix.
 *	This only applies to an IRQ only setup.
 */
			head[linecnt++] = inb(gs_data);
			if (linecnt == scan_info.bpl)
				linecomplete();
		}
	}
}

static void gs4500_timer(unsigned long ignored)
{
	gs4500_interrupt(0, NULL, NULL);
	if (looping) {
		gs4500_timer_list.expires = 0;
		add_timer(&gs4500_timer_list);
	}
}

static void waitforscan(void)
{
	if (gs_irq)
		interruptible_sleep_on(&gs4500_wait);
	else {
		while (!(inb(DMA1_STAT_REG) & (1 << gs_dma)))
			if (current->signal & ~current->blocked)
				return;
			else
				schedule();
		linecomplete();
	}
}
 
static void
stopscan(void)
{
	unsigned long flags;

	if (scanning) {
		scanning = 0;
		if (gs_dma) {
			save_flags(flags);
			cli();
			disable_dma(gs_dma);
			clear_dma_ff(gs_dma);
			restore_flags(flags);
		}
		looping = 0;
		del_timer(&gs4500_timer_list);
	}
}

/* check for interface card and determin DMA and IO settings */
static int gs4500_detect(void) 
{
	int set = 3;
	int val;

	do {
		check_region(gs_port[set].data, 
			     (gs_port[set].dmahold-gs_port[set].data)+1);
		val = inb(gs_port[set].status);
		if(val != 0xff) break;
		set--;
	} while(set >= 0);
	if(set >= 0) {
		gs_irq = 0;
		gs_dma = 0;
		gs_data = gs_port[set].data;
		gs_status = gs_port[set].status;
		gs_control = gs_port[set].control;
		gs_dmahold = gs_port[set].dmahold;
		if (val & 0x80)
			printk("GS4500: Plug the scanner in!\n");
		switch (val & 0x0A) {
			case 0x00:	printk("GS4500: Both DMA1 and DMA3 jumpered!\n");
			case 0x0A:	break;
			case 0x08:	gs_dma = 1;	break;
			case 0x02:	gs_dma = 3;	break;
		}
		switch (val & 0x50) {
			case 0x00:	printk("GS4500: Both IRQ3 and IRQ5 jumpered!\n");
			case 0x50:	break;
			case 0x40:	gs_irq = 3;	break;
			case 0x10:	gs_irq = 5;	break;
		}
		if (!gs_dma && !gs_irq) {
			printk("GS4500: Need a DMA channel or an IRQ.\n");
			set = -1;
		}
	}
	return set;	/* -1 means no device found, otherwise index */
}

static int gs4500_resolution(void)
{
	int	loops, tries;
	int	cc = 0;

	outb(1, gs_control);
	outb(1, gs_dmahold);
	for (tries = 4; --tries; ) {
		/* wait until AR is 1 fail after MANY reads */
		for (loops = 100000; --loops && !(inb(gs_status) & 0x80); ) ;
		/* wait until AR is 0 fail after MANY reads */
		if (loops) {
			for (loops = 100000; --loops && ((cc = inb(gs_status)) & 0x80); ) ;
			if (cc == inb(gs_status))
				break;
		}
	}
	outb(0, gs_control);
	return (tries && loops) ? cc : -EIO;
}

static int gs4500_open(struct inode * inode, struct file * file)
{
	int	cc;

	if ((file->f_flags & O_ACCMODE) != O_RDONLY)
		return -EINVAL;
	if (isopen)
		return -EBUSY;
	isopen = 1;
	if ((cc = gs4500_resolution()) < 0) {
		isopen = 0;
		return cc;
	}
	switch (cc & 0xA4) {			/* determine dpi */
		case 0x24:	resolution = 0;
				break;
		case 0x20:	resolution = 1;
				break;
		case 0x04:	resolution = 2;
				break;
		case 0x00:	resolution = 3;
				break;
		default:	isopen = 0;
				return -EIO;
	}
	scan_info.xdpi = scan_info.ydpi = dpi[resolution];
	scan_info.sc_mode = SCM_MONO;
	scan_info.depth = 1;
	scan_info.bpl = bytesperline[resolution];
	scan_info.left = scan_info.right = 0;
	scan_info.top = scan_info.bottom = 0;
	scan_info.bright = scan_info.contrast = 0;
	scan_info.hue = scan_info.sat = 0;
	scan_info.options = 0;
	head = tail = startbuf;
	buflen = ((BUFFER_SIZE / scan_info.bpl) * scan_info.bpl);
	endbuf = startbuf + buflen;
	MOD_INC_USE_COUNT;
	inb(gs_data);
	outb(gs_mode[resolution]|1, gs_control);/* turn scanner on */
	outb(1, gs_dmahold);			/* disable DMA */
	return 0; 
}

static void gs4500_release(struct inode * inode, struct file * file)
{
	if (isopen) {
		outb(0, gs_control);
		stopscan();
		isopen = 0;
		MOD_DEC_USE_COUNT;
	}
}

/*
 *	gs4500_read() and gs4500_ioctl() assume that they are only
 *	called after a successful gs4500_open().
 */

static int gs4500_read(struct inode * inode, struct file * file, char *buf, int size)
{
	int scanned = 0;
	int chars, scancount;

	looping = 0;
	del_timer(&gs4500_timer_list);
	while (size) {
		if ((scancount = head - tail) < 0)
			scancount += buflen;
		if (file->f_flags & O_NONBLOCK) {
			if (!scancount)
				return -EAGAIN;
			if (size > scancount)
				size = scancount;
		}
		while (!scancount) {
			startscan();
			waitforscan();
			if (current->signal & ~current->blocked)
				return scanned?scanned:-EIO;
			if ((scancount = head - tail) < 0)
				scancount += buflen;
		}
		if (tail + scancount > endbuf)
			scancount = endbuf - tail;
		chars = scancount < size ? scancount : size;
		memcpy_tofs(buf, tail, chars);
		if ((tail += chars) >= endbuf)
			tail -= buflen;
		buf += chars;
		scanned += chars;
		size -= chars;
	}
	return scanned;
}

static int gs4500_select(struct inode * inode, struct file * file, int type, select_table *table)
{
	if (type == SEL_IN) {
		if (head != tail)
			return 1;
		startscan();
		if (!gs_irq && !looping) {
			looping = 1;
			gs4500_timer_list.expires = 1;
			add_timer(&gs4500_timer_list);
		}
		if (table)
			select_wait(&gs4500_wait, table);
	}
	return 0;
}

static int gs4500_ioctl(struct inode * inode, struct file * file, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	case HSIOCGOPT: break;	/* no options supported */
	case HSIOCGSCC: if (arg)
			  memcpy_tofs((char *)arg, (char *)&scc_gs4500,
			              sizeof(struct scanner_capabilities));
			break;
	case HSIOCGSCT: if (arg)
			  memcpy_tofs((char *)arg, (char *)&sctxt,
			              sizeof(struct scanner_type));
			break;
	case HSIOCGMOD: if (arg)
			  memcpy_tofs((char *)arg, (char *)&scan_info,
			              sizeof(struct modeinfo_struct));
	                break;
	case HSIOCGBUF: break;	/* change of buffersize not supported */
	case HSIOCSBUF: if (arg)
			  put_fs_long((unsigned long) 1, arg);
			break;
	case HSIOCGSIB: if (arg)
			  put_fs_long((unsigned long) 0, arg);
			break;
	default:	return -EINVAL;
			break;
	}

	return 0;
}

static struct file_operations gs4500_fops = {
        NULL,		/* lseek */
	gs4500_read,	/* read */
	NULL,		/* write */
	NULL,		/* readdir */
	gs4500_select,	/* select */
	gs4500_ioctl,	/* ioctl */
	NULL,		/* mmap */
        gs4500_open,	/* open */
        gs4500_release,	/* release */
	NULL		/* fsync */
};

static struct file_operations *gs4500_init(void)
{
	if(gs4500_detect() != -1) {
		if (gs_irq && 
                    request_irq(gs_irq, gs4500_interrupt, SA_INTERRUPT, "gs4500", NULL) != 0) {
			printk("GS4500: unable to get IRQ %d\n", gs_irq);
			gs_irq = 0;
		}
		if (gs_dma && request_dma(gs_dma, "gs4500") != 0) {
			printk("GS4500: unable to get DMA %d\n", gs_dma);
			gs_dma = 0;
		}
		if (gs_dma || gs_irq) {
			printk("GS4500: ");
			if (gs_dma)
				printk("DMA %d ", gs_dma);
			if (gs_irq)
				printk("IRQ %d ", gs_irq);
			printk("at 0x%04x - 0x%04x\n", gs_data, gs_dmahold);
			return &gs4500_fops;
		}
		printk("GS4500: no way to get data from scanner.\n");
	}
	else printk("GS4500: no usable scanner found.\n");
	return NULL;
}

#ifdef MODULE

int init_module(void)
{
	printk("GeniScan GS4500 handheld scanner driver, version 1.6\n");
  	if(register_chrdev(SCANNER_MAJOR, SCANNER_NAME, gs4500_init())) {
    		printk("GS4500: cannot register major number %d\n", SCANNER_MAJOR);
    		return -EIO;
  	}
	request_region(gs_data, (gs_dmahold-gs_data)+1, "gs4500");
	startbuf = (char *)kmalloc(BUFFER_SIZE, GFP_KERNEL | GFP_DMA);
	if (startbuf == NULL) {
		printk("GS4500: cannot allocate buffer");
		return -EIO;
	}
  	return 0;
}

void cleanup_module(void)
{
  	printk("Removing GeniScan GS4500 driver from memory\n");
  	if(MOD_IN_USE)
    		printk("GS4500: busy, remove delayed\n");
        else {
  		unregister_chrdev(SCANNER_MAJOR, SCANNER_NAME);
		release_region(gs_data, (gs_dmahold-gs_data)+1);
		kfree_s(startbuf, BUFFER_SIZE);
		if (gs_dma)
			free_dma(gs_dma);  /* release DMA channel */
		if (gs_irq)
			free_irq(gs_irq, NULL);
	}
}

#else	/* part of the kernel */

unsigned long scanner_init(unsigned long kmem_start)
{
	struct file_operations *f;

	if ((f = gs4500_init()) != NULL) {
		if (register_chrdev(SCANNER_MAJOR, SCANNER_NAME, f))
    			printk("GS4500: cannot register major number %d\n", SCANNER_MAJOR);
	}
	request_region(gs_data, (gs_dmahold-gs_data)+1, "gs4500");
	scanline = (char *)kmem_start;
	return kmem_start += BUFFER_SIZE;
}
#endif
