/*
 * mda.c
 *
 * Kernel module to drive MDA nonitor
 *
 * v0.7 1995 Patrick J Caulfield
 *
 * Revision History:
 *
 * v0.1   3-Jun-1995  PJC  First implementation as a kernel module
 * v0.2   4-Jun-1995  PJC  Allowed multiple opens to be active
 * v0.3   4-Jun-1995  PJC  Added hardware scrolling support
 * v0.4   6-Jun-1995  PJC  Fixed odd scrolling bug.
 * v0.5   9-Jul-1995  PJC  Added DEL/BS support
 * v0.6  18-Jul-1995  PJC  Now include autoconf.h to get CONFIG_MODVERSIONS
 *                         Started vt100 support. 
 * v0.7  21-Jul-1995  PJC  Added card detection and reverse video in place of
 *                         flashing.
 * v0.8  25-Apr-1996  PJC  Added support for dynamic major number allocation
 * v0.9  22-Jun-1996  PJC  Fixed boldface bug.
 *
 ******************************************************************************

    (c) 1995 P.J. Caulfield     patrick@pandh.demon.co.uk
                                pcaulfield@cix.compulink.co.uk
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 ******************************************************************************
 */

/* Linux module includes */
#include <linux/module.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#ifdef HAVE_AUTOCONF
#include <linux/autoconf.h>  /* for CONFIG_MODVERSIONS */
#endif
#include <asm/io.h>
#include "mda.h"

/* Our name in the kernel */
#define MODULE_NAME "mono"

/* Make sure we have a major device number to use */
#ifndef MONO_MAJOR
#error No Major number specified
#endif

/* Function prototypes */
static void   set_cursor(unsigned short);
static int    scroll_screen(void);
static void   clear_blink_flag(void);
static inline void set_origin(unsigned short);


/* Globals */
static unsigned short offset = 0; /* Current offset into video memory */
static unsigned short base   = 0; /* Current location of top of display */
static unsigned short major  = 0; /* Major device number assigned */

/* Include the ANSI/vt100 emulation module */
#include "vt100.c"

/*
 * Clearing the blink flag in the control register makes the BLINK
 * attribute show reverse instead.
 * I *much* prefer this. If you don't then define DO_BLINK in mda.h
 */
static void clear_blink_flag()
{
    outb(0x09, video_mode_reg);
}

/*
 * Write a single character to the screen.
 */
static void mono_write_char(char outchar)
{
    vt100_write(outchar);
}

/*
 * Scroll up by one line
 */
static int scroll_screen()
{
    int i;
    char* videomem = (char*)VIDEOMEM_START;

    offset = VIDEOSCREEN_SIZE - VIDEO_LINELENGTH; /* Stay on the bottom line */
    base  += VIDEO_LINELENGTH;

/* Reached the end of video memory? -- go back to the beginning and start again */	
    if (base+VIDEOSCREEN_SIZE+VIDEOMEM_START >= VIDEOMEM_END)
    {
	memcpy((char*)(videomem), (char*)(videomem+base), VIDEOSCREEN_SIZE-VIDEO_LINELENGTH);
	base = 0;
    }
    
    /* Clear the bottom line */
    for (i=VIDEOSCREEN_SIZE-VIDEO_LINELENGTH; i<VIDEOSCREEN_SIZE; i++,i++)
    {
	videomem[base+i] = ' ';
	videomem[base+i+1] = ATTR_NORMAL;
    }
    
#ifdef DEBUG
    printk("mda: video base address is: %d\n", base);
#endif
    set_origin(base);

    return offset;
}

/*
 * Move the offset into video memory -- Hardware scrolling
 */
static inline void set_origin(unsigned short offset)
{

    base = offset;
    cli();

    outb(13, video_port_reg);
    outb(offset >> 1, video_port_val);
    outb(12, video_port_reg);
    outb(offset >> 9, video_port_val);
}


/*
 * Position the cursor
 */
void set_cursor(unsigned short pos)
{
    int video_mem_base = VIDEOMEM_START;

    cli();
    outb(14, video_port_reg);
    outb(0xff&((pos-video_mem_base+base)>>9), video_port_val); 
    outb(15, video_port_reg);
    outb(0xff&((pos-video_mem_base+base)>>1), video_port_val);
}

/*
 * Detect card.
 * Returns 0, for no MDA, 1 for MDA present...more???
 */
static int detect_card()
{
    unsigned char saved_pos, inbyte;
    unsigned char arbit_val = 0x66;
    
    outb(15, video_port_reg); /* Select cursor low register */
    saved_pos = inb(video_port_val);

/* Write an arbitrary value to the cursor register & read it back */
    outb(arbit_val, video_port_val);

    udelay(10); /* Wait a bit */
    
    inbyte = inb(video_port_val);
    if (arbit_val == inbyte)
    {
	return 1;
    }
    else
    {
	return 0;
    }
}

/*------------------------- Basic file operations ---------------------------*/

/*
 * Write a string to the screen
 */
static int mono_write(struct inode * inode, struct file * file, const char * buf, int count)
{
    int        i;
    const char *temp = buf;
    char       c;
    
    for (i=0; i<count; i++)
    {
	c = get_fs_byte(temp+i);
	mono_write_char(c);
    }
    return count;
}

/*
 * lseek?? you don't want to do that!
 */
static int mono_lseek(struct inode * inode, struct file * file,
         	      off_t offset, int origin)
{
	return -ESPIPE;
}

/*
 * Open -- no sharing check otherwise we can't redirect
 * both stderr and stdout to the mono monitor.
 */
static int mono_open(struct inode * inode, struct file * file)
{
    MOD_INC_USE_COUNT;
    return 0;
}

/*
 * Close
 */
static void mono_release(struct inode * inode, struct file * file)
{
    MOD_DEC_USE_COUNT;
}
    
/* ------------------------- Module support stuff -------------------------- */

/* File operations to register with the kernel */

static struct file_operations mono_fops = {
	mono_lseek,
	NULL,		/* _read    */
	mono_write,
	NULL,		/* _readdir */
	NULL,		/* _select  */
	NULL,           /* _ioctl   */
	NULL,		/* _mmap    */
	mono_open,
	mono_release
};
char kernel_version[]= UTS_RELEASE;

/*
 * Called at insmod time
 */
int init_module(void)
{
    
    /* Is there a card in the system ? */
    if (!detect_card())
    {
	printk("mda: unable to find mono adapter card\n");
	return -EIO;
    }
    

#if MONO_MAJOR == DYNAMIC
    major = register_chrdev(0, MODULE_NAME, &mono_fops);
    if (major <= 0)
    {
	printk("mda: unable to get a major number for mono monitor, retcode = %d\n", major);
	return -EIO;
    }
#else
    major = register_chrdev(MONO_MAJOR, MODULE_NAME, &mono_fops);
    if (major != 0)
    {
	printk("mda: unable to register driver for mono monitor, retcode = %d\n", major);
	return -EIO;
    }
#endif
    
#ifndef DO_BLINK
    clear_blink_flag();
#endif
    reset_terminal();
    return 0;
}

/*
 * Called at rmmod time
 */
void cleanup_module(void)
{
    if (MOD_IN_USE)
	printk("mda: busy - remove delayed\n");
    else
#if MONO_MAJOR == DYNAMIC
	unregister_chrdev(major, MODULE_NAME);
#else
        unregister_chrdev(MONO_MAJOR, MODULE_NAME);
#endif    
}
