/* $Id: isdn.c,v 1.18 1996/01/05 06:23:59 fritz Exp fritz $
 *
 * ISDN driver for Linux. (linklevel-module)
 *
 * Copyright 1994,95 by Fritz Elfert (fritz@wuemaus.franken.de)
 * Copyright 1995 Thinking Objects Software GmbH Wuerzburg
 *
 * This file is part of Isdn4Linux.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Log: isdn.c,v $
 * Revision 1.18  1996/01/05 06:23:59  fritz
 * Fixed Bug in isdn_net_find_icall regarding exclusive bound interfaces.
 * Changed load-balancing-strategy for better performance with more
 * than one slave.
 * Added protocol-type to CONNECT-message.
 *
 * Revision 1.17  1996/01/04 02:54:56  fritz
 * Changed copying policy to GPL
 * Added channel-bundling for net interfaces.
 * Added NET_DIAL and NET_ADDSLAVE ioctl's.
 * Added NET_GETMAPPING and NET_SETMAPPING ioctl's.
 *
 * Revision 1.16  1995/12/18  18:01:48  fritz
 * Bugfix: CD-bit was not reset on local hangup.
 * New: AT&X AT&W AT&F, Registers 17-19
 *
 * Revision 1.15  1995/12/07  00:10:00  fritz
 * Support for ICN 4B
 * Bugfix: Local hangup on ttyIx did not reset MSR Carrier detect
 *
 * Revision 1.14  1995/11/16  21:28:27  fritz
 * Changed /dev/isdninfo-Handling
 *
 * Revision 1.13  1995/10/29  21:37:19  fritz
 * Added support for select/wait for /dev/isdninfo.
 * Bugfixes in statemachine for callbacks
 *
 * Revision 1.12  1995/07/15  20:34:48  fritz
 * Added support fo leased lines 64S/U (1 B-Channel, no D-Channel)
 * Bugfix: Phone-number misplaced for /dev/isdninfo when using /dev/ttyI*
 * Added support for pre-binding and interface to a channel.
 *
 * Revision 1.11  1995/05/04  12:28:31  fritz
 * Special Dump-Feature for logging packets, which trigger dialing.
 * Debug-Version for fido.
 *
 * Revision 1.10  1995/04/29  13:10:17  fritz
 * Added support for maximum buffersize settable by HW-drivers.
 * Changed some dial-code for running correctly with the teles-module.
 * Bugfix: If remote-station is doing callback, sometimes the incoming
 * callback had been lost.
 * Added user-settable verbose-flag for incoming-call-logging.
 *
 * Revision 1.9  1995/04/23  13:35:52  fritz
 * Changed EAZ/MSN-Handling comletely to support real MSN's.
 * Added a second phone-list for net-interfaces to distinguish between
 * incoming and outgoing numbers.
 * Changed timer-code
 *
 * Revision 1.8  1995/03/25  23:25:51  fritz
 * readstat and writecmd will be called only if not NULL.
 * removed unneeded ISDN_IOCTL_NET_LISTIF
 * replaced internal modem_wait_until_sent() by tty_wait_until_sent(),
 * exported by new kernel-version.
 * added calls to tty_check_change()
 * some bugfixes in tty-close-routines.
 * Added new feature for disabling hangup on incoming net-calls.
 * Added patternmatching-routines for incoming-call-security.
 * Bugfix in initialization of isdn_net_find_icall().
 * Bugfix in on_hook().
 *
 * Revision 1.7  1995/03/15  12:39:32  fritz
 * Added isdn_net_getphones for displaying phones with isdnctrl
 * Added 'CISCO-Hack' by tsbogend@bigbug.franken.de
 * Added support for SPV's
 * Added new encapsulation 'IP-with-typefield' by tsbogend@bigbug.franken.de
 * Bugfix: Active B-Channel-Setup by Remotestation was not recognized
 * Bugfix: rcvcount had not beed initialized in register_isdn()
 *
 * Revision 1.6  1995/02/20  03:42:02  fritz
 * Lot of bugfixes in tty-code
 * Changed tty-code to searching a free channel when dialing and
 * not while opening the tty-device.
 * Added info-device.
 *
 * Revision 1.5  1995/02/01  10:45:38  fritz
 * Added escape-sequence-handling to AT-emulator, RING-message-timer
 * (RING's now repeat until ATA or remote-hangup), Modem-dialout
 * now aborts when a character is sent during dial-phase.
 * Introduced defines for timer-flags. Added more comments.
 *
 * Revision 1.4  1995/01/31  15:26:44  fritz
 * Added printk of Cause-Messages, Revision-output on load.
 * Changed net-chargeinfo routine to always calculate charging-interval
 * because the first interval is mostly longer than the following ones.
 *
 * Revision 1.3  1995/01/29  23:25:11  fritz
 * Rearranged timer-code to minimize usage of timers.
 * Bugfix in network-receiving.
 * Added feature-flags for lowlevel-drivers.
 *
 * Revision 1.2  1995/01/09  07:34:30  fritz
 * Added Loglevels to all printk-calls
 *
 * Revision 1.1  1994/12/14  18:06:12  fritz
 * Initial revision
 *
 */

#include <isdn.h>

/* Debugflags */
#undef  ISDN_DEBUG_STATCALLB
#undef  ISDN_DEBUG_MODEM_OPEN
#undef  ISDN_DEBUG_MODEM_IOCTL
#undef  ISDN_DEBUG_MODEM_WAITSENT
#undef  ISDN_DEBUG_MODEM_HUP
#undef  ISDN_DEBUG_MODEM_SENDOPT
#undef  ISDN_DEBUG_MODEM_ICALL
#undef  ISDN_DEBUG_MODEM_DUMP
#undef  ISDN_DEBUG_NET_DUMP
#undef  ISDN_DEBUG_NET_DIAL
#undef  ISDN_DEBUG_NET_BUILDHDR
#undef  ISDN_DEBUG_AT
#undef  ISDN_DEBUG_NET_ICALL

/* Option: Show packets, which trigger dialing. */
#define ISDN_LOG_PACKET

/* Option: Synchronous PPP */
#undef ISYNCPPP

/* Leave this unchanged unless you know what you do! */
#define MODEM_PARANOIA_CHECK
#define MODEM_DO_RESTART

/* Prototypes */
#ifdef ISYNCPPP
static int   free_ippp(isdn_net_local *);
static int   bind_ippp(isdn_net_local *);
static int   ippp_read(int, struct file *, char *, int);
static int   ippp_write(int, struct file *, char *, int);
static int   ippp_select(int, struct file *, int, select_table *);
static int   ippp_ioctl(int, struct file *, unsigned int, unsigned long);
static int   ippp_open(int, struct file *);
static void  ippp_release(int, struct file *);
static int   ippp_state(int minor);
static int   ippp_fill_rq(char *buf,int len,int minor);
static int   init_ippp(void);
static int   ippp_hangup(int);
static void  cleanup_ippp(void);
#endif
static int   modem_parse(const char *, int, modem_info *, int);
static void  modem_chk3plus(const u_char *, u_char, int, int *, int *, int);
static void  modem_result(int, modem_info *);
static void  modem_reset_regs(atemu *, int);
static void  modem_plus(void);
static void  modem_ring(void);
static void  modem_xmit(void);
static void  modem_hup(modem_info *);
static void  readmodem(void);
static int   try_readmodem(int, u_char *, int);
static int   isdn_modem_find_icall(int, int, char *);
#if FUTURE
static void  isdn_modem_bsent(int, int);
#endif
static char* isdn_net_new(char *, struct device *);
static char* isdn_net_newslave(char *);
static int   isdn_net_rm(char *);
static void  isdn_net_stat_callback(int, int, int);
static int   isdn_net_receive_callback(int, int, u_char *, int);
static int   isdn_net_setcfg(isdn_net_ioctl_cfg *);
static int   isdn_net_getcfg(isdn_net_ioctl_cfg *);
static int   isdn_net_addphone(isdn_net_ioctl_phone *);
static int   isdn_net_getphones(isdn_net_ioctl_phone *, char *);
static int   isdn_net_delphone(isdn_net_ioctl_phone *);
static int   isdn_net_find_icall(int, int, int, char *);
static void  isdn_net_hangup(struct device *);
static void  isdn_net_dial(void);
static void  isdn_net_autohup(void);
static int   isdn_net_force_hangup(char *);
static int   isdn_net_force_dial(char *);
static int   isdn_net_force_dial_lp(isdn_net_local *);
static int   isdn_get_allcfg(char *);
static int   isdn_set_allcfg(char *);
static void  isdn_info_update();
static int   wildmat(char *s, char *p);
static int   getnum(char **);
static int   my_atoi(char *s);

static char
*revision = "$Revision: 1.18 $";

#if defined(ISDN_DEBUG_NET_DUMP) || defined(ISDN_DEBUG_MODEM_DUMP)
static void
isdn_dumppkt (char *s, u_char *p, int len, int dumplen) {
  int dumpc;

  printk(KERN_DEBUG "%s(%d) ",s,len);
  for (dumpc=0;(dumpc<dumplen)&&(len);len--,dumpc++)
    printk(" %02x",*p++);
  printk("\n");
}
#endif

/* Try to allocate a new buffer, link it into queue. */
static  u_char*
new_buf(pqueue **queue, int length) {
  pqueue *p;
  pqueue *q;

  if ((p = *queue)) {
    while (p) {
      q = p;
      p = (pqueue*)p->next;
    }
    p = (pqueue*)kmalloc(sizeof(pqueue)+length,GFP_ATOMIC);
    q->next = (u_char*)p;
  } else
    p = *queue = (pqueue*)kmalloc(sizeof(pqueue)+length,GFP_ATOMIC);
  if (p) {
    p->size = sizeof(pqueue)+length;
    p->length = length;
    p->next = NULL;
    p->rptr = p->buffer;
    return p->buffer;
  } else {
    return (u_char *)NULL;
  }
}

static void
free_queue(pqueue **queue) {
  pqueue *p;
  pqueue *q;

  p = *queue;
  while (p) {
    q = p;
    p = (pqueue*) p->next;
    kfree_s(q,q->size);
  }
  *queue = (pqueue *)0;
}

static int
dc2minor(int di, int ch) {
  int i;
  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (dev->chanmap[i]==ch && dev->drvmap[i]==di)
      return i;
  return -1;
}

static int isdn_timer_cnt1 = 0;
static int isdn_timer_cnt2 = 0;

static void
isdn_timer_funct(ulong dummy) {
  int tf = dev->tflags;
  int flags;

  if (tf & ISDN_TIMER_FAST) {
    if (tf & ISDN_TIMER_MODEMREAD)
      readmodem();
    if (tf & ISDN_TIMER_MODEMPLUS)
      modem_plus();
    if (tf & ISDN_TIMER_MODEMXMIT)
      modem_xmit();
  }
  if (tf & ISDN_TIMER_SLOW)
    if (++isdn_timer_cnt1 > ISDN_TIMER_02SEC) {
      isdn_timer_cnt1 = 0;
      if (tf & ISDN_TIMER_NETDIAL)
	isdn_net_dial();
    }
    if (++isdn_timer_cnt2 > ISDN_TIMER_1SEC) {
      isdn_timer_cnt2 = 0;
      if (tf & ISDN_TIMER_NETHANGUP)
	isdn_net_autohup();
      if (tf & ISDN_TIMER_MODEMRING)
	modem_ring();
    }
  if (tf) {
    save_flags(flags);
    cli();
    del_timer(&dev->timer);
    dev->timer.function = isdn_timer_funct;
    dev->timer.expires  = RUN_AT(ISDN_TIMER_RES);
    add_timer(&dev->timer);
    restore_flags(flags);
  }
}

static void
isdn_timer_ctrl(int tf, int onoff) {
  int flags;

  save_flags(flags);
  cli();
  if ((tf & ISDN_TIMER_SLOW) && (!(dev->tflags & ISDN_TIMER_SLOW))) {
    /* If the slow-timer wasn't activated until now */
    isdn_timer_cnt1 = 0;
    isdn_timer_cnt2 = 0;
  }
  if (onoff)
    dev->tflags |= tf;
  else
    dev->tflags &= ~tf;
  if (dev->tflags) {
    del_timer(&dev->timer);
    dev->timer.function = isdn_timer_funct;
    dev->timer.expires  = RUN_AT(ISDN_TIMER_RES);
    add_timer(&dev->timer);
  }
  restore_flags(flags);
}

/* Receive a packet from B-Channel. (Called from low-level-module)
 * Parameters:
 *
 * di      = Driver-Index.
 * channel = Number of B-Channel (0...)
 * buf     = pointer to packet-data
 * len     = Length of packet-data
 *
 */
static void
receive_callback(int di, int channel, u_char *buf, int len) {
  ulong flags;
  char *p;
  int  i;
  int midx;

  if (dev->global_flags & ISDN_GLOBAL_STOPPED)
    return;
  /* First, try to deliver data to network-device */
  if (isdn_net_receive_callback(di,channel,buf,len))
    return;
  /* No network-device found, deliver to tty or raw-channel */
  if (len) {
    save_flags(flags);
    cli();
    if ((i = dc2minor(di,channel))>=0) {
      midx = dev->m_idx[i];
      if (dev->mdm.atmodem[midx].mdmreg[13] & 2)
	/* T.70 decoding: Simply throw away the T.70 header (4 bytes) */
	if ((buf[0] == 1) && ((buf[1] == 0) || (buf[1] == 1))) {
#ifdef ISDN_DEBUG_MODEM_DUMP
	  isdn_dumppkt("T70strip1:",buf,len,len);
#endif
	  buf += 4;
	  len -= 4;
#ifdef ISDN_DEBUG_MODEM_DUMP
	  isdn_dumppkt("T70strip2:",buf,len,len);
#endif
	}
      /* Try to deliver directly via tty-flip-buf if queue is empty */
      if (!dev->drv[di]->rpqueue[channel])
	if (try_readmodem(midx,buf,len)) {
	  restore_flags(flags);
	  return;
	}
      /* Direct deliver failed or queue wasn't empty.
       * Queue up for later dequeueing via timer-irq.
       */
      p = new_buf(&dev->drv[di]->rpqueue[channel],len);
      if (!p) {
	printk(KERN_WARNING "isdn: malloc of rcvbuf failed, dropping.\n");
	dev->drv[di]->rcverr[channel]++;
	restore_flags(flags);
	return;
      } else {
	memcpy(p,buf,len);
	dev->drv[di]->rcvcount[channel] += len;
      }
      /* Schedule dequeuing */
      if ((dev->modempoll) && (midx>=0)) {
	if (dev->mdm.rcvsched[midx])
	  isdn_timer_ctrl(ISDN_TIMER_MODEMREAD,1);
      }
      wake_up_interruptible(&dev->drv[di]->rcv_waitq[channel]);
    }
    restore_flags(flags);
  }
}

static void
isdn_all_eaz(int di, int ch) {
  isdn_ctrl cmd;

  cmd.driver  = di;
  cmd.arg     = ch;
  cmd.command = ISDN_CMD_SETEAZ;
  cmd.num[0] = '\0';
  (void)dev->drv[di]->interface->command(&cmd);
}

static int
status_callback(isdn_ctrl *c) {
  int di;
  int mi;
  ulong flags;
  int i;
  isdn_ctrl cmd;

  di = c->driver;
  switch (c->command) {
    case ISDN_STAT_STAVAIL:
      save_flags(flags);
      cli();
      dev->drv[di]->stavail += c->arg;
      restore_flags(flags);
      wake_up_interruptible(&dev->drv[di]->st_waitq);
      break;
    case ISDN_STAT_BSENT:
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      isdn_net_stat_callback(di,c->arg,c->command);
#if FUTURE
      isdn_modem_bsent(di,c->arg);
#endif
      wake_up_interruptible(&dev->drv[di]->snd_waitq[c->arg]);
      break;
    case ISDN_STAT_RUN:
      dev->drv[di]->running = 1;
      for (i=0;i<ISDN_MAX_CHANNELS;i++)
	if (dev->drvmap[i] == di)
	  isdn_all_eaz(di,dev->chanmap[i]);
      break;
    case ISDN_STAT_STOP:
      dev->drv[di]->running = 0;
      break;
    case ISDN_STAT_ICALL:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "ICALL (net): %d %ld %s\n",di,c->arg,c->num);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED) {
	cmd.driver  = di;
	cmd.arg     = c->arg;
	cmd.command = ISDN_CMD_HANGUP;
	dev->drv[di]->interface->command(&cmd);
	return 0;
      }
      if ((i = dc2minor(di,c->arg))>=0) {
	int r;

     	/* Try to find a network-interface which will accept incoming call */
	r = isdn_net_find_icall(di,c->arg,i,c->num);
	switch (r) {
	  case 0:
	    /* No network-device replies. Schedule RING-message to
	     * tty and set RI-bit of modem-status.
	     */
	    if ((mi = isdn_modem_find_icall(di,c->arg,c->num))>=0) {
	      dev->mdm.msr[mi] |= UART_MSR_RI;
	      modem_result(2,&dev->mdm.info[mi]);
	      isdn_timer_ctrl(ISDN_TIMER_MODEMRING,1);
	    } else
	      if (dev->drv[di]->reject_bus) {
		cmd.driver  = di;
		cmd.arg     = c->arg;
		cmd.command = ISDN_CMD_HANGUP;
		dev->drv[di]->interface->command(&cmd);
	      }
	    break;
	  case 1:
	    /* Schedule connection-setup */
	    isdn_net_dial();
	    cmd.driver  = di;
	    cmd.arg     = c->arg;
	    cmd.command = ISDN_CMD_ACCEPTD;
	    dev->drv[di]->interface->command(&cmd);
	    break;
	  case 2:   /* For calling back, first reject incoming call ... */
	  case 3:   /* Interface found, but down, reject call actively  */
	    cmd.driver  = di;
	    cmd.arg     = c->arg;
	    cmd.command = ISDN_CMD_HANGUP;
	    dev->drv[di]->interface->command(&cmd);
	    if (r==2)
	      /* ... then start dialing. */
	      isdn_net_dial();
	    break;
	}
      }
      return 0;
      break;
    case ISDN_STAT_CINF:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "CINF: %ld %s\n",c->arg,c->num);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      if (strcmp(c->num,"0"))
	isdn_net_stat_callback(di,c->arg,c->command);
      break;
    case ISDN_STAT_CAUSE:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "CAUSE: %ld %s\n",c->arg,c->num);
#endif
      printk(KERN_INFO "isdn: cause: %s\n",c->num);
      break;
    case ISDN_STAT_DCONN:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "DCONN: %ld\n",c->arg);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      /* Find any network-device, waiting for D-channel setup */
      isdn_net_stat_callback(di,c->arg,c->command);
      if ((i = dc2minor(di,c->arg))>=0)
	if ((mi=dev->m_idx[i])>=0)
	  /* If any tty has just dialed-out, setup B-Channel */
	  if (dev->mdm.info[mi].flags &
	      (ISDN_ASYNC_NORMAL_ACTIVE | ISDN_ASYNC_CALLOUT_ACTIVE)) {
	    if (dev->mdm.dialing[mi]==1) {
	      dev->mdm.dialing[mi] = 2;
	      cmd.driver  = di;
	      cmd.arg     = c->arg;
	      cmd.command = ISDN_CMD_ACCEPTB;
	      dev->drv[di]->interface->command(&cmd);
	      return 0;
	    }
	  }
      break;
    case ISDN_STAT_DHUP:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "DHUP: %ld\n",c->arg);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      dev->drv[di]->flags &= ~(1<<(c->arg));
      isdn_info_update();
      /* Signal hangup to network-devices */
      isdn_net_stat_callback(di,c->arg,c->command);
      if ((i = dc2minor(di,c->arg))>=0) {
	if ((mi=dev->m_idx[i])>=0) {
	  /* Signal hangup to tty-device */
	  if (dev->mdm.info[mi].flags &
	      (ISDN_ASYNC_NORMAL_ACTIVE | ISDN_ASYNC_CALLOUT_ACTIVE)) {
	    if (dev->mdm.dialing[mi]==1) {
	      dev->mdm.dialing[mi] = 0;
	      modem_result(7,&dev->mdm.info[mi]);
	    }
	    if (dev->mdm.online[mi])
	      modem_result(3,&dev->mdm.info[mi]);
#ifdef ISDN_DEBUG_MODEM_HUP
	    printk(KERN_DEBUG "Mhup in ISDN_STAT_DHUP\n");
#endif
	    modem_hup(&dev->mdm.info[mi]);
	    dev->mdm.msr[mi] &= ~(UART_MSR_DCD | UART_MSR_RI);
	    return 0;
	  }
	}
      }
      break;
    case ISDN_STAT_BCONN:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "BCONN: %ld\n",c->arg);
#endif
      /* Signal B-channel-connect to network-devices */
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      isdn_net_stat_callback(di,c->arg,c->command);
      dev->drv[di]->flags |= (1<<(c->arg));
      isdn_info_update();
      if ((i = dc2minor(di,c->arg))>=0) {
	if ((mi=dev->m_idx[i])>=0) {
	  /* Schedule CONNECT-Message to any tty, waiting for it and
	   * set DCD-bit of it's modem-status.
	   */
	  if (dev->mdm.info[mi].flags &
	      (ISDN_ASYNC_NORMAL_ACTIVE | ISDN_ASYNC_CALLOUT_ACTIVE)) {
	    dev->mdm.msr[mi] |= UART_MSR_DCD;
	    if (dev->mdm.dialing[mi])
	      dev->mdm.dialing[mi] = 0;
	    dev->mdm.rcvsched[mi] = 1;
	    modem_result(5,&dev->mdm.info[mi]);
	  }
	}
      }
      break;
    case ISDN_STAT_BHUP:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "BHUP: %ld\n",c->arg);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      dev->drv[di]->flags &= ~(1<<(c->arg));
      isdn_info_update();
      if ((i = dc2minor(di,c->arg))>=0) {
	if ((mi=dev->m_idx[i])>=0) {
	  /* Signal hangup to tty-device, schedule NO CARRIER-message */
	  if (dev->mdm.info[mi].flags &
	      (ISDN_ASYNC_NORMAL_ACTIVE | ISDN_ASYNC_CALLOUT_ACTIVE)) {
	    dev->mdm.msr[mi] &= ~(UART_MSR_DCD | UART_MSR_RI);
	    if (dev->mdm.online[mi])
	      modem_result(3,&dev->mdm.info[mi]);
#ifdef ISDN_DEBUG_MODEM_HUP
	    printk(KERN_DEBUG "Mhup in ISDN_STAT_BHUP\n");
#endif
	    modem_hup(&dev->mdm.info[mi]);
	  }
	}
      }
      break;
    case ISDN_STAT_NODCH:
#ifdef ISDN_DEBUG_STATCALLB
      printk(KERN_DEBUG "NODCH: %ld\n",c->arg);
#endif
      if (dev->global_flags & ISDN_GLOBAL_STOPPED)
	return 0;
      isdn_net_stat_callback(di,c->arg,c->command);
      if ((i = dc2minor(di,c->arg))>=0) {
	if ((mi=dev->m_idx[i])>=0) {
	  if (dev->mdm.info[mi].flags &
	      (ISDN_ASYNC_NORMAL_ACTIVE | ISDN_ASYNC_CALLOUT_ACTIVE)) {
	    if (dev->mdm.dialing[mi]) {
	      dev->mdm.dialing[mi] = 0;
	      modem_result(6,&dev->mdm.info[mi]);
	    }
	    dev->mdm.msr[mi] &= ~UART_MSR_DCD;
	    if (dev->mdm.online[mi]) {
	      modem_result(3,&dev->mdm.info[mi]);
	      dev->mdm.online[mi] = 0;
	    }
	  }
	}
      }
      break;
    case ISDN_STAT_ADDCH:
      break;
    case ISDN_STAT_UNLOAD:
      save_flags(flags);
      cli();
      for (i=0;i<ISDN_MAX_CHANNELS;i++)
	if (dev->drvmap[i]==di) {
	  dev->drvmap[i]=-1;
	  dev->chanmap[i]=-1;
	  dev->mdm.info[i].isdn_driver = -1;
	  dev->mdm.info[i].isdn_channel = -1;
	  isdn_info_update();
	}
      dev->drivers--;
      dev->channels -= dev->drv[di]->channels;
      kfree(dev->drv[di]->rcverr);
      kfree(dev->drv[di]->rcvcount);
      for (i=0;i<dev->drv[di]->channels;i++)
	free_queue(&dev->drv[di]->rpqueue[i]);
      kfree(dev->drv[di]->rcv_waitq);
      kfree(dev->drv[di]->snd_waitq);
      kfree(dev->drv[di]);
      dev->drv[di] = NULL;
      dev->drvid[di][0] = '\0';
      isdn_info_update();
      restore_flags(flags);
      break;
    default:
      return -1;
  }
  return 0;
}

static int
readbchan (int di, int channel, u_char *buf, u_char *fp, int len, int user) {
  int avail;
  int left;
  int count;
  int copy_l;
  int dflag;
  int flags;
  pqueue *p;
  u_char *cp;

  if (!dev->drv[di]->rpqueue[channel]) {
    if (user)
      interruptible_sleep_on(&dev->drv[di]->rcv_waitq[channel]);
    else
      return 0;
  }
  if (!dev->drv[di])
    return 0;
  save_flags(flags);
  cli();
  avail = dev->drv[di]->rcvcount[channel];
  restore_flags(flags);
  left = MIN(len,avail);
  cp = buf;
  count = 0;
  while (left) {
    if ((copy_l = dev->drv[di]->rpqueue[channel]->length)>left) {
      copy_l = left;
      dflag = 0;
    } else
      dflag = 1;
    p = dev->drv[di]->rpqueue[channel];
    if (user)
      memcpy_tofs(cp,p->rptr,copy_l);
    else
      memcpy(cp,p->rptr,copy_l);
    if (fp) {
      memset(fp,0,copy_l);
      fp += copy_l;
    }
    left -= copy_l;
    count += copy_l;
    cp += copy_l;
    if (dflag) {
      if (fp)
	*(fp-1) = 0xff;
      save_flags(flags);
      cli();
      dev->drv[di]->rpqueue[channel] = (pqueue*)p->next;
      kfree_s(p,p->size);
      restore_flags(flags);
    } else {
      p->rptr += copy_l;
      p->length -= copy_l;
    }
    save_flags(flags);
    cli();
    dev->drv[di]->rcvcount[channel] -= copy_l;
    restore_flags(flags);
  }
  return count;
}

static int
minor2drv(int minor) {
  return(dev->drvmap[minor]);
}

static int
minor2chan(int minor) {
  return(dev->chanmap[minor]);
}

static int
try_readmodem(int i, u_char *buf, int len) {
  int c;
  struct tty_struct *tty;

  if (i<0)
    return 0;
  if (dev->mdm.online[i]) {
    if ((tty = dev->mdm.info[i].tty)) {
      if (dev->mdm.info[i].MCR & UART_MCR_RTS) {
	c = TTY_FLIPBUF_SIZE - tty->flip.count - 1;
	if (c>=len) {
	  if (len>1) {
	    memcpy(tty->flip.char_buf_ptr,buf,len);
	    tty->flip.count += len;
	    memset(tty->flip.flag_buf_ptr,0,len);
	    if (dev->mdm.atmodem[i].mdmreg[12] & 128)
	      tty->flip.flag_buf_ptr[len-1] = 0xff;
	    tty->flip.flag_buf_ptr += len;
	    tty->flip.char_buf_ptr += len;
	  } else
	    tty_insert_flip_char(tty,buf[0],0);
	  queue_task_irq_off(&tty->flip.tqueue, &tq_timer);
	  return 1;
	}
      }
    }
  }
  return 0;
}

static void
readmodem(void) {
  int resched = 0;
  int midx;
  int i;
  int c;
  int r;
  ulong flags;
  struct tty_struct *tty;
  modem_info *info;

  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    if ((midx = dev->m_idx[i])>=0)
      if (dev->mdm.online[midx]) {
	save_flags(flags);
	cli();
	r = 0;
	info = &dev->mdm.info[midx];
	if ((tty = info->tty)) {
	  if (info->MCR & UART_MCR_RTS) {
	    c = TTY_FLIPBUF_SIZE - tty->flip.count - 1;
	    if (c>0) {
	      r = readbchan(info->isdn_driver,info->isdn_channel,
			    tty->flip.char_buf_ptr,
			    tty->flip.flag_buf_ptr,c,0);
	      if (!(dev->mdm.atmodem[midx].mdmreg[12] & 128))
		memset(tty->flip.flag_buf_ptr,0,r);
	      tty->flip.count += r;
	      tty->flip.flag_buf_ptr += r;
	      tty->flip.char_buf_ptr += r;
	      if (r)
		queue_task_irq_off(&tty->flip.tqueue, &tq_timer);
	    }
	  } else
	    r = 1;
	} else
	  r = 1;
	restore_flags(flags);
	if (r) {
	  dev->mdm.rcvsched[midx] = 0;
	  resched = 1;
	} else
	  dev->mdm.rcvsched[midx] = 1;
      }
  }
  if (!resched)
    isdn_timer_ctrl(ISDN_TIMER_MODEMREAD,0);
}

static char *
isdn_statstr(void) {
  static char istatbuf[2048];
  char *p;
  int i;

  sprintf(istatbuf,"idmap:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    sprintf(p,"%s ",(dev->drvmap[i]<0)?"-":dev->drvid[dev->drvmap[i]]);
    p = istatbuf + strlen(istatbuf);
  }
  sprintf(p,"\nchmap:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    sprintf(p,"%d ",dev->chanmap[i]);
    p = istatbuf + strlen(istatbuf);
  }
  sprintf(p,"\ndrmap:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    sprintf(p,"%d ",dev->drvmap[i]);
    p = istatbuf + strlen(istatbuf);
  }
  sprintf(p,"\nusage:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    sprintf(p,"%d ",dev->usage[i]);
    p = istatbuf + strlen(istatbuf);
  }
  sprintf(p,"\nflags:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_DRIVERS;i++) {
    if (dev->drv[i]) {
      sprintf(p,"%ld ",dev->drv[i]->flags);
      p = istatbuf + strlen(istatbuf);
    } else {
      sprintf(p,"? ");
      p = istatbuf + strlen(istatbuf);
    }
  }
  sprintf(p,"\nphone:\t");
  p = istatbuf + strlen(istatbuf);
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    sprintf(p,"%s ",dev->num[i]);
    p = istatbuf + strlen(istatbuf);
  }
  sprintf(p,"\n");
  return istatbuf;
}

/* Module interface-code */

static void
isdn_info_update() {
  infostruct *p = dev->infochain;

  while (p) {
    *(p->private) = 1;
    p = (infostruct*)p->next;
  }
  wake_up_interruptible(&(dev->info_waitq));
}

static int
isdn_read(struct inode *inode, struct file *file,char *buf, int count) {
  uint minor = MINOR(inode->i_rdev);
  int len = 0;
  ulong flags;
  int  drvidx;
  int  chidx;

  if (minor==ISDN_MINOR_STATUS) {
    char *p;
    if (!file->private_data)
      interruptible_sleep_on(&(dev->info_waitq));
    save_flags(flags);
    p = isdn_statstr();
    restore_flags(flags);
    file->private_data = 0;
    if ((len = strlen(p))<=count) {
      memcpy_tofs(buf,p,len);
      file->f_pos += len;
      return len;
    }
    return 0;
  }
  if (!dev->drivers)
    return -ENODEV;
  if (minor<ISDN_MINOR_CTRL) {
    drvidx = minor2drv(minor);
    if (drvidx<0)
      return -ENODEV;
    if (!dev->drv[drvidx]->running)
      return -ENODEV;
    chidx  = minor2chan(minor);
    len = readbchan(drvidx,chidx,buf,0,count,1);
    file->f_pos += len;
    return len;
  }
  if (minor<=ISDN_MINOR_CTRLMAX) {
    drvidx = minor2drv(minor-ISDN_MINOR_CTRL);
    if (drvidx<0)
      return -ENODEV;
    if (!dev->drv[drvidx]->stavail)
      interruptible_sleep_on(&(dev->drv[drvidx]->st_waitq));
    if (dev->drv[drvidx]->interface->readstat)
      len = dev->drv[drvidx]->interface->
	readstat(buf,MIN(count,dev->drv[drvidx]->stavail),1);
    else
      len = 0;
    save_flags(flags);
    cli();
    dev->drv[drvidx]->stavail -= len;
    restore_flags(flags);
    file->f_pos += len;
    return len;
  }
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    return(ippp_read(minor-ISDN_MINOR_PPP,file,buf,count));
#endif
  return -ENODEV;
}

static int
isdn_lseek(struct inode * inode, struct file * file, off_t offset, int orig) {
  return -ESPIPE;
}

static int
isdn_write(struct inode *inode, struct file *file, FOPS_CONST char *buf, int count) {
  uint minor = MINOR(inode->i_rdev);
  int  drvidx;
  int  chidx;

  if (minor==ISDN_MINOR_STATUS)
    return -EPERM;
  if (!dev->drivers)
    return -ENODEV;
  if (minor<ISDN_MINOR_CTRL) {
    drvidx = minor2drv(minor);
    if (drvidx<0)
      return -ENODEV;
    if (!dev->drv[drvidx]->running)
      return -ENODEV;
    chidx  = minor2chan(minor);
    while (dev->drv[drvidx]->interface->writebuf(chidx,buf,count,1)!=count)
      interruptible_sleep_on(&dev->drv[drvidx]->snd_waitq[chidx]);
    return count;
  }
  if (minor<=ISDN_MINOR_CTRLMAX) {
    drvidx = minor2drv(minor-ISDN_MINOR_CTRL);
    if (drvidx<0)
      return -ENODEV;
    if (!dev->drv[drvidx]->running)
      return -ENODEV;
    if (dev->drv[drvidx]->interface->writecmd)
      return(dev->drv[drvidx]->interface->writecmd(buf,count,1));
    else
      return count;
  }
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    return(ippp_write(minor-ISDN_MINOR_PPP,file,buf,count));
#endif
  return -ENODEV;
}

static int
isdn_select(struct inode *inode, struct file *file,int type,select_table *st) {
  uint minor = MINOR(inode->i_rdev);

  if (minor==ISDN_MINOR_STATUS) {
    if (file->private_data)
      return 1;
    else {
      if (st)
	select_wait(&(dev->info_waitq),st);
      return 0;
    }
  }
  if (minor<=ISDN_MINOR_CTRLMAX)
    return 1;
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    return(ippp_select(minor-ISDN_MINOR_PPP,file,type,st));
#endif
  return -ENODEV;
}

#ifdef  ISDN_DEBUG_MODEM_SENDOPT
static int mwmax = 0;
#endif

static int
isdn_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg) {
  uint minor = MINOR(inode->i_rdev);
  isdn_ctrl c;
  int  drvidx;
  int  chidx;
  int  ret;
  char *s;
  char name[10];
  char bname[21];
  isdn_ioctl_struct iocts;
  isdn_net_ioctl_phone phone;
  isdn_net_ioctl_cfg   cfg;

  if (minor==ISDN_MINOR_STATUS)
    return -EPERM;
  if (!dev->drivers)
    return -ENODEV;
  if (minor<ISDN_MINOR_CTRL) {
    drvidx = minor2drv(minor);
    if (drvidx<0)
      return -ENODEV;
    chidx  = minor2chan(minor);
    if (!dev->drv[drvidx]->running)
      return -ENODEV;
    return 0;
  }
  if (minor<=ISDN_MINOR_CTRLMAX) {
    switch (cmd) {
#ifdef  ISDN_DEBUG_MODEM_SENDOPT
      case ISDN_IOCTL_GETMAX:
	memcpy_tofs((char*)arg,&mwmax,sizeof(int));
	return 0;
      case ISDN_IOCTL_RESETMAX:
	mwmax = 0;
	return 0;
#endif
      case ISDN_IOCTL_NET_ADDIF:
	/* Add a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(name))))
	    return ret;
	  memcpy_fromfs(name,(char*)arg,sizeof(name));
	  s = name;
	} else
	  s = NULL;
	if ((s = isdn_net_new(s,NULL))) {
	  if ((ret = verify_area(VERIFY_WRITE,(void*)arg, strlen(s)+1)))
	    return ret;
	  memcpy_tofs((char*)arg,s,strlen(s)+1);
	  return 0;
	} else 
	  return -ENODEV;
      case ISDN_IOCTL_NET_ADDSLAVE:
	/* Add a slave to a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(bname))))
	    return ret;
	  memcpy_fromfs(bname,(char*)arg,sizeof(bname));
	} else
	  return -EINVAL;
	if ((s = isdn_net_newslave(bname))) {
	  if ((ret = verify_area(VERIFY_WRITE,(void*)arg, strlen(s)+1)))
	    return ret;
	  memcpy_tofs((char*)arg,s,strlen(s)+1);
	  return 0;
	} else
	  return -ENODEV;
      case ISDN_IOCTL_NET_DELIF:
	/* Delete a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(name))))
	    return ret;
	  memcpy_fromfs(name,(char*)arg,sizeof(name));
	  return isdn_net_rm(name);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_SETCFG:
	/* Set configurable parameters of a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(cfg))))
	    return ret;
	  memcpy_fromfs((char*)&cfg,(char*)arg,sizeof(cfg));
	  return isdn_net_setcfg(&cfg);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_GETCFG:
	/* Get configurable parameters of a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(cfg))))
	    return ret;
	  memcpy_fromfs((char*)&cfg,(char*)arg,sizeof(cfg));
	  if (!(ret = isdn_net_getcfg(&cfg))) {
	    if ((ret = verify_area(VERIFY_WRITE,(void*)arg, sizeof(cfg))))
	      return ret;
	    memcpy_tofs((char*)arg,(char*)&cfg,sizeof(cfg));
	  }
	  return ret;
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_ADDNUM:
	/* Add a phone-number to a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(phone))))
	    return ret;
	  memcpy_fromfs((char*)&phone,(char*)arg,sizeof(phone));
	  return isdn_net_addphone(&phone);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_GETNUM:
	/* Get list of phone-numbers of a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(phone))))
	    return ret;
	  memcpy_fromfs((char*)&phone,(char*)arg,sizeof(phone));
	  return isdn_net_getphones(&phone,(char*)arg);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_DELNUM:
	/* Delete a phone-number of a network-interface */
	if (arg) {
	  memcpy_fromfs((char*)&phone,(char*)arg,sizeof(phone));
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(phone))))
	    return ret;
	  return isdn_net_delphone(&phone);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_NET_DIAL:
	/* Force dialing of a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(name))))
	    return ret;
	  memcpy_fromfs(name,(char*)arg,sizeof(name));
	  return isdn_net_force_dial(name);
	} else
	  return -EINVAL;
      case ISDN_IOCTL_SET_VERBOSE:
	dev->net_verbose = arg;
	printk(KERN_INFO "isdn: Verbose-Level is %d\n", dev->net_verbose);
	return 0;
      case ISDN_IOCTL_SET_GLOBALSTOP:
	if (arg)
	  dev->global_flags |= ISDN_GLOBAL_STOPPED;
	else
	  dev->global_flags &= ~ISDN_GLOBAL_STOPPED;
	printk(KERN_INFO "isdn: Global Mode %s\n",
	       (dev->global_flags&ISDN_GLOBAL_STOPPED)?"stopped":"running");
	return 0;
      case ISDN_IOCTL_SET_BUSREJECT:
	drvidx = -1;
	if (arg) {
	  int  i;
	  char *p;
	  if ((ret = verify_area(VERIFY_READ,(void*)arg,
				 sizeof(isdn_ioctl_struct))))
	    return ret;
	  memcpy_fromfs((char*)&iocts,(char*)arg,sizeof(isdn_ioctl_struct));
	  if (strlen(iocts.drvid)) {
	    if ((p = strchr(iocts.drvid,',')))
	      *p = 0;
	    drvidx = -1;
	    for (i=0;i<ISDN_MAX_DRIVERS;i++)
	      if (!(strcmp(dev->drvid[i],iocts.drvid))) {
		drvidx = i;
		break;
	    }
	  }
	}
	if (drvidx==-1)
	  return -ENODEV;
	dev->drv[drvidx]->reject_bus = iocts.arg;
	return 0;
      case ISDN_IOCTL_GET_SETUP:
	/* Get complete setup (all network-interfaces and profile-
           settings of all tty-devices */
	if (arg)
	  return(isdn_get_allcfg((char *)arg));
	else
	  return -EINVAL;
	break;
      case ISDN_IOCTL_SET_SETUP:
	/* Set complete setup (all network-interfaces and profile-
           settings of all tty-devices */
	if (arg)
	  return(isdn_set_allcfg((char *)arg));
	else
	  return -EINVAL;
	break;
      case ISDN_IOCTL_NET_HANGUP:
	/* Force hangup of a network-interface */
	if (arg) {
	  if ((ret = verify_area(VERIFY_READ,(void*)arg, sizeof(name))))
	    return ret;
	  memcpy_fromfs(name,(char*)arg,sizeof(name));
	  return isdn_net_force_hangup(name);
	} else
	  return -EINVAL;
	break;
      case ISDN_IOCTL_PROFILESIG:
	dev->profd = current;
	return 0;
	break;
      case ISDN_IOCTL_PROFILEGET:
	/* Get all Modem-Profiles */
	if (arg) {
	  char *p = (char *)arg;
	  int i;

	  if ((ret = verify_area(VERIFY_WRITE,(void*)arg,
				 (ISDN_MODEM_ANZREG+ISDN_MSNLEN)
				 *ISDN_MAX_CHANNELS)))
	    return ret;
	  
	  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
	    memcpy_tofs(p,dev->mdm.atmodem[i].profile,ISDN_MODEM_ANZREG);
	    p += ISDN_MODEM_ANZREG;
	    memcpy_tofs(p,dev->mdm.atmodem[i].pmsn,ISDN_MSNLEN);
	    p += ISDN_MSNLEN;
	  }
	  return (ISDN_MODEM_ANZREG+ISDN_MSNLEN)*ISDN_MAX_CHANNELS;
	} else
	  return -EINVAL;
	break;
      case ISDN_IOCTL_PROFILESET:
	/* Get all Modem-Profiles */
	if (arg) {
	  char *p = (char *)arg;
	  int i;

	  if ((ret = verify_area(VERIFY_READ,(void*)arg,
				 (ISDN_MODEM_ANZREG+ISDN_MSNLEN)
				 *ISDN_MAX_CHANNELS)))
	    return ret;
	  
	  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
	    memcpy_fromfs(dev->mdm.atmodem[i].profile,p,ISDN_MODEM_ANZREG);
	    p += ISDN_MODEM_ANZREG;
	    memcpy_fromfs(dev->mdm.atmodem[i].pmsn,p,ISDN_MSNLEN);
	    p += ISDN_MSNLEN;
	  }
	  return 0;
	} else
	  return -EINVAL;
	break;
      case ISDN_IOCTL_SETMAPPING:
      case ISDN_IOCTL_GETMAPPING:
	/* Set MSN->EAZ-Mapping for a driver */
	if (arg) {
	  int  i;
	  char *p;
	  char nstring[255];

	  if ((ret = verify_area(VERIFY_READ,(void*)arg,
				 sizeof(isdn_ioctl_struct))))
	    return ret;
	  memcpy_fromfs((char*)&iocts,(char*)arg,sizeof(isdn_ioctl_struct));
	  if (strlen(iocts.drvid)) {
	    drvidx = -1;
	    for (i=0;i<ISDN_MAX_DRIVERS;i++)
	      if (!(strcmp(dev->drvid[i],iocts.drvid))) {
		drvidx = i;
		break;
	    }
	  } else
	    drvidx = 0;
	  if (drvidx==-1)
	    return -ENODEV;
	  if (cmd == ISDN_IOCTL_SETMAPPING) {
	    if ((ret = verify_area(VERIFY_READ,(void*)iocts.arg,255)))
	      return ret;
	    memcpy_fromfs(nstring,(char*)iocts.arg,255);
	    memset(dev->drv[drvidx]->msn2eaz,0,
		   sizeof(dev->drv[drvidx]->msn2eaz));
	    p = strtok(nstring,",");
	    i = 0;
	    while ((p) && (i < 10)) {
	      strcpy(dev->drv[drvidx]->msn2eaz[i++],p);
	      p = strtok(NULL,",");
	    }
	  } else {
	    p = nstring;
	    for (i=0;i<10;i++)
	      p += sprintf(p,"%s%s",
			   strlen(dev->drv[drvidx]->msn2eaz[i])?
			   dev->drv[drvidx]->msn2eaz[i]:"-",
			   (i<9)?",":"\0");
	    if ((ret = verify_area(VERIFY_WRITE,(void*)iocts.arg,
		strlen(nstring)+1)))
	      return ret;
	    memcpy_tofs((char*)iocts.arg,nstring,strlen(nstring)+1);
	  }
	  return 0;
	} else
	  return -EINVAL;
      case ISDN_IOCTL_DEBUGVAR:
	if (arg) {
	  if ((ret = verify_area(VERIFY_WRITE,(void*)arg,sizeof(ulong))))
	    return ret;
	  memcpy_tofs((char*)arg,(char*)&dev,sizeof(ulong));
	  return 0;
	} else
	  return -EINVAL;
	break;
      default:
	cmd -= ISDN_IOCTL_DRVIOCTL;
	if (arg) {
	  int  i;
	  char *p;
	  if ((ret = verify_area(VERIFY_READ,(void*)arg,
				 sizeof(isdn_ioctl_struct))))
	    return ret;
	  memcpy_fromfs((char*)&iocts,(char*)arg,sizeof(isdn_ioctl_struct));
	  if (strlen(iocts.drvid)) {
	    if ((p = strchr(iocts.drvid,',')))
	      *p = 0;
	    drvidx = -1;
	    for (i=0;i<ISDN_MAX_DRIVERS;i++)
	      if (!(strcmp(dev->drvid[i],iocts.drvid))) {
		drvidx = i;
		break;
	    }
	  } else
	    drvidx = 0;
	  if (drvidx==-1)
	    return -ENODEV;
	  c.driver = drvidx;
	  c.command = ISDN_CMD_IOCTL;
	  c.arg = cmd;
	  memcpy(c.num,(char *)&iocts.arg,sizeof(ulong));
	  return(dev->drv[drvidx]->interface->command(&c));
	} else
	  return -EINVAL;
    }
  }
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    return(ippp_ioctl(minor-ISDN_MINOR_PPP,file,cmd,arg));
#endif
  return -ENODEV;
}

/*
 * Open the device code.
 * MOD_INC_USE_COUNT make sure that the driver memory is not freed
 * while the device is in use.
 */
static int
isdn_open( struct inode* ino, struct file* filep) {
  uint minor = MINOR(ino->i_rdev);
  int  drvidx;
  int  chidx;
  isdn_ctrl c;

  if (minor==ISDN_MINOR_STATUS) {
    infostruct *p;

    if ((p = (infostruct*)kmalloc(sizeof(infostruct),GFP_KERNEL))) {
      MOD_INC_USE_COUNT;
      p->next = (char*)dev->infochain;
      p->private = (char*)&(filep->private_data);
      dev->infochain = p;
      /* At opening we allow a single update */
      filep->private_data = (char*)1;
      return 0;
    } else
      return -ENOMEM;
  }
  if (!dev->channels)
    return -ENODEV;
  if (minor<ISDN_MINOR_CTRL) {
    drvidx = minor2drv(minor);
    if (drvidx<0)
      return -ENODEV;
    chidx  = minor2chan(minor);
    if (!dev->drv[drvidx]->running)
      return -ENODEV;
    if (!(dev->drv[drvidx]->flags & (1<<chidx)))
      return -ENODEV;
    c.command = ISDN_CMD_LOCK;
    c.driver  = drvidx;
    (void)dev->drv[drvidx]->interface->command(&c);
    MOD_INC_USE_COUNT;
    return 0;
  }
  if (minor<=ISDN_MINOR_CTRLMAX) {
    drvidx = minor2drv(minor-ISDN_MINOR_CTRL);
    if (drvidx<0)
      return -ENODEV;
    c.command = ISDN_CMD_LOCK;
    c.driver  = drvidx;
    (void)dev->drv[drvidx]->interface->command(&c);
    MOD_INC_USE_COUNT;
    return 0;
  }
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    return(ippp_open(minor-ISDN_MINOR_PPP,filep));
#endif
  return -ENODEV;
}

static void
isdn_close( struct inode* ino, struct file* filep) {
  uint minor = MINOR(ino->i_rdev);
  int drvidx;
  isdn_ctrl c;

  if (!dev->channels)
    return;
  MOD_DEC_USE_COUNT;
  if (minor==ISDN_MINOR_STATUS) {
    infostruct *p = dev->infochain;
    infostruct *q = NULL;
    while (p) {
      if (p->private == (char *)&(filep->private_data)) {
	if (q)
	  q->next = p->next;
	else
	  dev->infochain = (infostruct *)(p->next);
	return;
      }
      q = p;
      p = (infostruct *)(p->next);
    }
    printk(KERN_WARNING "isdn: No private data while closing isdnctrl\n");
  }
  if (minor<ISDN_MINOR_CTRL) {
    drvidx = minor2drv(minor);
    if (drvidx<0)
      return;
    c.command = ISDN_CMD_UNLOCK;
    c.driver  = drvidx;
    (void)dev->drv[drvidx]->interface->command(&c);
    return;
  }
  if (minor<=ISDN_MINOR_CTRLMAX) {
    drvidx = minor2drv(minor-ISDN_MINOR_CTRL);
    if (drvidx<0)
      return;
    if (dev->profd == current)
      dev->profd = NULL;
    c.command = ISDN_CMD_UNLOCK;
    c.driver  = drvidx;
    (void)dev->drv[drvidx]->interface->command(&c);
    return;
  }
#ifdef ISYNCPPP
  if (minor<=ISDN_MINOR_PPPMAX)
    ippp_release(minor-ISDN_MINOR_PPP,filep);
#endif
}

/*
static struct
file_operations ippp_fops = {
  NULL,
  ippp_read,
  ippp_write,
  NULL,
  ippp_select,
  ippp_ioctl,
  NULL,
  ippp_open,
  ippp_release,
};
*/

static struct file_operations isdn_fops = {
  isdn_lseek,
  isdn_read,
  isdn_write,
  NULL,		/* isdn_readdir */
  isdn_select, 	/* isdn_select */
  isdn_ioctl,	/* isdn_ioctl */
  NULL,         /* isdn_mmap */
  isdn_open,
  isdn_close,
  NULL		/* fsync */
};

static char *
map_eaz2msn(char *msn, int di) {
  driver *this = dev->drv[di];
  int i;

  if (strlen(msn)==1) {
    i = msn[0] - '0';
    if ((i>=0)&&(i<=9))
      if (strlen(this->msn2eaz[i]))
	return(this->msn2eaz[i]);
  }
  return(msn);
}

/*
 * Find an unused ISDN-channel, whose feature-flags match the
 * given L2- and L3-protocols.
 */
static int
get_free_channel(int usage, int l2_proto, int l3_proto, int pre_dev
		 ,int pre_chan) {
  int i;
  ulong flags;
  ulong features;

  save_flags(flags);
  cli();
  features = (1 << l2_proto) | (0x100 << l3_proto);
  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (USG_NONE(dev->usage[i]) &&
	(dev->drvmap[i]!=-1)) {
      int d = dev->drvmap[i];
      if ((dev->usage[i]&ISDN_USAGE_EXCLUSIVE) &&
	  ((pre_dev != d)||(pre_chan != dev->chanmap[i])))
	continue;
      if ((dev->drv[d]->running)) {
	if ((dev->drv[d]->interface->features & features) == features) {
	  if ((pre_dev<0) || (pre_chan<0)) {
	    dev->usage[i] &= ISDN_USAGE_EXCLUSIVE;
	    dev->usage[i] |= usage;
	    isdn_info_update();
	    restore_flags(flags);
	    return i;
	  } else {
	    if ((pre_dev == d) && (pre_chan == dev->chanmap[i])) {
	      dev->usage[i] &= ISDN_USAGE_EXCLUSIVE;
	      dev->usage[i] |= usage;
	      isdn_info_update();
	      restore_flags(flags);
	      return i;
	    }
	  }
	}
      }
    }
  restore_flags(flags);
  return -1;
}

/*
 * Set state of ISDN-channel to 'unused'
 */
static void
free_channel(int di, int ch, int usage) {
  int i;
  ulong flags;

  save_flags(flags);
  cli();
  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (((dev->usage[i] & ISDN_USAGE_MASK)== usage) && 
	(dev->drvmap[i]==di)     &&
	(dev->chanmap[i]==ch)       ) {
      dev->usage[i] &= (ISDN_USAGE_NONE | ISDN_USAGE_EXCLUSIVE);
      strcpy(dev->num[i],"???");
      isdn_info_update();
      restore_flags(flags);
      return ;
    }
  restore_flags(flags);
}

/*
 * Delete Exclusive-Flag for ISDN-channel
 */
static void
unexclusive_channel(int di, int ch) {
  int i;
  ulong flags;

  save_flags(flags);
  cli();
  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if ((dev->drvmap[i]==di)     &&
	(dev->chanmap[i]==ch)       ) {
      dev->usage[i] &= ~ISDN_USAGE_EXCLUSIVE;
      isdn_info_update();
      restore_flags(flags);
      return ;
    }
  restore_flags(flags);
}

/************************************************************
 *
 * Modem-functions
 *
 * "stolen" from original Linux-serial.c and friends.
 *
 ************************************************************/

static void
modem_dial(char *n, modem_info *info, atemu *m) {
  isdn_ctrl cmd;
  ulong     flags;
  int       i;

  save_flags(flags);
  cli();
  i = get_free_channel(ISDN_USAGE_MODEM,m->mdmreg[14],m->mdmreg[15],-1,-1);
  if (i<0) {
    restore_flags(flags);
    modem_result(6,info);
  } else {
    if (strlen(m->msn)) {
      info->isdn_driver  = dev->drvmap[i];
      info->isdn_channel = dev->chanmap[i];
      info->drv_index    = i;
      dev->m_idx[i]      = info->line;
      dev->usage[i]      |= ISDN_USAGE_OUTGOING;
      isdn_info_update();
      restore_flags(flags);
      cmd.driver  = info->isdn_driver;
      cmd.arg     = info->isdn_channel;
      cmd.command = ISDN_CMD_CLREAZ;
      dev->drv[info->isdn_driver]->interface->command(&cmd);
      strcpy(cmd.num,map_eaz2msn(m->msn,info->isdn_driver));
      cmd.driver  = info->isdn_driver;
      cmd.command = ISDN_CMD_SETEAZ;
      dev->drv[info->isdn_driver]->interface->command(&cmd);
      cmd.driver  = info->isdn_driver;
      cmd.command = ISDN_CMD_SETL2;
      cmd.arg     = info->isdn_channel + (m->mdmreg[14] << 8);
      dev->drv[info->isdn_driver]->interface->command(&cmd);
      cmd.driver  = info->isdn_driver;
      cmd.command = ISDN_CMD_SETL3;
      cmd.arg     = info->isdn_channel + (m->mdmreg[15] << 8);
      dev->drv[info->isdn_driver]->interface->command(&cmd);
      cmd.driver  = info->isdn_driver;
      cmd.arg     = info->isdn_channel;
      sprintf(cmd.num,"%s,%s,%d,%d",n,map_eaz2msn(m->msn,info->isdn_driver),
	      m->mdmreg[18],m->mdmreg[19]);
      cmd.command = ISDN_CMD_DIAL;
      dev->mdm.dialing[info->line] = 1;
      strcpy(dev->num[i],n);
      isdn_info_update();
      dev->drv[info->isdn_driver]->interface->command(&cmd);
    } else
      restore_flags(flags);
  }
}

static void
modem_hup(modem_info *info) {
  isdn_ctrl cmd;

  dev->mdm.rcvsched[info->line] = 0;
  dev->mdm.online[info->line] = 0;
  if (info->isdn_driver>=0) {
    cmd.driver  = info->isdn_driver;
    cmd.command = ISDN_CMD_HANGUP;
    cmd.arg     = info->isdn_channel;
    dev->drv[info->isdn_driver]->interface->command(&cmd);
    isdn_all_eaz(info->isdn_driver,info->isdn_channel);
    free_channel(info->isdn_driver,info->isdn_channel,ISDN_USAGE_MODEM);
  }
  dev->m_idx[info->drv_index] = -1;
  info->isdn_driver  = -1;
  info->isdn_channel = -1;
  info->drv_index    = -1;
}

static inline int
serial_paranoia_check(modem_info *info,	dev_t device, const char *routine) {
#ifdef MODEM_PARANOIA_CHECK
  if (!info) {
    printk(KERN_WARNING "isdn: null info_struct for (%d, %d) in %s\n",
	   MAJOR(device), MINOR(device), routine);
    return 1;
  }
  if (info->magic != ISDN_ASYNC_MAGIC) {
    printk(KERN_WARNING "isdn: bad magic for modem struct (%d, %d) in %s\n",
	   MAJOR(device), MINOR(device), routine);
    return 1;
  }
#endif
  return 0;
}

/*
 * This routine is called to set the UART divisor registers to match
 * the specified baud rate for a serial port.
 */
static void
change_speed(modem_info *info) {
  uint  cflag,
        cval,
        fcr,
        quot;
  int	i;
  
  if (!info->tty || !info->tty->termios)
    return;
  cflag = info->tty->termios->c_cflag;

  quot = i = cflag & CBAUD;  
  if (i & CBAUDEX) {
    i &= ~CBAUDEX;
    if (i < 1 || i > 2) 
      info->tty->termios->c_cflag &= ~CBAUDEX;
    else
      i += 15;
  }
  if (quot) {
    info->MCR |= UART_MCR_DTR;
  } else {
    info->MCR &= ~UART_MCR_DTR;
    modem_reset_regs(&dev->mdm.atmodem[info->line],0);
    if (dev->mdm.online[info->line]) {
#ifdef ISDN_DEBUG_MODEM_HUP
      printk(KERN_DEBUG "Mhup in changespeed\n");
#endif
      modem_hup(info);
      modem_result(3,info);
    }
    return;
  }
  /* byte size and parity */
  cval = cflag & (CSIZE | CSTOPB);
  cval >>= 4;
  if (cflag & PARENB)
    cval |= UART_LCR_PARITY;
  if (!(cflag & PARODD))
    cval |= UART_LCR_EPAR;
  fcr = 0;

  /* CTS flow control flag and modem status interrupts */
  if (cflag & CRTSCTS) {
    info->flags |= ISDN_ASYNC_CTS_FLOW;
  } else
    info->flags &= ~ISDN_ASYNC_CTS_FLOW;
  if (cflag & CLOCAL)
    info->flags &= ~ISDN_ASYNC_CHECK_CD;
  else {
    info->flags |= ISDN_ASYNC_CHECK_CD;
  }
}

static int
startup(modem_info * info) {
  ulong flags;
  
  if (info->flags & ISDN_ASYNC_INITIALIZED)
    return 0;  
  if (!info->type) {
    if (info->tty)
      set_bit(TTY_IO_ERROR, &info->tty->flags);
    return 0;
  }
  save_flags(flags); cli();

#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "starting up ttyi%d ...\n", info->line);
#endif
  /*
   * Now, initialize the UART 
   */
  info->MCR = UART_MCR_DTR | UART_MCR_RTS | UART_MCR_OUT2;
  if (info->tty)
    clear_bit(TTY_IO_ERROR, &info->tty->flags);
  /*
   * and set the speed of the serial port
   */
  change_speed(info);
  
  info->flags |= ISDN_ASYNC_INITIALIZED;
  dev->mdm.msr[info->line] |= UART_MSR_DSR;
#if FUTURE
  info->send_outstanding = 0;
#endif
  MOD_INC_USE_COUNT;
  restore_flags(flags);
  return 0;
}

/*
 * This routine will shutdown a serial port; interrupts are disabled, and
 * DTR is dropped if the hangup on close termio flag is on.
 */
static void
shutdown(modem_info * info) {
  ulong	flags;
  
  if (!(info->flags & ISDN_ASYNC_INITIALIZED))
    return;
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "Shutting down isdnmodem port %d ....\n", info->line);
#endif
  save_flags(flags); cli(); /* Disable interrupts */
  if (!info->tty || (info->tty->termios->c_cflag & HUPCL)) {
    info->MCR &= ~(UART_MCR_DTR|UART_MCR_RTS);
    modem_reset_regs(&dev->mdm.atmodem[info->line],0);
    if (dev->mdm.online[info->line]) {
#ifdef ISDN_DEBUG_MODEM_HUP
      printk(KERN_DEBUG "Mhup in shutdown\n");
#endif
      modem_hup(info);
    }
  }
  
  if (info->tty)
    set_bit(TTY_IO_ERROR, &info->tty->flags);
  
  info->flags &= ~ISDN_ASYNC_INITIALIZED;
  MOD_DEC_USE_COUNT;
  restore_flags(flags);
}

static int
modem_write(struct tty_struct * tty, int from_user, FOPS_CONST u_char *buf, int count) {
  int	c, total = 0;
  modem_info *info = (modem_info *)tty->driver_data;
  ulong flags;
  int i;

  if (serial_paranoia_check(info, tty->device, "modem_write"))
    return 0;
  if (!tty)
    return 0;
  save_flags(flags);
  cli();
  while (1) {
    c = MIN(count,info->xmit_size-info->xmit_count-1);
    if (info->isdn_driver >= 0) {
#if 0
      if (info->isdn_driver != 0) {
	printk(KERN_DEBUG "FIDO: Zwei HW-Treiber geladen? Ansonsten ist was faul.\n");
        break;
      }
      int drvidx = info->isdn_driver;
      driver *driv = dev->drv[drvidx];
      i = driv->maxbufsize;
#else
      i = dev->drv[info->isdn_driver]->maxbufsize;
#endif
      c = MIN(c,i);
    }
    if (c<=0)
      break;
    i = info->line;
    if (dev->mdm.online[i]) {
#ifdef  ISDN_DEBUG_MODEM_SENDOPT
      mwmax = (mwmax<c)?c:mwmax;
#endif
      modem_chk3plus(buf,dev->mdm.atmodem[i].mdmreg[2],c,
		     &(dev->mdm.atmodem[i].pluscount),
		     &(dev->mdm.atmodem[i].lastplus),from_user);
      if (from_user)
	memcpy_fromfs(&(info->xmit_buf[info->xmit_count]),buf,c);
      else
	memcpy(&(info->xmit_buf[info->xmit_count]),buf,c);
      info->xmit_count += c;
      if (dev->mdm.atmodem[i].mdmreg[13] & 1) {
	char *bufptr;
	int  buflen;
#if 0
	printk(KERN_DEBUG "WB1: %d\n",info->xmit_count);
#endif
	bufptr = info->xmit_buf;
	buflen = info->xmit_count;
	if (dev->mdm.atmodem[i].mdmreg[13] & 2) {
	  /* Add T.70 simplyfied header */
	  
#ifdef ISDN_DEBUG_MODEM_DUMP
	  isdn_dumppkt("T70pack1:",bufptr,buflen,40);
#endif
	  bufptr -= 4;
	  buflen += 4;
	  memcpy(bufptr,"\1\0\1\0",4);
#ifdef ISDN_DEBUG_MODEM_DUMP
	  isdn_dumppkt("T70pack2:",bufptr,buflen,40);
#endif
	}
	if (dev->drv[info->isdn_driver]->interface->
	    writebuf(info->isdn_channel,bufptr,buflen,0)>0) {
	  info->xmit_count = 0;
	  info->xmit_size = dev->mdm.atmodem[i].mdmreg[16] * 16;
#if FUTURE
	  info->send_outstanding++;
	  dev->mdm.msr[i] &= ~UART_MSR_CTS;
#endif
	  if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	      tty->ldisc.write_wakeup)
	    (tty->ldisc.write_wakeup)(tty);
	  wake_up_interruptible(&tty->write_wait);
	}
      }
    } else {
      if (dev->mdm.dialing[i]) {
	dev->mdm.dialing[i] = 0;
#ifdef ISDN_DEBUG_MODEM_HUP
	printk(KERN_DEBUG "Mhup in modem_write\n");
#endif
	modem_hup(info);
	modem_result(3,info);
      } else
	c = modem_parse(buf,c,info,from_user);
    }
    buf += c;
    count -= c;
    total += c;
  }
  if (info->xmit_count)
    isdn_timer_ctrl(ISDN_TIMER_MODEMXMIT,1);
  restore_flags(flags);
  return total;
}

static int
modem_write_room(struct tty_struct *tty) {
  modem_info *info = (modem_info *)tty->driver_data;
  int ret;

  if (serial_paranoia_check(info, tty->device, "modem_write_room"))
    return 0;
  if (!dev->mdm.online[info->line])
    return info->xmit_size - 1;
  ret = info->xmit_size - info->xmit_count - 1;
  return (ret<0)?0:ret;
}

static int
modem_chars_in_buffer(struct tty_struct *tty) {
  modem_info *info = (modem_info *)tty->driver_data;

  if (serial_paranoia_check(info, tty->device, "modem_chars_in_buffer"))
    return 0;
  if (!dev->mdm.online[info->line])
    return 0;
  return (info->xmit_count);
}

static void
modem_flush_buffer(struct tty_struct *tty) {
  modem_info *info = (modem_info *)tty->driver_data;
  uint flags;
  
  if (serial_paranoia_check(info, tty->device, "modem_flush_buffer"))
    return;
  save_flags(flags);
  cli();
  info->xmit_count = 0;
  restore_flags(flags);
  wake_up_interruptible(&tty->write_wait);
  if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
      tty->ldisc.write_wakeup)
    (tty->ldisc.write_wakeup)(tty);
}

static void
modem_flush_chars(struct tty_struct *tty) {
  modem_info *info = (modem_info *)tty->driver_data;
  
  if (serial_paranoia_check(info, tty->device, "modem_flush_buffer"))
    return;
  if (info->xmit_count>0)
    isdn_timer_ctrl(ISDN_TIMER_MODEMXMIT,1);
}

/*
 * ------------------------------------------------------------
 * modem_throttle()
 * 
 * This routine is called by the upper-layer tty layer to signal that
 * incoming characters should be throttled.
 * ------------------------------------------------------------
 */
static void
modem_throttle(struct tty_struct * tty) {
  modem_info *info = (modem_info *)tty->driver_data;
  
  if (serial_paranoia_check(info, tty->device, "modem_throttle"))
    return;
  if (I_IXOFF(tty))
    info->x_char = STOP_CHAR(tty);
  info->MCR &= ~UART_MCR_RTS;
}

static void
modem_unthrottle(struct tty_struct * tty) {
  modem_info *info = (modem_info *)tty->driver_data;
  
  if (serial_paranoia_check(info, tty->device, "modem_unthrottle"))
    return;
  if (I_IXOFF(tty)) {
    if (info->x_char)
      info->x_char = 0;
    else
      info->x_char = START_CHAR(tty);
  }
  info->MCR |= UART_MCR_RTS;
}

/*
 * ------------------------------------------------------------
 * modem_ioctl() and friends
 * ------------------------------------------------------------
 */

/*
 * get_lsr_info - get line status register info
 *
 * Purpose: Let user call ioctl() to get info when the UART physically
 * 	    is emptied.  On bus types like RS485, the transmitter must
 * 	    release the bus after transmitting. This must be done when
 * 	    the transmit shift register is empty, not be done when the
 * 	    transmit holding register is empty.  This functionality
 * 	    allows RS485 driver to be written in user space. 
 */
static int
get_lsr_info(modem_info * info, uint *value) {
  u_char status;
  uint   result;
  ulong  flags;

  save_flags(flags);
  cli();
  status = dev->mdm.msr[info->line];
  restore_flags(flags);
  result = ((status & UART_LSR_TEMT) ? TIOCSER_TEMT : 0);
  put_fs_long(result,(ulong *) value);
  return 0;
}


static int
get_modem_info(modem_info * info, uint *value) {
  u_char control, status;
  uint result;
  ulong  flags;
  
  control = info->MCR;
  save_flags(flags);
  cli();
  status = dev->mdm.msr[info->line];
  restore_flags(flags);
  result =  ((control & UART_MCR_RTS) ? TIOCM_RTS : 0)
    | ((control & UART_MCR_DTR) ? TIOCM_DTR : 0)
    | ((status  & UART_MSR_DCD) ? TIOCM_CAR : 0)
    | ((status  & UART_MSR_RI)  ? TIOCM_RNG : 0)
    | ((status  & UART_MSR_DSR) ? TIOCM_DSR : 0)
    | ((status  & UART_MSR_CTS) ? TIOCM_CTS : 0);
  put_fs_long(result,(ulong *) value);
  return 0;
}

static int
set_modem_info(modem_info * info, uint cmd, uint *value) {
  uint arg = get_fs_long((ulong *) value);
  
  switch (cmd) {
    case TIOCMBIS: 
      if (arg & TIOCM_RTS) {
	info->MCR |= UART_MCR_RTS;
      }
      if (arg & TIOCM_DTR) {
	info->MCR |= UART_MCR_DTR;
      }
      break;
    case TIOCMBIC:
      if (arg & TIOCM_RTS) {
	info->MCR &= ~UART_MCR_RTS;
      }
      if (arg & TIOCM_DTR) {
	info->MCR &= ~UART_MCR_DTR;
	modem_reset_regs(&dev->mdm.atmodem[info->line],0);
	if (dev->mdm.online[info->line]) {
#ifdef ISDN_DEBUG_MODEM_HUP
	  printk(KERN_DEBUG "Mhup in TIOCMBIC\n");
#endif
	  modem_hup(info);
	  modem_result(3,info);
	}
      }
      break;
    case TIOCMSET:
      info->MCR = ((info->MCR & ~(UART_MCR_RTS | UART_MCR_DTR))
		   | ((arg & TIOCM_RTS) ? UART_MCR_RTS : 0)
		   | ((arg & TIOCM_DTR) ? UART_MCR_DTR : 0));
      if (!(info->MCR & UART_MCR_DTR)) {
	modem_reset_regs(&dev->mdm.atmodem[info->line],0);
	if (dev->mdm.online[info->line]) {
#ifdef ISDN_DEBUG_MODEM_HUP
	  printk(KERN_DEBUG "Mhup in TIOCMSET\n");
#endif
	  modem_hup(info);
	  modem_result(3,info);
	}
      }
      break;
    default:
      return -EINVAL;
  }
  return 0;
}

static int
modem_ioctl(struct tty_struct *tty, struct file * file,
	    uint cmd, ulong arg) {
  modem_info * info = (modem_info *)tty->driver_data;
  int error;
  int retval;

  if (serial_paranoia_check(info, tty->device, "modem_ioctl"))
    return -ENODEV;
  switch (cmd) {
    case TCSBRK:	/* SVID version: non-zero arg --> no break */
      retval = tty_check_change(tty);
      if (retval)
	return retval;
      tty_wait_until_sent(tty, 0);
      return 0;
    case TCSBRKP:	/* support for POSIX tcsendbreak() */
      retval = tty_check_change(tty);
      if (retval)
	return retval;
      tty_wait_until_sent(tty, 0);
      return 0;
    case TIOCGSOFTCAR:
      error = verify_area(VERIFY_WRITE, (void *) arg,sizeof(long));
      if (error)
	return error;
      put_fs_long(C_CLOCAL(tty) ? 1 : 0,(ulong *) arg);
      return 0;
    case TIOCSSOFTCAR:
      arg = get_fs_long((ulong *) arg);
      tty->termios->c_cflag =
	((tty->termios->c_cflag & ~CLOCAL) |
	 (arg ? CLOCAL : 0));
      return 0;
    case TIOCMGET:
      error = verify_area(VERIFY_WRITE, (void *) arg,sizeof(uint));
      if (error)
	return error;
      return get_modem_info(info, (uint *) arg);
    case TIOCMBIS:
    case TIOCMBIC:
    case TIOCMSET:
      return set_modem_info(info, cmd, (uint *) arg);
    case TIOCSERGETLSR: /* Get line status register */
      error = verify_area(VERIFY_WRITE, (void *) arg, sizeof(uint));
      if (error)
	return error;
      else
	return get_lsr_info(info, (uint *) arg);
      
    case TIOCGSERIAL:
      return -ENOIOCTLCMD;
#if 0
      error = verify_area(VERIFY_WRITE, (void *) arg,
			  sizeof(struct serial_struct));
      if (error)
	return error;
      return get_serial_info(info,
			     (struct serial_struct *) arg);
#endif
    case TIOCSSERIAL:
      return -ENOIOCTLCMD;
#if 0
      return set_serial_info(info,
			     (struct serial_struct *) arg);
#endif
    case TIOCSERCONFIG:
      return -ENOIOCTLCMD;
#if 0
      return do_autoconfig(info);
#endif
      
    case TIOCSERGWILD:
      return -ENOIOCTLCMD;
#if 0
      error = verify_area(VERIFY_WRITE, (void *) arg,
			  sizeof(int));
      if (error)
	return error;
      put_fs_long(modem_wild_int_mask, (ulong *) arg);
      return 0;
#endif      
    case TIOCSERSWILD:
      return -ENOIOCTLCMD;
#if 0
      if (!suser())
	return -EPERM;
      modem_wild_int_mask = get_fs_long((ulong *) arg);
      if (modem_wild_int_mask < 0)
	modem_wild_int_mask = check_wild_interrupts(0);
      return 0;
#endif      
    case TIOCSERGSTRUCT:
      return -ENOIOCTLCMD;
#if 0
      error = verify_area(VERIFY_WRITE, (void *) arg,
			  sizeof(modem_info));
      if (error)
	return error;
      memcpy_tofs((modem_info *) arg,
		  info, sizeof(modem_info));
      return 0;
#endif      
    default:
#ifdef ISDN_DEBUG_MODEM_IOCTL
      printk(KERN_DEBUG "unsupp. ioctl 0x%08x on ttyi%d\n",cmd,info->line);
#endif
      return -ENOIOCTLCMD;
  }
  return 0;
}

static void
modem_set_termios(struct tty_struct *tty, struct termios *old_termios) {
  modem_info *info = (modem_info *)tty->driver_data;

  if (tty->termios->c_cflag == old_termios->c_cflag)
    return;
  change_speed(info);
  if ((old_termios->c_cflag & CRTSCTS) &&
      !(tty->termios->c_cflag & CRTSCTS)) {
    tty->hw_stopped = 0;
  }
}

/*
 * ------------------------------------------------------------
 * modem_open() and friends
 * ------------------------------------------------------------
 */
static int
block_til_ready(struct tty_struct *tty, struct file * filp, modem_info *info) {
  struct wait_queue wait = { current, NULL };
  int		do_clocal = 0;
  int		retval;
  
  /*
   * If the device is in the middle of being closed, then block
   * until it's done, and then try again.
   */
  if (info->flags & ISDN_ASYNC_CLOSING) {
    interruptible_sleep_on(&info->close_wait);
#ifdef MODEM_DO_RESTART
    if (info->flags & ISDN_ASYNC_HUP_NOTIFY)
      return -EAGAIN;
    else
      return -ERESTARTSYS;
#else
    return -EAGAIN;
#endif
  }
  /*
   * If this is a callout device, then just make sure the normal
   * device isn't being used.
   */
  if (tty->driver.subtype == ISDN_SERIAL_TYPE_CALLOUT) {
    if (info->flags & ISDN_ASYNC_NORMAL_ACTIVE)
      return -EBUSY;
    if ((info->flags & ISDN_ASYNC_CALLOUT_ACTIVE) &&
	(info->flags & ISDN_ASYNC_SESSION_LOCKOUT) &&
	(info->session != current->session))
      return -EBUSY;
    if ((info->flags & ISDN_ASYNC_CALLOUT_ACTIVE) &&
	(info->flags & ISDN_ASYNC_PGRP_LOCKOUT) &&
	(info->pgrp != current->pgrp))
      return -EBUSY;
    info->flags |= ISDN_ASYNC_CALLOUT_ACTIVE;
    return 0;
  }
  /*
   * If non-blocking mode is set, then make the check up front
   * and then exit.
   */
  if (filp->f_flags & O_NONBLOCK) {
    if (info->flags & ISDN_ASYNC_CALLOUT_ACTIVE)
      return -EBUSY;
    info->flags |= ISDN_ASYNC_NORMAL_ACTIVE;
    return 0;
  }
  if (info->flags & ISDN_ASYNC_CALLOUT_ACTIVE) {
    if (info->normal_termios.c_cflag & CLOCAL)
      do_clocal = 1;
  } else {
    if (tty->termios->c_cflag & CLOCAL)
      do_clocal = 1;
  }
  /*
   * Block waiting for the carrier detect and the line to become
   * free (i.e., not in use by the callout).  While we are in
   * this loop, info->count is dropped by one, so that
   * modem_close() knows when to free things.  We restore it upon
   * exit, either normal or abnormal.
   */
  retval = 0;
  add_wait_queue(&info->open_wait, &wait);
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "block_til_ready before block: ttyi%d, count = %d\n",
	 info->line, info->count);
#endif
  info->count--;
  info->blocked_open++;
  while (1) {
    current->state = TASK_INTERRUPTIBLE;
    if (tty_hung_up_p(filp) ||
	!(info->flags & ISDN_ASYNC_INITIALIZED)) {
#ifdef MODEM_DO_RESTART
      if (info->flags & ISDN_ASYNC_HUP_NOTIFY)
	retval = -EAGAIN;
      else
	retval = -ERESTARTSYS;	
#else
      retval = -EAGAIN;
#endif
      break;
    }
    if (!(info->flags & ISDN_ASYNC_CALLOUT_ACTIVE) &&
	!(info->flags & ISDN_ASYNC_CLOSING) &&
	(do_clocal || (
	  dev->mdm.msr[info->line] &
	  UART_MSR_DCD))) {
      break;
    }
    if (current->signal & ~current->blocked) {
      retval = -ERESTARTSYS;
      break;
    }
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "block_til_ready blocking: ttyi%d, count = %d\n",
	   info->line, info->count);
#endif
    schedule();
  }
  current->state = TASK_RUNNING;
  remove_wait_queue(&info->open_wait, &wait);
  if (!tty_hung_up_p(filp))
    info->count++;
  info->blocked_open--;
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "block_til_ready after blocking: ttyi%d, count = %d\n",
	 info->line, info->count);
#endif
  if (retval)
    return retval;
  info->flags |= ISDN_ASYNC_NORMAL_ACTIVE;
  return 0;
}	

/*
 * This routine is called whenever a serial port is opened.  It
 * enables interrupts for a serial port, linking in its async structure into
 * the IRQ chain.   It also performs the serial-specific
 * initialization for the tty structure.
 */
static int
modem_open(struct tty_struct *tty, struct file * filp) {
  modem_info *info;
  int         retval, line;

  line = MINOR(tty->device) - tty->driver.minor_start;
  if (line < 0 || line > ISDN_MAX_CHANNELS)
    return -ENODEV;
  info = &dev->mdm.info[line];
  if (serial_paranoia_check(info, tty->device, "modem_open"))
    return -ENODEV;
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "modem_open %s%d, count = %d\n", tty->driver.name,
	 info->line,info->count);
#endif
  info->count++;
  tty->driver_data = info;
  info->tty        = tty;
  /*
   * Start up serial port
   */
  retval = startup(info);
  if (retval) {
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "modem_open return after startup\n");
#endif
    return retval;
  }
  retval = block_til_ready(tty, filp, info);
  if (retval) {
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "modem_open return after block_til_ready \n");
#endif
    return retval;
  }
  if ((info->count == 1) && (info->flags & ISDN_ASYNC_SPLIT_TERMIOS)) {
    if (tty->driver.subtype == ISDN_SERIAL_TYPE_NORMAL)
      *tty->termios = info->normal_termios;
    else 
      *tty->termios = info->callout_termios;
    change_speed(info);
  }
  info->session = current->session;
  info->pgrp    = current->pgrp;
#ifdef ISDN_DEBUG_MODEM_OPEN
  printk(KERN_DEBUG "modem_open ttyi%d successful...\n", info->line);
#endif
  dev->modempoll++;
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "modem_open normal exit\n");
#endif
  return 0;
}

static void
modem_close(struct tty_struct *tty, struct file *filp) {
  modem_info   *info = (modem_info*)tty->driver_data;
  ulong flags;
  ulong timeout;

  if (!info || serial_paranoia_check(info,tty->device,"modem_close"))
    return;
  save_flags(flags);
  cli();
  if (tty_hung_up_p(filp)) {
    restore_flags(flags);
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "modem_close return after tty_hung_up_p\n");
#endif
    return;
  }
  dev->modempoll--;
  if ((tty->count == 1) && (info->count != 1)) {
    /*
     * Uh, oh.  tty->count is 1, which means that the tty
     * structure will be freed.  Info->count should always
     * be one in these conditions.  If it's greater than
     * one, we've got real problems, since it means the
     * serial port won't be shutdown.
     */
    printk(KERN_ERR "modem_close: bad port count; tty->count is 1, "
	   "info->count is %d\n", info->count);
    info->count = 1;
  }
  if (--info->count < 0) {
    printk(KERN_ERR "modem_close: bad port count for ttyi%d: %d\n",
	   info->line, info->count);
    info->count = 0;
  }
  if (info->count) {
    restore_flags(flags);
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "modem_close after info->count != 0\n");
#endif
    return;
  }
  info->flags |= ISDN_ASYNC_CLOSING;
  /*
   * Save the termios structure, since this port may have
   * separate termios for callout and dialin.
   */
  if (info->flags & ISDN_ASYNC_NORMAL_ACTIVE)
    info->normal_termios = *tty->termios;
  if (info->flags & ISDN_ASYNC_CALLOUT_ACTIVE)
    info->callout_termios = *tty->termios;

  tty->closing = 1;
  
  /*
   * At this point we stop accepting input.  To do this, we
   * disable the receive line status interrupts, and tell the
   * interrupt driver to stop checking the data ready bit in the
   * line status register.
   */
  if (info->flags & ISDN_ASYNC_INITIALIZED) {
    tty_wait_until_sent(tty, 3000); /* 30 seconds timeout */
    /*
     * Before we drop DTR, make sure the UART transmitter
     * has completely drained; this is especially
     * important if there is a transmit FIFO!
     */
    timeout = jiffies+HZ;
    while (!(dev->mdm.mlr[info->line]
	     & UART_LSR_TEMT)) {
      current->state = TASK_INTERRUPTIBLE;
      current->timeout = jiffies + 20;
      schedule();
      if (jiffies > timeout)
	break;
    }
  }
  shutdown(info);
  if (tty->driver.flush_buffer)
    tty->driver.flush_buffer(tty);
  if (tty->ldisc.flush_buffer)
    tty->ldisc.flush_buffer(tty);
  info->tty = 0;
  tty->closing = 0;
#if 00
  if (tty->ldisc.num != ldiscs[N_TTY].num) {
    if (tty->ldisc.close)
      (tty->ldisc.close)(tty);
    tty->ldisc = ldiscs[N_TTY];
    tty->termios->c_line = N_TTY;
    if (tty->ldisc.open)
      (tty->ldisc.open)(tty);
  }
#endif
  if (info->blocked_open) {
    if (info->close_delay) {
      current->state = TASK_INTERRUPTIBLE;
      current->timeout = jiffies + info->close_delay;
      schedule();
    }
    wake_up_interruptible(&info->open_wait);
  }
  info->flags &= ~(ISDN_ASYNC_NORMAL_ACTIVE|ISDN_ASYNC_CALLOUT_ACTIVE|
		   ISDN_ASYNC_CLOSING);
  wake_up_interruptible(&info->close_wait);
  restore_flags(flags);
#ifdef ISDN_DEBUG_MODEM_OPEN
    printk(KERN_DEBUG "modem_close normal exit\n");
#endif
}

/*
 * modem_hangup() --- called by tty_hangup() when a hangup is signaled.
 */
static void
modem_hangup(struct tty_struct *tty) {
  modem_info * info = (modem_info *)tty->driver_data;
	
  if (serial_paranoia_check(info, tty->device, "modem_hangup"))
    return;
  shutdown(info);
  info->count = 0;
  info->flags &= ~(ISDN_ASYNC_NORMAL_ACTIVE|ISDN_ASYNC_CALLOUT_ACTIVE);
  info->tty = 0;
  wake_up_interruptible(&info->open_wait);
}

static void
modem_reset_profile(atemu *m) {
  m->profile[0]  = 0;
  m->profile[1]  = 0;
  m->profile[2]  = 43;
  m->profile[3]  = 13;
  m->profile[4]  = 10;
  m->profile[5]  = 8;
  m->profile[6]  = 3;
  m->profile[7]  = 60;
  m->profile[8]  = 2;
  m->profile[9]  = 6;
  m->profile[10] = 7;
  m->profile[11] = 70;
  m->profile[12] = 0x45;
  m->profile[13] = 0;
  m->profile[14] = ISDN_PROTO_L2_X75I;
  m->profile[15] = ISDN_PROTO_L3_TRANS;
  m->profile[16] = ISDN_SERIAL_XMIT_SIZE / 16;
  m->profile[17] = ISDN_MODEM_WINSIZE;
  m->profile[18] = 7;
  m->profile[19] = 0;
  m->pmsn[0] = '\0';
}

static void
modem_reset_regs(atemu *m, int force) {
  if ((m->mdmreg[12] & 32) || force) {
    memcpy(m->mdmreg,m->profile,ISDN_MODEM_ANZREG);
    memcpy(m->msn,m->pmsn,ISDN_MSNLEN);
  }
  m->mdmcmdl = 0;
}

static void
modem_write_profile(atemu *m) {
  memcpy(m->profile,m->mdmreg,ISDN_MODEM_ANZREG);
  memcpy(m->pmsn,m->msn,ISDN_MSNLEN);
  if (dev->profd)
    send_sig(SIGIO,dev->profd,1);
}

static int
modem_init() {
  modem *m;
  int   i;
  modem_info *info;

  m = &dev->mdm;
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    modem_reset_profile(&m->atmodem[i]);
    modem_reset_regs(&m->atmodem[i],1);
  }
  memset(&m->tty_modem,0,sizeof(struct tty_driver));
  m->tty_modem.magic                = TTY_DRIVER_MAGIC;
  m->tty_modem.name                 = ttyI;
  m->tty_modem.major                = ISDN_TTY_MAJOR;
  m->tty_modem.minor_start          = 0;
  m->tty_modem.num                  = ISDN_MAX_CHANNELS;
  m->tty_modem.type                 = TTY_DRIVER_TYPE_SERIAL;
  m->tty_modem.subtype              = ISDN_SERIAL_TYPE_NORMAL;
  m->tty_modem.init_termios         = tty_std_termios;
  m->tty_modem.init_termios.c_cflag = B9600|CS8|CREAD|HUPCL|CLOCAL;
  m->tty_modem.flags                = TTY_DRIVER_REAL_RAW;
  m->tty_modem.refcount             = &m->refcount;
  m->tty_modem.table                = m->modem_table;
  m->tty_modem.termios              = m->modem_termios;
  m->tty_modem.termios_locked       = m->modem_termios_locked;
  m->tty_modem.open                 = modem_open;
  m->tty_modem.close                = modem_close;
  m->tty_modem.write                = modem_write;
  m->tty_modem.put_char             = NULL;
  m->tty_modem.flush_chars          = modem_flush_chars;
  m->tty_modem.write_room           = modem_write_room;
  m->tty_modem.chars_in_buffer      = modem_chars_in_buffer;
  m->tty_modem.flush_buffer         = modem_flush_buffer;
  m->tty_modem.ioctl                = modem_ioctl;
  m->tty_modem.throttle             = modem_throttle;
  m->tty_modem.unthrottle           = modem_unthrottle;
  m->tty_modem.set_termios          = modem_set_termios;
  m->tty_modem.stop                 = NULL;
  m->tty_modem.start                = NULL;
  m->tty_modem.hangup               = modem_hangup;
  /*
   * The callout device is just like normal device except for
   * major number and the subtype code.
   */
  m->cua_modem             = m->tty_modem;
  m->cua_modem.name        = cui;
  m->cua_modem.major       = ISDN_TTYAUX_MAJOR;
  m->tty_modem.minor_start = 0;
  m->cua_modem.subtype     = ISDN_SERIAL_TYPE_CALLOUT;

  if (tty_register_driver(&m->tty_modem)) {
    printk(KERN_WARNING "isdn: Unable to register modem-device\n");
    return -1;
  }
  if (tty_register_driver(&m->cua_modem)) {
    printk(KERN_WARNING "Couldn't register modem-callout-device\n");
    return -2;
  }
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    info = &(m->info[i]);
    info->magic           = ISDN_ASYNC_MAGIC;
    info->line            = i;
    info->tty             = 0;
    info->close_delay     = 50;
    info->x_char          = 0;
    info->count           = 0;
    info->blocked_open    = 0;
    info->callout_termios = m->cua_modem.init_termios;
    info->normal_termios  = m->tty_modem.init_termios;
    info->open_wait       = 0;
    info->close_wait      = 0;
    info->type            = ISDN_PORT_16550A;
    info->isdn_driver     = -1;
    info->isdn_channel    = -1;
    info->drv_index       = -1;
    info->xmit_size       = ISDN_SERIAL_XMIT_SIZE;
    if (!(info->xmit_buf  = kmalloc(ISDN_SERIAL_XMIT_SIZE+5,GFP_KERNEL))) {
      printk(KERN_ERR "Could not allocate modem xmit-buffer\n");
      return -3;
    }
    info->xmit_buf += 4; /* Make room for T.70 header */
  }
  return 0;
}

/*
 * Low-level-driver register
 */

int
register_isdn(isdn_if* i) {
  driver *d;
  int    n,j,k;
  ulong  flags;
  int    drvidx;

  if (dev->drivers>=ISDN_MAX_DRIVERS) {
    printk(KERN_WARNING "register_isdn: Max. %d drivers supported\n",
	   ISDN_MAX_DRIVERS);
    return 0;
  }
  n = i->channels;
  if (dev->channels+n>=ISDN_MAX_CHANNELS) {
    printk(KERN_WARNING "register_isdn: Max. %d channels supported\n",
	   ISDN_MAX_CHANNELS);
    return 0;
  }
  if (!(d = (driver*)kmalloc(sizeof(driver),GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc driver-struct\n");
    return 0;
  }
  memset((char *)d,0,sizeof(driver));
  if (!(d->rcverr = (int*)kmalloc(sizeof(int)*n,GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc rcverr\n");
    kfree(d);
    return 0;
  }
  memset((char *)d->rcverr,0,sizeof(int)*n);
  if (!(d->rcvcount = (int*)kmalloc(sizeof(int)*n,GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc rcvcount\n");
    kfree(d->rcverr);
    kfree(d);
    return 0;
  }
  memset((char *)d->rcvcount,0,sizeof(int)*n);
  if (!(d->rpqueue = (pqueue**)kmalloc(sizeof(pqueue*)*n,GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc rpqueue\n");
    kfree(d->rcvcount);
    kfree(d->rcverr);
    kfree(d);
    return 0;
  }
  memset((char *)d->rpqueue,0,sizeof(pqueue*)*n);
  if (!(d->rcv_waitq = (struct wait_queue**)
	kmalloc(sizeof(struct wait_queue*)*n,GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc rcv_waitq\n");
    kfree(d->rpqueue);
    kfree(d->rcvcount);
    kfree(d->rcverr);
    kfree(d);
    return 0;
  }
  memset((char *)d->rcv_waitq,0,sizeof(struct wait_queue*)*n);
  if (!(d->snd_waitq = (struct wait_queue**)
	kmalloc(sizeof(struct wait_queue*)*n,GFP_KERNEL))) {
    printk(KERN_WARNING "register_isdn: Could not alloc snd_waitq\n");
    kfree(d->rcv_waitq);
    kfree(d->rpqueue);
    kfree(d->rcvcount);
    kfree(d->rcverr);
    kfree(d);
    return 0;
  }
  memset((char *)d->snd_waitq,0,sizeof(struct wait_queue*)*n);
  d->channels = n;
  d->loaded = 1;
  d->maxbufsize = i->maxbufsize;
  d->pktcount = 0;
  d->stavail = 0;
  d->running = 0;
  d->flags = 0;
  d->interface = i;
  for (drvidx=0;drvidx<ISDN_MAX_DRIVERS;drvidx++)
    if (!dev->drv[drvidx])
      break;
  i->channels = drvidx;
  i->rcvcallb = receive_callback;
  i->statcallb = status_callback;
  if (!strlen(i->id))
    sprintf(i->id,"line%d",drvidx);
  save_flags(flags);
  cli();
  for (j=0;j<n;j++)
    for (k=0;k<ISDN_MAX_CHANNELS;k++)
      if (dev->chanmap[k]<0) {
	dev->chanmap[k] = j;
	dev->drvmap[k] = drvidx;
	break;
      }
  dev->drv[drvidx] = d;
  dev->channels += n;
  strcpy(dev->drvid[drvidx],i->id);
  isdn_info_update();
  dev->drivers++;
  restore_flags(flags);
  return 1;
}


/*
 * Code for raw-networking over ISDN
 */

static void 
isdn_net_reset(struct device *dev) {
  ulong flags;
  
  save_flags(flags);
  cli();			/* Avoid glitch on writes to CMD regs */
  dev->interrupt = 0;
  dev->tbusy = 0;
  restore_flags(flags);
}

/* Open/initialize the board. */
static int
isdn_net_open(struct device *dev) {
  int i;
  struct device *p;

  isdn_net_reset(dev);
  dev->start = 1;
  /* Fill in the MAC-level header. */
  for (i=0; i < ETH_ALEN - sizeof(ulong); i++)
    dev->dev_addr[i] = 0xfc;
  memcpy(&(dev->dev_addr[i]), &dev->pa_addr, sizeof(ulong));
  if ((p = (((isdn_net_local *)dev->priv)->slave))) {
    /* If this interface has slaves, start them also */
    while (p) {
      isdn_net_reset(p);
      p->start = 1;
      p = (((isdn_net_local *)p->priv)->slave);
    }
  }
  MOD_INC_USE_COUNT;
  return 0;
}

/*
 * Assign an ISDN-channel to a net-interface
 */
static void
bind_channel(isdn_net_local *lp, int idx) {
  lp->isdn_device = dev->drvmap[idx];
  lp->isdn_channel = dev->chanmap[idx];
}

/*
 * Perform auto-hangup and cps-calculation for net-interfaces.
 *
 * auto-hangup:
 * Increment idle-counter (this counter is reset on any incoming or
 * outgoing packet), if counter exceeds configured limit either do a
 * hangup immediately or - if configured - wait until just before the next
 * charge-info.
 *
 * cps-calculation (needed for dynamic channel-bundling):
 * Since this function is called every second, simply reset the
 * byte-counter of the interface after copying it to the cps-variable.
 */
static void
isdn_net_autohup() {
  isdn_net_dev *p = dev->netdev;
  ulong flags;

  save_flags(flags);
  cli();
  while (p) {
    isdn_net_local *l = (isdn_net_local *)&(p->local);
    l->cps = l->transcount;
    l->transcount = 0;
    if (dev->net_verbose > 3)
      printk(KERN_DEBUG "%s: %d bogocps\n",l->name,l->cps);
    if ((l->flags & ISDN_NET_CONNECTED) && (!l->dialstate)) {
      l->huptimer++ ;
      if ((l->onhtime) && (l->huptimer > l->onhtime))
	if (l->outgoing) {
	  if (l->hupflags & 4) {
	    if (l->hupflags & 1)
	      isdn_net_hangup(&p->dev);
	    else if (jiffies - l->chargetime > l->chargeint)
	      isdn_net_hangup(&p->dev);
	  } else
	    isdn_net_hangup(&p->dev);
	} else
	  if (l->hupflags & 8)
	    isdn_net_hangup(&p->dev);
    }
    p = (isdn_net_dev *)p->next;
  }
  restore_flags(flags);
}

/*
 * Handle status-messages from ISDN-interfacecard.
 * This function is called from within the main-status-dispatcher
 * status_callback, which itself is called from the lowlevel-driver.
 */
static void
isdn_net_stat_callback(int di, int ch, int cmd) {
  isdn_net_dev *p = dev->netdev;
  ulong flags;

  save_flags(flags);
  cli();
  while (p) {
    if (p->local.isdn_device==di && p->local.isdn_channel==ch)
      switch (cmd) {
	case ISDN_STAT_BSENT:
	  /* A packet has successfully been sent out */
	  if ((p->local.flags & ISDN_NET_CONNECTED) &&
	      (!p->local.dialstate)) {
	    p->local.stats.tx_packets++;
	  }
	  break;
	case ISDN_STAT_DCONN:
	  /* D-Channel is up */
	  if (p->local.dialstate == 4 || p->local.dialstate == 7)
	    p->local.dialstate++;
	  break;
	case ISDN_STAT_DHUP:
	  /* Either D-Channel-hangup or error during dialout */
	  if ((!p->local.dialstate) && (p->local.flags & ISDN_NET_CONNECTED)) {
	    p->local.flags &= ~ISDN_NET_CONNECTED;
	    free_channel(p->local.isdn_device,p->local.isdn_channel,
			 ISDN_USAGE_NET);
#ifdef ISYNCPPP
	    free_ippp(&p->local);
#endif
	    isdn_all_eaz(p->local.isdn_device,p->local.isdn_channel);
	    printk(KERN_INFO "%s: remote hangup\n",p->local.name);
	    printk(KERN_INFO "%s: Chargesum is %d\n",p->local.name,
		   p->local.charge);
	    p->local.isdn_device = -1;
	    p->local.isdn_channel = -1;
          }
	  break;
	case ISDN_STAT_BCONN:
	  /* B-Channel is up */
	  if (p->local.dialstate >= 5 && p->local.dialstate <= 9) {
	    if (p->local.dialstate <= 6) {
	      int i = dc2minor(p->local.isdn_device,p->local.isdn_channel);
	      if (i>=0) {
		dev->usage[i] |= ISDN_USAGE_OUTGOING;
		isdn_info_update();
	      }
	    }
	    p->local.dialstate = 0;
	    printk(KERN_INFO "isdn_net: %s connected\n",p->local.name);
	    /* If first Chargeinfo comes before B-Channel connect,
             * we correct the timestamp here.
             */
	    p->local.chargetime = jiffies;
	  }
	  break;
	case ISDN_STAT_NODCH:
	  /* No D-Channel avail. */
	  if (p->local.dialstate == 4)
	    p->local.dialstate--;
	  break;
	case ISDN_STAT_CINF:
	  /* Charge-info from TelCo. Calculate interval between
           * charge-infos and set timestamp for last info for
           * usage by isdn_net_autohup()
           */
	  p->local.charge++;
	  if (p->local.hupflags & 2) {
	    p->local.hupflags &= ~1;
	    p->local.chargeint = jiffies - p->local.chargetime - (2*HZ);
	  }
	  if (p->local.hupflags & 1)
	    p->local.hupflags |= 2;
	  p->local.chargetime = jiffies;
	  break;
      }
    p = (isdn_net_dev *)p->next;
  }
  restore_flags(flags);
}

/*
 * Check, if a numer contains wilcard-characters, in which case it
 * is for incoming purposes only.
 */
static int
isdn_net_checkwild(char *num) {
  return ((strchr(num,'?')) ||
	  (strchr(num,'*')) ||
	  (strchr(num,'[')) ||
	  (strchr(num,']')) ||
	  (strchr(num,'^')));
}

/*
 * Perform dialout for net-interfaces and timeout-handling for
 * D-Channel-up and B-Channel-up Messages.
 * This function is initially called from within isdn_net_start_xmit() or
 * or isdn_net_find_icall() after initializing the dialstate for an
 * interface. If further calls are needed, the function schedules itself
 * for a timer-callback via isdn_timer_function().
 * The dialstate is also affected by incoming status-messages from
 * the ISDN-Channel which are handled in isdn_net_stat_callback() above.
 */
static void
isdn_net_dial(void) {
  isdn_net_dev *p = dev->netdev;
  int anymore = 0;
  int i;
  isdn_ctrl cmd;

  while (p) {
    switch (p->local.dialstate) {
      case 0:
	/* Nothing to do for this interface */
	break;
      case 1:
	/* Initiate dialout. Set phone-number-pointer to first number
         * of interface.
         */
	p->local.dial = p->local.phone[1];
	anymore = 1;
	p->local.dialstate++;
	break;
	/* Prepare dialing. Clear EAZ, then set EAZ. */
      case 2:
	cmd.driver  = p->local.isdn_device;
	cmd.arg     = p->local.isdn_channel;
	cmd.command = ISDN_CMD_CLREAZ;
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
        sprintf(cmd.num,"%s",map_eaz2msn(p->local.msn,cmd.driver));
	cmd.command = ISDN_CMD_SETEAZ;
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	p->local.dialretry = 0;
	anymore = 1;
	p->local.dialstate++;
	break;
      case 3:
	/* Setup interface, dial current phone-number, switch to next number.
         * If list of phone-numbers is exhausted, increment
         * retry-counter.
         */
	cmd.driver  = p->local.isdn_device;
	cmd.command = ISDN_CMD_SETL2;
	cmd.arg     = p->local.isdn_channel + (p->local.l2_proto << 8);
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	cmd.driver  = p->local.isdn_device;
	cmd.command = ISDN_CMD_SETL3;
	cmd.arg     = p->local.isdn_channel + (p->local.l3_proto << 8);
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	cmd.driver  = p->local.isdn_device;
	cmd.arg     = p->local.isdn_channel;
	p->local.huptimer = 0;
	p->local.outgoing = 1;
	p->local.hupflags |= 1;
	if (!strcmp(p->local.dial->num,"LEASED")) {
	  p->local.dialstate = 4;
	  printk(KERN_INFO "%s: Open leased line ...\n",p->local.name);
	} else {
	  cmd.command = ISDN_CMD_DIAL;
	  sprintf(cmd.num,"%s,%s,7,0",p->local.dial->num,
		  map_eaz2msn(p->local.msn,cmd.driver));
	  i = dc2minor(p->local.isdn_device,p->local.isdn_channel);
	  if (i>=0) {
	    strcpy(dev->num[i],p->local.dial->num);
	    isdn_info_update();
	  }
	  printk(KERN_INFO "%s: dialing %d %s...\n",p->local.name,
		 p->local.dialretry,p->local.dial->num);
	  /*
	   * Switch to next number or back to start if at end of list.
	   */
	  if (!(p->local.dial = (isdn_net_phone *)p->local.dial->next)) {
	    p->local.dial = p->local.phone[1];
	    p->local.dialretry++;
	  }
	  p->local.dtimer = 0;
#ifdef ISDN_DEBUG_NET_DIAL
	  printk(KERN_DEBUG "dial: d=%d c=%d\n",p->local.isdn_device,
		 p->local.isdn_channel);
#endif
	  dev->drv[p->local.isdn_device]->interface->command(&cmd);
	}
	anymore = 1;
	p->local.dialstate++;
	break;
      case 4:
	/* Wait for D-Channel-connect or incoming call, if passive
         * callback configured. If timeout and max retries not
         * reached, switch back to state 3.
         */
	if (p->local.dtimer++ > ISDN_TIMER_DTIMEOUT10)
	  if (p->local.dialretry < p->local.dialmax) {
	    p->local.dialstate = 3;
	  } else
	    isdn_net_hangup(&p->dev);
	    anymore = 1;
	break;
      case 5:
        /* Got D-Channel-Connect, send B-Channel-request */
	cmd.driver  = p->local.isdn_device;
	cmd.arg     = p->local.isdn_channel;
	cmd.command = ISDN_CMD_ACCEPTB;
	anymore = 1;
	p->local.dtimer = 0;
	p->local.dialstate++;
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	break;
      case 6:
        /* Wait for B-Channel-connect. If timeout, switch back to
         * state 3.
         */
#ifdef ISDN_DEBUG_NET_DIAL
	printk(KERN_DEBUG "dialtimer2: %d\n",p->local.dtimer);
#endif
	if (p->local.dtimer++ > ISDN_TIMER_DTIMEOUT10)
	  p->local.dialstate = 3;
	anymore = 1;
	break;
      case 7:
        /* Got incoming Call, setup L2 and L3 protocols, send accept,
	   then wait for D-Channel-connect */
#ifdef ISDN_DEBUG_NET_DIAL
	printk(KERN_DEBUG "dialtimer4: %d\n",p->local.dtimer);
#endif
	cmd.driver  = p->local.isdn_device;
	cmd.command = ISDN_CMD_SETL2;
	cmd.arg     = p->local.isdn_channel + (p->local.l2_proto << 8);
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	cmd.driver  = p->local.isdn_device;
	cmd.command = ISDN_CMD_SETL3;
	cmd.arg     = p->local.isdn_channel + (p->local.l3_proto << 8);
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	if (p->local.dtimer++ > ISDN_TIMER_DTIMEOUT15)
	  isdn_net_hangup(&p->dev);
	else
	  anymore = 1;
	break;
      case 8:
        /* Got incoming D-Channel-Connect, send B-Channel-request */
	cmd.driver  = p->local.isdn_device;
	cmd.arg     = p->local.isdn_channel;
	cmd.command = ISDN_CMD_ACCEPTB;
	dev->drv[p->local.isdn_device]->interface->command(&cmd);
	anymore = 1;
	p->local.dtimer = 0;
	p->local.dialstate++;
	break;
      case 9:
        /*  Wait for B-channel-connect */
#ifdef ISDN_DEBUG_NET_DIAL
	printk(KERN_DEBUG "dialtimer4: %d\n",p->local.dtimer);
#endif
	if (p->local.dtimer++ > ISDN_TIMER_DTIMEOUT10)
	  isdn_net_hangup(&p->dev);
	else
	  anymore = 1;
	break;
      default:
	printk(KERN_WARNING "isdn_net: Illegal dialstate %d for device %s\n",
	       p->local.dialstate,p->local.name);
    }
    p = (isdn_net_dev *)p->next;
  }
  isdn_timer_ctrl(ISDN_TIMER_NETDIAL,anymore);
}

/*
 * Send-data-helpfunction for net-interfaces
 */
static int
isdn_net_send(u_char* buf, int di, int ch, int len) {
  int l;

  if ((l = dev->drv[di]->interface->writebuf(ch,buf,len,0))==len)
    return 1;
  /* Device driver queue full (or packet > 4000 bytes, should never
   * happen)
   */
  if (l==-EINVAL)
    printk(KERN_ERR "isdn_net: Huh, sending pkt too big!\n");
  return 0;
}

/*
 * Perform hangup for a net-interface.
 */
static void
isdn_net_hangup(struct device *d) {
  isdn_net_local *lp = (isdn_net_local *)d->priv;
  isdn_ctrl cmd;
  ulong flags;

  save_flags(flags);
  cli();
  if (lp->flags & ISDN_NET_CONNECTED) {
    printk(KERN_INFO "isdn_net: local hangup %s\n",lp->name);
    lp->dialstate = 0;
    free_channel(lp->isdn_device,lp->isdn_channel,ISDN_USAGE_NET);
#ifdef ISYNCPPP
    free_ippp(lp);
#endif
    lp->flags  &= ~ISDN_NET_CONNECTED;
    cmd.driver  = lp->isdn_device;
    cmd.command = ISDN_CMD_HANGUP;
    cmd.arg     = lp->isdn_channel;
    (void)dev->drv[cmd.driver]->interface->command(&cmd);
    printk(KERN_INFO "%s: Chargesum is %d\n",lp->name,lp->charge);
    isdn_all_eaz(lp->isdn_device,lp->isdn_channel);
    lp->isdn_device = -1;
    lp->isdn_channel = -1;
  }
  restore_flags(flags);
}

#ifdef ISDN_LOG_PACKET
typedef struct {
  unsigned short source;
  unsigned short dest;
} ip_ports;

static void
log_packet(u_char *buf, isdn_net_local *lp, int mode) {
  int data_ofs = ((buf[0] & 15)*4);
  ip_ports *ipp;
  char addinfo[100];

  switch (mode) {
    case 0:
      /* Open a Connection */
      addinfo[0] = '\0';
      switch (buf[9]) {
	case 1:
	  strcpy(addinfo," ICMP");
	  break;
	case 2:
	  strcpy(addinfo," IGMP");
	  break;
	case 4:
	  strcpy(addinfo," IPIP");
	  break;
	case 6:
	  ipp = (ip_ports*)(&buf[data_ofs]);
	  sprintf(addinfo," TCP, port: %d -> %d",ntohs(ipp->source),
		  ntohs(ipp->dest));
	  break;
	case 8:
	  strcpy(addinfo," EGP");
	  break;
	case 12:
	  strcpy(addinfo," PUP");
	  break;
	case 17:
	  ipp = (ip_ports*)(&buf[data_ofs]);
	  sprintf(addinfo," UDP, port: %d -> %d",ntohs(ipp->source),
		  ntohs(ipp->dest));
	  break;
	case 22:
	  strcpy(addinfo," IDP");
	  break;
      }
      printk(KERN_INFO "OPEN: %d.%d.%d.%d -> %d.%d.%d.%d%s\n",
	     buf[12],buf[13],buf[14],buf[15],
	     buf[16],buf[17],buf[18],buf[19],
	     addinfo);
      break;
    case 1:
      /* Send a packet */
      break;
    case 2:
      /* Receive a packet */
      break;
  } /* case mode ... */
}
#else
#define log_packet(b,lp,mode) ()
#endif

#define SEND_AND_FREE_PACKET {                                           \
  lp->transcount += pktlen;                                              \
  if (isdn_net_send(&buf[bo],lp->isdn_device,lp->isdn_channel,pktlen)) { \
    ndev->tbusy  = 0;                                                    \
    dev_kfree_skb(skb, FREE_WRITE);                                      \
  }                                                                      \
}

/*
 * Try sending a packet. Strip off MAC-Header if using raw-IP.
 * If this interface isn't connected to a ISDN-Channel, find a free channel,
 * and start dialing.
 */
static int
isdn_net_start_xmit(struct sk_buff *skb, struct device *ndev) {
  isdn_net_local *lp = (isdn_net_local *)ndev->priv;
  int bo = 0;

  if (ndev->tbusy) {
    if (jiffies - ndev->trans_start < 20) {
      return 1;
    }
    lp->stats.tx_errors++;
    ndev->tbusy = 0;
    ndev->trans_start = jiffies;
  }
  if (skb == NULL) {
    dev_tint(ndev);
    return 0;
  }

  /* Avoid timer-based retransmission conflicts. */
  if (set_bit(0, (void*)&ndev->tbusy) != 0)
    printk(KERN_WARNING "%s: Transmitter access conflict.\n", ndev->name);
  else {
    u_char *buf = skb->data;
    int pktlen  = skb->len;
#ifdef ISDN_DEBUG_NET_DUMP
    isdn_dumppkt("S:",buf,pktlen,40);
#endif
    if (!(lp->flags & ISDN_NET_CONNECTED)) {
      int  chi;
      if (lp->phone[1]) {
	ulong flags;
	save_flags(flags);
	cli();
	/* Grab a free ISDN-Channel */
	if ((chi = get_free_channel(ISDN_USAGE_NET,lp->l2_proto,
				    lp->l3_proto,
				    lp->pre_device,
				    lp->pre_channel))<0) {
	  printk(KERN_WARNING "isdn_net: No channel for %s\n",ndev->name);
	  restore_flags(flags);
	  return 1;
	}
	log_packet(&buf[14],lp,0);
	lp->dialstate = 1;
	lp->flags |= ISDN_NET_CONNECTED;
	/* Connect interface with channel */
	bind_channel(lp,chi);
#ifdef ISYNCPPP
        if(bind_ippp(lp) < 0) {
          printk(KERN_WARNING "isdn_net: Can't find usable ippp device.\n");
          free_channel(lp->isdn_device,lp->isdn_channel,ISDN_USAGE_NET);
          return 1;
        }
#endif
	/* Initiate dialing */
	isdn_net_dial();
	restore_flags(flags);
      } else {
	printk(KERN_WARNING "isdn_net: No phone number for %s\n",ndev->name);
	return 1;
      }
    } else {
      ndev->trans_start = jiffies;

      if (!lp->dialstate) {
	/* Connection is established, try sending */
	switch (lp->p_encap) {
	  case ISDN_NET_ENCAP_ETHER:
	    /* Ethernet over ISDN (no stripping) */
	    bo = 0;
	    break;
	  case ISDN_NET_ENCAP_RAWIP:
	    /* RAW-IP without MAC-Header (strip off 14 bytes) */
	    bo = 14;
	    break;
	  case ISDN_NET_ENCAP_IPTYP:
	    /* IP with type field (strip of MAC addresses) */
	    bo = 12;
	    break;
	  case ISDN_NET_ENCAP_CISCOHDLC:
	    /* CISCO-HDLC IP with type field and  fake I-frame-header */
	    bo = 10;
	    /* The I-frame: */
	    buf[bo] = 0x0f;
	    buf[11] = 0;
	    break;
#ifdef ISYNCPPP
          case ISDN_NET_ENCAP_SYNCPPP:
	    /* PPP header, raw IP with no compression */
            bo = 10;
            buf[bo] = 0xff;    
            buf[bo+1] = 0x03;
            buf[bo+2] = 0x00;
            buf[bo+3] = 0x21;
            if(!ippp_state(lp->ppp_minor)) { /* PPP connected ? */
              lp->huptimer = 0;
              printk(KERN_WARNING "isdn_ppp: Packet blocked: %08lx %d %d\n",
		     (long) lp, lp->isdn_device, lp->isdn_channel);
              return 1; 
            }
            break;
#endif
	}
	log_packet(&buf[14],lp,1);
	/* Reset hangup-timeout */
	lp->huptimer = 0;
	pktlen -= bo;
	if (lp->cps>7000) {
	  /* Device overloaded */

	  /* Packet-delivery via round-robin over master and all connected
           * slaves.
           */
	  if (lp->master)
	    /* Slaves always deliver themselves */
	    SEND_AND_FREE_PACKET
	  else {
	    isdn_net_local *slp = (isdn_net_local*)(lp->srobin->priv);
	    /* Master delivers via srobin and maintains srobin */
	    if (lp->srobin == ndev)
	      SEND_AND_FREE_PACKET
	    else
	      ndev->tbusy = isdn_net_start_xmit(skb,lp->srobin);
	    lp->srobin = (slp->slave)?slp->slave:ndev;
	    slp = (isdn_net_local*)(lp->srobin->priv);
	    if (!((slp->flags & ISDN_NET_CONNECTED) && (slp->dialstate==0)))
	      lp->srobin = ndev;
	  }
	  /* Slave-startup using delay-variable */
	  if (lp->slave) {
	    if (!lp->sqfull) {
	      /* First time overload: set timestamp only */
	      lp->sqfull = 1;
	      lp->sqfull_stamp = jiffies;
	    } else {
	      /* subsequent overload: if slavedelay exceeded, start dialing */
	      if ((jiffies-lp->sqfull_stamp)>lp->slavedelay)
		isdn_net_force_dial_lp((isdn_net_local*)lp->slave->priv);
	    }
	  }
	} else {
	  /* Not overloaded, deliver locally */
	  SEND_AND_FREE_PACKET
	  if (lp->sqfull && ((jiffies-lp->sqfull_stamp)>(lp->slavedelay*3)))
	    lp->sqfull = 0;
	}
      } else
	return 1;
    }
  }
  return ndev->tbusy;
}

/*
 * Shutdown a net-interface.
 */
static int
isdn_net_close(struct device *dev) {
  struct device *p;

  dev->tbusy = 1;
  dev->start = 0;
  isdn_net_hangup(dev);
  if ((p = (((isdn_net_local *)dev->priv)->slave))) {
    /* If this interface has slaves, stop them also */
    while (p) {
      isdn_net_hangup(p);
      p->tbusy = 1;
      p->start = 0;
      p = (((isdn_net_local *)p->priv)->slave);
    }
  }
  MOD_DEC_USE_COUNT;
  return 0;
}

/*
 * Get statistics
 */
static struct enet_statistics *
isdn_net_get_stats(struct device *dev) {
  isdn_net_local *lp = (isdn_net_local *)dev->priv;
  return &lp->stats;
}

/* Got a packet from ISDN-Channel.
 * Add fake MAC-Header if using RAW-IP, allocate a new sk_buff.and pass it
 * to the upper layers.
 */
static void
isdn_net_receive(struct device *dev, u_char *buf, int pkt_len) {
  isdn_net_local *lp = (isdn_net_local *)dev->priv;
  struct sk_buff *skb;
  int bo = 0;
#ifdef ISYNCPPP
  int b;
#endif

  lp->transcount += pkt_len;
  lp->stats.rx_packets++;
  lp->huptimer = 0;
  if (lp->master) {
    /* Bundling: If device is a slave-device, deliver to master, also
     * handle master's statistics and hangup-timeout
     */
    /* Reset hangup-timer */
    dev = lp->master;
    lp  = (isdn_net_local *)dev->priv;
    lp->stats.rx_packets++;
    lp->huptimer = 0;
  }
  switch (lp->p_encap) {
    case ISDN_NET_ENCAP_ETHER:
      /* Ethernet over ISDN (no adding) */
      bo = 0;
      break;
    case ISDN_NET_ENCAP_RAWIP:
      /* RAW-IP without MAC-Header (add fake header 14 bytes) */
      bo = 14;
      break;
    case ISDN_NET_ENCAP_IPTYP:
      /* IP with type field (add 12 byte "MAC addresses") */
      bo = 12;
      break;
    case ISDN_NET_ENCAP_CISCOHDLC:
      /* CISCO-HDLC IP with type field and  fake I-frame-header */
      bo = 10; /* I-Frame get's overwritten below */
      break;
#ifdef ISYNCPPP
    case ISDN_NET_ENCAP_SYNCPPP:
      if(buf[0] == 0xff && buf[1] == 0x03)
        b = 2;
      else
        b = 0;
      if(buf[b] == 0x00 &&
	 (buf[b+1] == 0x21 || buf[b+1] == 0x2d || buf[b+1] == 0x2f)) {
        buf+=2+b;
        pkt_len -= 2+b;
        bo=14;   /* ip packet RAW/VJ/.. */
      } else {
	/* push data to pppd device */
        ippp_fill_rq(buf,pkt_len,lp->ppp_minor); 
        return;
      }
      break;
#endif
  }
  ALLOC_SKB(skb,pkt_len+bo);
  if (skb == NULL) {
    printk(KERN_WARNING "%s: Memory squeeze, dropping packet.\n", dev->name);
    lp->stats.rx_dropped++;
    return;
  } else {
    INIT_SKB(skb,pkt_len+bo,dev);
    memcpy(&skb->data[bo],buf,pkt_len);
    if (lp->p_encap) {
      struct ethhdr *eth = (struct ethhdr *)skb->data;
      int i;
      switch (lp->p_encap) {
        case ISDN_NET_ENCAP_SYNCPPP:
	case ISDN_NET_ENCAP_RAWIP:
	  /* RAW-IP: Insert fake MAC Typefield */
	  eth->h_proto = htons(ETH_P_IP);
	  /* fall through */
	case ISDN_NET_ENCAP_IPTYP:
	case ISDN_NET_ENCAP_CISCOHDLC:
	  /* IP: Insert fake MAC Adresses */
	  for (i=0;i<ETH_ALEN;i++)
	    eth->h_source[i] = 0xfc;
	  memcpy(&(eth->h_dest[0]), dev->dev_addr, ETH_ALEN);
	  break;
      }
    }
    SKB_PROTO(skb,dev);
#ifdef ISDN_DEBUG_NET_DUMP
    isdn_dumppkt("R:",skb->data,pkt_len+bo,40);
#endif
    log_packet(&skb->data[14],lp,2);
    netif_rx(skb);
    lp->stats.rx_packets++;
    /* Reset hangup-timer */
    lp->huptimer = 0;
  }
  return;
}

/*
 * A packet arrived via ISDN. Search interface-chain for a corresponding
 * interface. If found, deliver packet to receiver-function and return 1,
 * else return 0.
 */
static int
isdn_net_receive_callback(int di, int ch, u_char *buf, int len) {
  isdn_net_dev *p = dev->netdev;

  while (p) {
    isdn_net_local *lp = &p->local;
    if ((lp->isdn_device==di) &&
	(lp->isdn_channel==ch) &&
	(lp->flags & ISDN_NET_CONNECTED) &&
	(!lp->dialstate)) {
      isdn_net_receive(&p->dev,buf,len);
      return 1;
    }
    p = (isdn_net_dev *)p->next;
  }
  return 0;
}

/* We don't need to send arp, because we have point-to-point connections. */
static int
isdn_net_rebuild_header(void *buff, struct device *dev, ulong dst,
		    struct sk_buff *skb) {
    struct ethhdr *eth = (struct ethhdr *)buff;
    int i;

    if (eth->h_proto != htons(ETH_P_IP)) {
#ifdef ISDN_DEBUG_NET_BUILDHDR
	printk(KERN_DEBUG "isdn_net_rebuild_header: eth->proto = %04x ?\n",
	       eth->h_proto);
#endif
	memcpy(eth->h_source, dev->dev_addr, dev->addr_len);
	return 0;
    }
    for (i=0; i < ETH_ALEN - sizeof(ulong); i++)
	eth->h_dest[i] = 0xfc;
    memcpy(&(eth->h_dest[i]), &dst, sizeof(ulong));
    return 0;
}

/*
 * Interface-setup. (called just after registering a new interface)
 */
static int
isdn_net_init(struct device *dev) {
  if (dev == NULL) {
    printk(KERN_WARNING "isdn_net_init: dev = NULL!\n");
    return -ENODEV;
  }
  if (dev->priv == NULL) {
    printk(KERN_WARNING "isdn_net_init: dev->priv = NULL!\n");
    return -ENODEV;
  }
  /* Setup the generic properties */
  ether_setup(dev);
  /* The ISDN-specific entries in the device structure. */
  dev->open = &isdn_net_open;
  dev->hard_start_xmit = &isdn_net_start_xmit;
  dev->stop = &isdn_net_close;
  dev->get_stats = &isdn_net_get_stats;
  dev->rebuild_header = &isdn_net_rebuild_header;
  return 0;
}

/*
I picked the pattern-matching-functions from an old GNU-tar version (1.10)
It was originaly written and put to PD by rs@mirror.TMC.COM (Rich Salz)
*/

static int
Star(char *s, char *p) {
  while (wildmat(s, p) == 0)
    if (*++s == '\0')
      return(0);
  return(1);
}

/* Shell-type Pattern-matching for incoming caller-Ids
 * This function gets a string in s and checks, if it matches the pattern
 * given in p. It returns 1 on success, 0 otherwise.
 *
 * Posible Patterns:
 *
 * '?'     matches one character
 * '*'     matches zero or more characters
 * [xyz]   matches the set of charcters in brackets.
 * [^xyz]  matches any single character not in the set of characters
 */

static int
wildmat(char *s, char *p) {
  register int	 last;
  register int 	 matched;
  register int 	 reverse;

  for ( ; *p; s++, p++)
    switch (*p) {
      case '\\':
	/* Literal match with following character; fall through. */
	p++;
      default:
	if (*s != *p)
	  return(0);
	continue;
      case '?':
	/* Match anything. */
	if (*s == '\0')
	  return(0);
	continue;
      case '*':
	/* Trailing star matches everything. */
	return(*++p ? Star(s, p) : 1);
      case '[':
	/* [^....] means inverse character class. */
	if ((reverse = (p[1] == '^')))
	  p++;
	for (last = 0, matched = 0; *++p && (*p != ']'); last = *p)
	  /* This next line requires a good C compiler. */
	  if (*p == '-' ? *s <= *++p && *s >= last : *s == *p)
	    matched = 1;
	if (matched == reverse)
	  return(0);
	continue;
    }
  return(*s == '\0');
}

static void
isdn_net_swapbind(int drvidx) {
  isdn_net_dev *p;

#ifdef ISDN_DEBUG_NET_ICALL
  printk(KERN_DEBUG "n_fi: swapping ch of %d\n",drvidx);
#endif
  p = dev->netdev;
  while (p) {
    if (p->local.pre_device==drvidx)
      switch (p->local.pre_channel) {
        case 0:
	  p->local.pre_channel = 1;
	  break;
	case 1:
	  p->local.pre_channel = 0;
	  break;
      }
    p = (isdn_net_dev*)p->next;
  }
}

static int
my_atoi(char *s) {
  int i,n;
  
  n = 0;
  if (!s)
    return -1;
  for (i=0;*s>='0' && *s<='9';i++,s++)
    n = 10*n+(*s - '0');
  return n;
}

static void
isdn_net_swap_usage(int i1, int i2) {
  int u1 = dev->usage[i1] & ISDN_USAGE_EXCLUSIVE;
  int u2 = dev->usage[i2] & ISDN_USAGE_EXCLUSIVE;

#ifdef ISDN_DEBUG_NET_ICALL
  printk(KERN_DEBUG "n_fi: usage of %d and %d\n",i1,i2);
#endif
  dev->usage[i1] &= ~ISDN_USAGE_EXCLUSIVE;
  dev->usage[i1] |= u2;
  dev->usage[i2] &= ~ISDN_USAGE_EXCLUSIVE;
  dev->usage[i2] |= u1;
  isdn_info_update();
}

/*
 * An incoming call-request has arrived.
 * Search the interface-chain for an aproppriate interface.
 * If found, connect the interface to the ISDN-channel and initiate
 * D- and B-Channel-setup. If secure-flag is set, accept only
 * configured phone-numbers. If callback-flag is set initiate
 * callback-dialing.
 *
 * Return-Value: 0 = No appropriate interface for this call.
 *               1 = Call accepted
 *               2 = Do callback
 */
static int
isdn_net_find_icall(int di, int ch, int idx, char *num) {
  char *eaz;
  int si1;
  int si2;
  int ematch;
  int swapped;
  int sidx = 0;
  isdn_net_dev *p;
  isdn_net_phone *n;
  ulong flags;
  char nr[31];
  char *s;

  /* Search name in netdev-chain */
  save_flags(flags);
  cli();
  if (num[0] == ',') {
    nr[0] = '0';
    strncpy(&nr[1],num,30);
    printk(KERN_WARNING "isdn_net: Incoming call without OAD, assuming '0'\n");
  } else
    strncpy(nr,num,30);
  s = strtok(nr,",");
  s = strtok(NULL,",");
  if (!s) {
    printk(KERN_WARNING "isdn_net: Incoming callinfo garbled, ignored: %s\n",
	   num);
    restore_flags(flags);
    return 0;
  }
  si1 = my_atoi(s);
  s = strtok(NULL,",");
  if (!s) {
    printk(KERN_WARNING "isdn_net: Incoming callinfo garbled, ignored: %s\n",
	   num);
    restore_flags(flags);
    return 0;
  }
  si2 = my_atoi(s);
  eaz = strtok(NULL,",");
  if (!eaz) {
    printk(KERN_WARNING "isdn_net: Incoming call without CPN, assuming '0'\n");
    eaz = "0";
  }
  if (dev->net_verbose > 1)
    printk(KERN_INFO "isdn_net: call from %s,%d,%d -> %s\n",nr,si1,si2,eaz);
  /* Accept only calls with Si1 = 7 (Data-Transmission) */
  if (si1!=7) {
    if (dev->net_verbose > 1)
      printk(KERN_INFO "isdn_net: Service-Indicator not 7, ignored\n");
    return 0;
  }
  n = (isdn_net_phone *)0;
  p = dev->netdev;
  ematch = 0;
#ifdef ISDN_DEBUG_NET_ICALL
  printk(KERN_DEBUG "n_fi: di=%d ch=%d idx=%d usg=%d\n",di,ch,idx,
         dev->usage[idx]);
#endif
  swapped = 0;
  while (p) {
    /* If last check has trigered as binding-swap, revert it */
    switch (swapped) {
      case 2:
        isdn_net_swap_usage(idx,sidx);
        /* fall through */
      case 1:
        isdn_net_swapbind(di);
        break;
    }
    swapped = 0;
    if (!strcmp(map_eaz2msn(p->local.msn,di),eaz))
      ematch = 1;
#ifdef ISDN_DEBUG_NET_ICALL
    printk(KERN_DEBUG "n_fi: if='%s', l.msn=%s, l.flags=%d, l.dstate=%d\n",
	   p->local.name,p->local.msn,p->local.flags,p->local.dialstate);
#endif
    if ((!strcmp(map_eaz2msn(p->local.msn,di),eaz)) &&  /* EAZ is matching   */
	(((!(p->local.flags & ISDN_NET_CONNECTED)) &&   /* but not connected */
	  (USG_NONE(dev->usage[idx]))) ||               /* and ch. unused or */
	 (((p->local.dialstate == 4) &&                 /* if dialing        */
	   (!(p->local.flags & ISDN_NET_CALLBACK)))     /* but no callback   */
	   ))) {
#ifdef ISDN_DEBUG_NET_ICALL
      printk(KERN_DEBUG "n_fi: match1, pdev=%d pch=%d\n",
	     p->local.pre_device,p->local.pre_channel);
#endif
      if (dev->usage[idx] & ISDN_USAGE_EXCLUSIVE) {
	if ((p->local.pre_channel!=ch) ||
	   (p->local.pre_device!=di)) {
	  /* Here we got a problem:
	     If using an ICN-Card, an incoming call is always signaled on
             on the first channel of the card, if both channels are
             down. However this channel may be bound exclusive. If the
             second channel is free, this call should be accepted.
	     The solution is horribly but it runs, so what:
	     We exchange the exclusive bindings of the two channels, the
	     corresponding variables in the interface-structs.
	  */
	  if (ch==0) {
	    sidx = dc2minor(di,1);
#ifdef ISDN_DEBUG_NET_ICALL
	    printk(KERN_DEBUG "n_fi: ch is 0\n");
#endif
	    if (USG_NONE(dev->usage[sidx])) {
	      /* Second Channel is free, now see if it is bound
                 exclusive too. */
	      if (dev->usage[sidx] & ISDN_USAGE_EXCLUSIVE) {
#ifdef ISDN_DEBUG_NET_ICALL
		printk(KERN_DEBUG "n_fi: 2nd channel is down and bound\n");
#endif
		/* Yes, swap bindings only, if the original
		   binding is bound to channel 1 of this driver */
		if ((p->local.pre_device==di) &&
		    (p->local.pre_channel==1)   ) {
		  isdn_net_swapbind(di);
		  swapped = 1;
		} else {
		  /* ... else iterate next device */
		  p = (isdn_net_dev*)p->next;
		  continue;
		}
	      } else {
#ifdef ISDN_DEBUG_NET_ICALL
		printk(KERN_DEBUG "n_fi: 2nd channel is down and unbound\n");
#endif
		/* No, swap always and swap excl-usage also */
		isdn_net_swap_usage(idx,sidx);
		isdn_net_swapbind(di);
		swapped = 2;
	      }
	      /* Now check for exclusive binding again */
#ifdef ISDN_DEBUG_NET_ICALL
	      printk(KERN_DEBUG "n_fi: final check\n");
#endif
	      if ((dev->usage[idx] & ISDN_USAGE_EXCLUSIVE) &&
		  ((p->local.pre_channel!=ch) ||
		   (p->local.pre_device!=di))) {
#ifdef ISDN_DEBUG_NET_ICALL
		printk(KERN_DEBUG "n_fi: final check failed\n");
#endif
		p = (isdn_net_dev*)p->next;
		continue;
	      }
	    }
	  } else {
	    /* We are already on the second channel, so nothing to do */
#ifdef ISDN_DEBUG_NET_ICALL
	    printk(KERN_DEBUG "n_fi: already on 2nd channel\n");
#endif
	    p = (isdn_net_dev*)p->next;
	    continue;
	  }
	}
      }
#ifdef ISDN_DEBUG_NET_ICALL
  printk(KERN_DEBUG "n_fi: match2\n");
#endif
      n = p->local.phone[0];
      if (p->local.flags & ISDN_NET_SECURE) {
	while (n) {
	  if (wildmat(nr,n->num))
	    break;
	  n = (isdn_net_phone*)n->next;
	}
      }
      if (n || (!(p->local.flags & ISDN_NET_SECURE))) {
	isdn_net_local *lp = &(p->local);
#ifdef ISDN_DEBUG_NET_ICALL
	printk(KERN_DEBUG "n_fi: match3\n");
#endif
	/* Here we got an interface matched, now see if it is up.
         * If not, reject the call actively.
	 */
	if (!p->dev.start) {
	  restore_flags(flags);
	  printk(KERN_INFO "%s: incoming call, if down -> rejected\n",
		 lp->name);
	  return 3;
	}
	/* Interface is up, now see if it's a slave. If so, see if
         * it's master and parent slave is online. If not, reject the call.
	 */
	if (lp->master) {
	  isdn_net_local *mlp = (isdn_net_local*)lp->master->priv;
	  printk(KERN_DEBUG "ICALLslv: %s\n",lp->name);
	  printk(KERN_DEBUG "master=%s\n",mlp->name);
	  if (mlp->flags & ISDN_NET_CONNECTED) {
	    printk(KERN_DEBUG "master online\n");
	    /* Master is online, find parent-slave (master if first slave) */
	    while (mlp->slave) {
	      if ((isdn_net_local*)mlp->slave->priv == lp)
		break;
	      mlp = (isdn_net_local*)mlp->slave->priv;
	    }
	  } else
	    printk(KERN_DEBUG "master offline\n");
	  /* Found parent, if it's offline iterate next device */
	  printk(KERN_DEBUG "mlpf: %d\n",mlp->flags & ISDN_NET_CONNECTED);
	  if (!(mlp->flags & ISDN_NET_CONNECTED)) {
	    p = (isdn_net_dev*)p->next;
	    continue;
	  }
	}
	if (lp->flags & ISDN_NET_CALLBACK) {
	  int  chi;
	  printk(KERN_DEBUG "%s: call from %s -> %s, start callback\n",
		 lp->name,nr,eaz);
	  if (lp->phone[1]) {
	    /* Grab a free ISDN-Channel */
	    if ((chi = get_free_channel(ISDN_USAGE_NET,lp->l2_proto,
					lp->l3_proto,
					lp->pre_device,
					lp->pre_channel))<0) {
	      printk(KERN_WARNING "isdn_net: No channel for %s\n",lp->name);
	      restore_flags(flags);
	      return 0;
	    }
	    /* Setup dialstate. */
	    lp->dialstate = 1;
	    lp->flags |= ISDN_NET_CONNECTED;
	    /* Connect interface with channel */
	    bind_channel(lp,chi);
#ifdef ISYNCPPP
            if(bind_ippp(lp)<0) {
              printk(KERN_WARNING "isdn_net: Can't find usable ippp device.\n");
              free_channel(p->local.isdn_device,p->local.isdn_channel,
			   ISDN_USAGE_NET);
	      restore_flags(flags);
              return 0;
            }
#endif
	    /* Initiate dialing by returning 2 */
	    restore_flags(flags);
	    return 2;
	  } else
	    printk(KERN_WARNING "isdn_net: %s: No phone number\n",lp->name);
	  restore_flags(flags);
	  return 0;
	} else {
	  printk(KERN_DEBUG "%s: call from %s -> %s accepted\n",lp->name,nr,
		 eaz);
#ifdef ISYNCPPP
	  if (p->local.isdn_device != -1) {
	    free_channel(p->local.isdn_device,p->local.isdn_channel,
			 ISDN_USAGE_NET);
            free_ippp(&p->local);
          }
#endif
	  /* if this interface is dialing, it does it probably on a different
	     device, so free this device */
	  if (p->local.dialstate == 4)
	    free_channel(p->local.isdn_device,p->local.isdn_channel,
			 ISDN_USAGE_NET);
	  dev->usage[idx] &= ISDN_USAGE_EXCLUSIVE;
	  dev->usage[idx] |= ISDN_USAGE_NET;
	  strcpy(dev->num[idx],nr);
	  isdn_info_update();
	  p->local.isdn_device = di;
	  p->local.isdn_channel = ch;
          p->local.ppp_minor = -1;
	  p->local.flags |= ISDN_NET_CONNECTED;
	  p->local.dialstate = 7;
	  p->local.dtimer = 0;
	  p->local.outgoing = 0;
	  p->local.huptimer = 0;
	  p->local.hupflags |= 1;
	  restore_flags(flags);
	  return 1;
	}
      }
    }
    p = (isdn_net_dev*)p->next;
  }
  /* If none of configured EAZ/MSN matched and not verbose, be silent */
  if (ematch || dev->net_verbose) 
    printk(KERN_INFO "isdn_net: call from %s -> %d %s ignored\n",nr,di,eaz);
  restore_flags(flags);
  return 0;
}

/*
 * An incoming call-request has arrived.
 * Search the tty-devices for an aproppriate device and bind
 * it to the ISDN-Channel.
 * Return Index to dev->mdm or -1 if none found.
 */
static int
isdn_modem_find_icall(int di, int ch, char *num) {
  char *eaz;
  int i;
  int idx;
  int si1;
  int si2;
  char *s;
  char nr[31];
  ulong flags;

  save_flags(flags);
  cli();
  if (num[0] == ',') {
    nr[0] = '0';
    strncpy(&nr[1],num,29);
    printk(KERN_WARNING "isdn: Incoming call without OAD, assuming '0'\n");
  } else
    strncpy(nr,num,30);
  s = strtok(nr,",");
  s = strtok(NULL,",");
  if (!s) {
    printk(KERN_WARNING "isdn: Incoming callinfo garbled, ignored: %s\n",
	   num);
    restore_flags(flags);
    return -1;
  }
  si1 = my_atoi(s);
  s = strtok(NULL,",");
  if (!s) {
    printk(KERN_WARNING "isdn: Incoming callinfo garbled, ignored: %s\n",
	   num);
    restore_flags(flags);
    return -1;
  }
  si2 = my_atoi(s);
  eaz = strtok(NULL,",");
  if (!eaz) {
    printk(KERN_WARNING "isdn: Incoming call without CPN, assuming '0'\n");
    eaz ="0";
  }
#ifdef ISDN_DEBUG_MODEM_ICALL
  printk(KERN_DEBUG "m_fi: eaz=%s si1=%d si2=%d\n",eaz,si1,si2);
#endif
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
#ifdef ISDN_DEBUG_MODEM_ICALL
    printk(KERN_DEBUG "m_fi: i=%d msn=%s mmsn=%s mreg18=%d mreg19=%d\n",i,
	   dev->mdm.atmodem[i].msn,map_eaz2msn(dev->mdm.atmodem[i].msn,di),
	   dev->mdm.atmodem[i].mdmreg[18],dev->mdm.atmodem[i].mdmreg[19]);
#endif
    if ((!strcmp(map_eaz2msn(dev->mdm.atmodem[i].msn,di)
		 ,eaz)) &&                            /* EAZ is matching   */
        (dev->mdm.atmodem[i].mdmreg[18] == si1) &&    /* SI1 is matching   */
        (dev->mdm.atmodem[i].mdmreg[19] == si2)   ) { /* SI2 is matching   */
      modem_info *info = &dev->mdm.info[i];
      idx = dc2minor(di,ch);
#ifdef ISDN_DEBUG_MODEM_ICALL
      printk(KERN_DEBUG "m_fi: match1\n");
      printk(KERN_DEBUG "m_fi: idx=%d flags=%08lx drv=%d ch=%d usg=%d\n",idx,
	     info->flags,info->isdn_driver,info->isdn_channel,
	     dev->usage[idx]);
#endif
      if ((info->flags & ISDN_ASYNC_NORMAL_ACTIVE) &&
	  (info->isdn_driver == -1) &&
	  (info->isdn_channel == -1) &&
	  (USG_NONE(dev->usage[idx]))) {
	info->isdn_driver  = di;
	info->isdn_channel = ch;
	info->drv_index    = idx;
	dev->m_idx[idx]    = info->line;
	dev->usage[idx]    &= ISDN_USAGE_EXCLUSIVE;
	dev->usage[idx]    |= ISDN_USAGE_MODEM;
	strcpy(dev->num[idx],nr);
	isdn_info_update();
	restore_flags(flags);
	printk(KERN_INFO "isdn_tty: call from %s, -> RING on ttyI%d\n",nr,
	       info->line);
	return info->line;
      }
    }
  }
  printk(KERN_INFO "isdn_tty: call from %s -> %s %s\n",nr,eaz,
	 dev->drv[di]->reject_bus?"rejected":"ignored");
  restore_flags(flags);
  return -1;
}

/*
 * Search list of net-interfaces for an interface with given name.
 */
static isdn_net_dev*
isdn_net_findif(char *name) {
  isdn_net_dev *p = dev->netdev;

  while (p) {
    if (!strcmp(p->local.name,name))
      return p;
    p = (isdn_net_dev*)p->next;
  }
  return (isdn_net_dev *)NULL;
}

/*
 * Force a net-interface to dial out.
 * This is called from the userlevel-routine below or
 * from isdn_net_start_xmit().
 */
static int
isdn_net_force_dial_lp(isdn_net_local *lp) {
  if ((!(lp->flags & ISDN_NET_CONNECTED)) && !lp->dialstate) {
    int  chi;
    if (lp->phone[1]) {
      ulong flags;
      save_flags(flags);
      cli();
      /* Grab a free ISDN-Channel */
      if ((chi = get_free_channel(ISDN_USAGE_NET,lp->l2_proto,
				  lp->l3_proto,
				  lp->pre_device,
				  lp->pre_channel))<0) {
	printk(KERN_WARNING "isdn_net: No channel for %s\n",lp->name);
	restore_flags(flags);
	return -EAGAIN;
      }
      lp->dialstate = 1;
      lp->flags |= ISDN_NET_CONNECTED;
      /* Connect interface with channel */
      bind_channel(lp,chi);
      /* Initiate dialing */
      isdn_net_dial();
      restore_flags(flags);
      return 0;
    } else
      return -EINVAL;
  } else
    return -EBUSY;
}

/*
 * Force a net-interface to dial out.
 * This is always called from within userspace (ISDN_IOCTL_NET_DIAL).
 */
static int
isdn_net_force_dial(char *name) {
  isdn_net_dev *p = isdn_net_findif(name);

  if (!p)
    return -ENODEV;
  return(isdn_net_force_dial_lp(&p->local));
}

/*
 * Allocate a new network-interface and initialize its data structures.
 */
static char*
isdn_net_new(char *name, struct device *master) {
  isdn_net_dev *netdev;

  /* Avoid creating an existing interface */
  if (isdn_net_findif(name)) {
    printk(KERN_WARNING "isdn_net: interface %s already exists\n",name);
    return NULL;
  }
  if (!(netdev = (isdn_net_dev*)kmalloc(sizeof(isdn_net_dev),GFP_KERNEL))) {
    printk(KERN_WARNING "isdn_net: Could not allocate net-device\n");
    return NULL;
  }
  memset(netdev,0,sizeof(isdn_net_dev));
  if (name==NULL)
    strcpy(netdev->local.name,"         ");
  else
    strcpy(netdev->local.name,name);
  netdev->dev.name = netdev->local.name;
  netdev->dev.priv = &netdev->local;
  netdev->dev.init = isdn_net_init;
  if (master) {
    /* Device shall be a slave */
    struct device *p = (((isdn_net_local *)master->priv)->slave);
    struct device *q = master;

    netdev->local.master = master;
    /* Put device at end of slave-chain */
    while (p) {
      q = p;
      p = (((isdn_net_local *)p->priv)->slave);
    }
    ((isdn_net_local *)q->priv)->slave = &(netdev->dev);
    q->interrupt = 0;
    q->tbusy = 0;
    q->start = master->start;
  } else {
    /* Device shall be a master */
    if (register_netdev(&netdev->dev) != 0) {
      printk(KERN_WARNING "isdn_net: Could not register net-device\n");
      kfree(netdev);
      return NULL;
    }
  }
  netdev->local.magic        = ISDN_NET_MAGIC;
  netdev->local.isdn_device  = -1;
  netdev->local.isdn_channel = -1;
  netdev->local.pre_device   = -1;
  netdev->local.pre_channel  = -1;
  netdev->local.exclusive    = -1;
  netdev->local.ppp_minor    = -1;
  netdev->local.p_encap      = ISDN_NET_ENCAP_RAWIP;
  netdev->local.l2_proto     = ISDN_PROTO_L2_X75I;
  netdev->local.l3_proto     = ISDN_PROTO_L3_TRANS;
  netdev->local.slavedelay   = 10*HZ;
  netdev->local.srobin       = &netdev->dev;
  netdev->local.hupflags     = 8;  /* Do hangup even on incoming calls */
  netdev->local.onhtime      = 10; /* Default hangup-time for saving costs
                                  of those who forget configuring this */
  /* The following should be configurable via ioctl */
  netdev->local.dialmax = 1;
  /* Put into to netdev-chain */
  netdev->next = (void*)dev->netdev;
  dev->netdev = netdev;
  /* Enable auto-hangup timer */
  isdn_timer_ctrl(ISDN_TIMER_NETHANGUP,1);
  return netdev->dev.name;
}

static char*
isdn_net_newslave(char *parm) {
  char *p = strchr(parm,',');
  isdn_net_dev *n;
  char newname[10];

  if (p) {
    /* Slave-Name MUST not be empty */
    if (!strlen(p+1))
      return NULL;
    strcpy(newname,p+1);
    *p = 0;
    /* Master must already exist */
    if (!(n = isdn_net_findif(parm)))
      return NULL;
    /* Master must be a real interface, not a slave */
    if (n->local.master)
      return NULL;
    return(isdn_net_new(newname,&(n->dev)));
  }
  return NULL;
}

/*
 * Set interface-parameters.
 * Allways set all parameters, so the user-level application is responsible
 * for not overwriting existing setups. It has to get the current
 * setup first, if only selected parameters are to be changed.
 */
static int
isdn_net_setcfg(isdn_net_ioctl_cfg *cfg) {
  isdn_net_dev *p = isdn_net_findif(cfg->name);
  ulong features;
  int i;
  int drvidx;
  int chidx;
  char drvid[25];

  if (p) {
    /* See if any registered driver supports the features we want */
    features = (1 << cfg->l2_proto) | (256 << cfg->l3_proto);
    for (i=0;i<ISDN_MAX_DRIVERS;i++)
      if (dev->drv[i])
	if ((dev->drv[i]->interface->features & features) == features)
	  break;
    if (i==ISDN_MAX_DRIVERS) {
      printk(KERN_WARNING "isdn_net: No driver with selected features\n");
      return -ENODEV;
    }
    if (strlen(cfg->drvid)) {
      /* A bind has been requested ... */
      char *c;

      drvidx = -1;
      chidx  = -1;
      strcpy(drvid,cfg->drvid);
      if ((c = strchr(drvid,','))) {
	/* The channel-number is appended to the driver-Id with a comma */
	chidx = my_atoi(c+1);
	*c = '\0';
      }
      for (i=0;i<ISDN_MAX_DRIVERS;i++)
	/* Lookup driver-Id in array */
	if (!(strcmp(dev->drvid[i],drvid))) {
	  drvidx = i;
	  break;
	}
      if ((drvidx==-1) || (chidx==-1))
	/* Either driver-Id or channel-number invalid */
	return -ENODEV;
    } else {
      /* Parameters are valid, so get them */
      drvidx = p->local.pre_device;
      chidx = p->local.pre_channel;
    }
    if (cfg->exclusive>0) {
      int flags;

      /* If binding is exclusive, try to grab the channel */
      save_flags(flags);
      if ((i = get_free_channel(ISDN_USAGE_NET,p->local.l2_proto,
				p->local.l3_proto,
				drvidx,
				chidx))<0) {
	/* Grab failed, because desired channel is in use */
	p->local.exclusive = -1;
	restore_flags(flags);
	return -EBUSY;
      }
      /* All went ok, so update isdninfo */
      dev->usage[i] = ISDN_USAGE_EXCLUSIVE;
      isdn_info_update();
      restore_flags(flags);
      p->local.exclusive = i;
    } else {
      /* Non-exclusive binding or unbind. */
      p->local.exclusive = -1;
      if ((p->local.pre_device!=-1) && (cfg->exclusive==-1)) {
	unexclusive_channel(p->local.pre_device,p->local.pre_channel);
	drvidx = -1;
	chidx = -1;
      }
    }
    strcpy(p->local.msn,cfg->eaz);
    p->local.pre_device  = drvidx;
    p->local.pre_channel = chidx;
    p->local.onhtime     = cfg->onhtime;
    p->local.charge      = cfg->charge;
    p->local.l2_proto    = cfg->l2_proto;
    p->local.l3_proto    = cfg->l3_proto;
    p->local.p_encap     = cfg->p_encap;
    p->local.slavedelay  = cfg->slavedelay*HZ;
    if (cfg->secure)
      p->local.flags |= ISDN_NET_SECURE;
    else
      p->local.flags &= ~ISDN_NET_SECURE;
    if (cfg->callback)
      p->local.flags |= ISDN_NET_CALLBACK;
    else
      p->local.flags &= ~ISDN_NET_CALLBACK;
    if (cfg->chargehup)
      p->local.hupflags |= 4;
    else
      p->local.hupflags &= ~4;
    if (cfg->ihup)
      p->local.hupflags |= 8;
    else
      p->local.hupflags &= ~8;
    return 0;
  }
  return -ENODEV;
}

/*
 * Perform get-interface-parameters.ioctl
 */
static int
isdn_net_getcfg(isdn_net_ioctl_cfg *cfg) {
  isdn_net_dev *p = isdn_net_findif(cfg->name);

  if (p) {
    strcpy(cfg->eaz,p->local.msn);
    cfg->exclusive   = p->local.exclusive;
    if (p->local.pre_device>=0) {
      sprintf(cfg->drvid,"%s,%d",dev->drvid[p->local.pre_device],
	      p->local.pre_channel);
    } else
      cfg->drvid[0] = '\0';
    cfg->onhtime     = p->local.onhtime;
    cfg->charge      = p->local.charge;
    cfg->l2_proto    = p->local.l2_proto;
    cfg->l3_proto    = p->local.l3_proto;
    cfg->p_encap     = p->local.p_encap;
    cfg->secure      = (p->local.flags & ISDN_NET_SECURE)?1:0;
    cfg->callback    = (p->local.flags & ISDN_NET_CALLBACK)?1:0;
    cfg->chargehup   = (p->local.hupflags & 4)?1:0;
    cfg->ihup        = (p->local.hupflags & 8)?1:0;
    cfg->slavedelay  = p->local.slavedelay/HZ;
    if (p->local.slave)
      strcpy(cfg->slave,((isdn_net_local *)p->local.slave->priv)->name);
    else
      cfg->slave[0] = '\0';
    if (p->local.master)
      strcpy(cfg->master,((isdn_net_local *)p->local.master->priv)->name);
    else
      cfg->master[0] = '\0';
    return 0;
  }
  return -ENODEV;
}

/*
 * Add a phone-number to an interface.
 */
static int
isdn_net_addphone(isdn_net_ioctl_phone *phone) {
  isdn_net_dev *p = isdn_net_findif(phone->name);
  isdn_net_phone *n;

  if (isdn_net_checkwild(phone->phone) && (phone->outgoing & 1))
    return -EINVAL;
  if (p) {
    if (!(n = (isdn_net_phone*)kmalloc(sizeof(isdn_net_phone),GFP_KERNEL)))
      return -ENOMEM;
    strcpy(n->num,phone->phone);
    n->next = p->local.phone[phone->outgoing & 1];
    p->local.phone[phone->outgoing & 1] = n;
    return 0;
  }
  return -ENODEV;
}

/*
 * Return a string of all phone-numbers of an interface.
 */
static int
isdn_net_getphones(isdn_net_ioctl_phone *phone, char *phones) {
  isdn_net_dev   *p     = isdn_net_findif(phone->name);
  int             inout = phone->outgoing & 1;
  int             more  = 0;
  int             count = 0;
  isdn_net_phone *n;
  int             flags;
  int             ret;

  if (!p)
    return -ENODEV;
  save_flags(flags);
  cli();
  inout &= 1;
  n = p->local.phone[inout];
  while (n) {
    if (more) {
      put_fs_byte(' ',phones++);
      count++;
    }
    if ((ret = verify_area(VERIFY_WRITE,(void*)phones, strlen(n->num)+1))) {
      restore_flags(flags);
      return ret;
    }
    memcpy_tofs(phones,n->num,strlen(n->num)+1);
    phones += strlen(n->num);
    count += strlen(n->num);
    n = n->next;
    more = 1;
  }
  restore_flags(flags);
  count++;
  return count;
}

/*
 * Delete a phone-number from an interface.
 */

static int
isdn_net_delphone(isdn_net_ioctl_phone *phone) {
  isdn_net_dev   *p     = isdn_net_findif(phone->name);
  int             inout = phone->outgoing & 1;
  isdn_net_phone *n;
  isdn_net_phone *m;

  if (p) {
    n = p->local.phone[inout];
    m = NULL;
    while (n) {
      if (!strcmp(n->num,phone->phone)) {
	if (m)
	  m->next = n->next;
	else
	  p->local.phone[inout] = n->next;
	kfree(n);
	return 0;
      }
      m = n;
      n = (isdn_net_phone*)n->next;
    }
    return -EINVAL;
  }
  return -ENODEV;
}

/*
 * Delete all phone-numbers of an interface.
 */
static int
isdn_net_rmallphone(isdn_net_dev *p) {
  isdn_net_phone *n;
  isdn_net_phone *m;
  int           flags;
  int           i;

  save_flags(flags);
  cli();
  for (i=0;i<2;i++) {
    n = p->local.phone[i];
    while (n) {
      m = n->next;
      kfree(n);
      n = m;
    }
    p->local.phone[i] = NULL;
  }
  restore_flags(flags);
  return 0;
}

/*
 * Force a hangup of a network-interface.
 */
static int
isdn_net_force_hangup(char *name) {
  isdn_net_dev *p = isdn_net_findif(name);
  int           flags;
  struct device *q;

  if (p) {
    save_flags(flags);
    cli();
    if (p->local.isdn_device<0) {
      restore_flags(flags);
      return 1;
    }
    isdn_net_hangup(&p->dev);
    q = p->local.slave;
    /* If this interface has slaves, do a hangup for them also. */
    while (q) {
      isdn_net_hangup(q);
      q = (((isdn_net_local *)q->priv)->slave);
    }
    restore_flags(flags);
    return 0;
  }
  return -ENODEV;
}

/*
 * Helper-function for isdn_net_rm: Do the real work.
 */
static int
isdn_net_realrm(isdn_net_dev *p, isdn_net_dev *q) {
  int           flags;

  save_flags(flags);
  cli();
  if (p->local.master) {
    /* If it's a slave, it may be removed even if it is busy. However
     * it has to be hung up first.
     */
    isdn_net_hangup(&p->dev);
    p->dev.start = 0;
  }
  if (p->dev.start) {
    restore_flags(flags);
    return -EBUSY;
  }
  /* Free all phone-entries */
  isdn_net_rmallphone(p);
  /* If interface is bound exclusive, free channel-usage */
  if (p->local.exclusive!=-1)
    unexclusive_channel(p->local.pre_device,p->local.pre_channel);
  if (p->local.master) {
    /* It's a slave-device, so update master's slave-pointer if necessary */
    if (((isdn_net_local*)(p->local.master->priv))->slave == &p->dev)
      ((isdn_net_local*)(p->local.master->priv))->slave = p->local.slave;
  } else
    /* Unregister only if it's a master-device */
    unregister_netdev(&p->dev);
  /* Unlink device from chain */
  if (q)
    q->next = p->next;
  else
    dev->netdev = p->next;
  if (p->local.slave) {
    /* If this interface has a slave, remove it also */
    char *slavename = ((isdn_net_local*)(p->local.slave->priv))->name;
    isdn_net_dev *n = dev->netdev;
    q = NULL;
    while (n) {
      if (!strcmp(n->local.name,slavename)) {
	isdn_net_realrm(n,q);
	break;
      }
      q = n;
      n = (isdn_net_dev*)n->next;
    }
  }
  /* If no more net-devices remain, disable auto-hangup timer */
  if (dev->netdev == NULL)
    isdn_timer_ctrl(ISDN_TIMER_NETHANGUP,0);
  restore_flags(flags);
  kfree(p);
  return 0;
}

/*
 * Remove a single network-interface.
 */
static int
isdn_net_rm(char *name) {
  isdn_net_dev *p;
  isdn_net_dev *q;

  /* Search name in netdev-chain */
  p = dev->netdev;
  q = NULL;
  while (p) {
    if (!strcmp(p->local.name,name))
      return(isdn_net_realrm(p,q));
    q = p;
    p = (isdn_net_dev*)p->next;
  }
  /* If no more net-devices remain, disable auto-hangup timer */
  if (dev->netdev == NULL)
    isdn_timer_ctrl(ISDN_TIMER_NETHANGUP,0);
  return -ENODEV;
}

/*
 * Remove all network-interfaces
 */
static int
isdn_net_rmall(void) {
  int           flags;

  /* Walk through netdev-chain */
  save_flags(flags);
  cli();
  while (dev->netdev) {
    if (!dev->netdev->local.master) {
      /* Remove master-devices only, slaves get removed with their master */
      isdn_net_realrm(dev->netdev,NULL);
    }
  }
  dev->netdev = NULL;
  restore_flags(flags);
  return 0;
}

static int
isdn_set_allcfg(char *src) {
  int ret;
  int i;
  ulong flags;
  char  buf[1024];
  isdn_net_ioctl_cfg cfg;
  isdn_net_ioctl_phone phone;

  if ((ret = isdn_net_rmall()))
    return ret;
  save_flags(flags);
  cli();
  if ((ret = verify_area(VERIFY_READ,(void*)src,sizeof(int)))) {
    restore_flags(flags);
    return ret;
  }
  memcpy_tofs((char*)&i,src,sizeof(int));
  while (i) {
    char *c;
    char *c2;

    if ((ret = verify_area(VERIFY_READ,(void*)src,sizeof(cfg)))) {
      restore_flags(flags);
      return ret;
    }
    memcpy_tofs((char*)&cfg,src,sizeof(cfg));
    src += sizeof(cfg);
    if (!isdn_net_new(cfg.name,NULL)) {
      restore_flags(flags);
      return -EIO;
    }
    if ((ret = isdn_net_setcfg(&cfg))) {
      restore_flags(flags);
      return ret;
    }
    if ((ret = verify_area(VERIFY_READ,(void*)src,sizeof(buf)))) {
      restore_flags(flags);
      return ret;
    }
    memcpy_fromfs(buf,src,sizeof(buf));
    src += sizeof(buf);
    c = buf;
    while (*c) {
      if ((c2 = strchr(c,' ')))
	*c2++ = '\0';
      strcpy(phone.phone,c);
      strcpy(phone.name,cfg.name);
      phone.outgoing = 0;
      if ((ret = isdn_net_addphone(&phone))) {
	restore_flags(flags);
	return ret;
      }
      if (c2)
	c = c2;
      else
	c += strlen(c);
    }
    if ((ret = verify_area(VERIFY_READ,(void*)src,sizeof(buf)))) {
      restore_flags(flags);
      return ret;
    }
    memcpy_fromfs(buf,src,sizeof(buf));
    src += sizeof(buf);
    c = buf;
    while (*c) {
      if ((c2 = strchr(c,' ')))
	*c2++ = '\0';
      strcpy(phone.phone,c);
      strcpy(phone.name,cfg.name);
      phone.outgoing = 1;
      if ((ret = isdn_net_addphone(&phone))) {
	restore_flags(flags);
	return ret;
      }
      if (c2)
	c = c2;
      else
	c += strlen(c);
    }
    i--;
  }
  restore_flags(flags);
  return 0;
}

static int
isdn_get_allcfg(char *dest) {
  isdn_net_ioctl_cfg   cfg;
  isdn_net_ioctl_phone phone;
  isdn_net_dev         *p;
  ulong                flags;
  int                  ret;

  /* Walk through netdev-chain */
  save_flags(flags);
  cli();
  p = dev->netdev;
  while (p) {
    if ((ret = verify_area(VERIFY_WRITE,(void*)dest,sizeof(cfg)+10))) {
      restore_flags(flags);
      return ret;
    }
    strcpy(cfg.eaz,p->local.msn);
    cfg.exclusive   = p->local.exclusive;
    if (p->local.pre_device>=0) {
      sprintf(cfg.drvid,"%s,%d",dev->drvid[p->local.pre_device],
	      p->local.pre_channel);
    } else
      cfg.drvid[0] = '\0';
    cfg.onhtime     = p->local.onhtime;
    cfg.charge      = p->local.charge;
    cfg.l2_proto    = p->local.l2_proto;
    cfg.l3_proto    = p->local.l3_proto;
    cfg.p_encap     = p->local.p_encap;
    cfg.secure      = (p->local.flags & ISDN_NET_SECURE)?1:0;
    cfg.callback    = (p->local.flags & ISDN_NET_CALLBACK)?1:0;
    cfg.chargehup   = (p->local.hupflags & 4)?1:0;
    cfg.ihup        = (p->local.hupflags & 8)?1:0;
    memcpy_tofs(dest,p->local.name,10);
    dest += 10;
    memcpy_tofs(dest,(char *)&cfg,sizeof(cfg));
    dest += sizeof(cfg);
    strcpy(phone.name,p->local.name);
    phone.outgoing = 0;
    if ((ret = isdn_net_getphones(&phone,dest))<0) {
      restore_flags(flags);
      return ret;
    } else
      dest += ret;
    strcpy(phone.name,p->local.name);
    phone.outgoing = 1;
    if ((ret = isdn_net_getphones(&phone,dest))<0) {
      restore_flags(flags);
      return ret;
    } else
      dest += ret;
    p = p->next;
  }
  restore_flags(flags);
  return 0;
}

/*
 *****************************************************************************
 * And now the modules code.
 ****************************************************************************
 */

extern int printk( const char* fmt, ...);

/*
 * Allocate and initialize all module-data, register modem-devices
 */
int
init_module(void) {
  int i;
  char rev[10];
  char *p;

  if (!(dev = (isdn_dev*)kmalloc(sizeof(isdn_dev),GFP_KERNEL))) {
    printk(KERN_WARNING "isdn: Could not allocate device-struct.\n");
    return -EIO;
  }
  memset((char *)dev,0,sizeof(isdn_dev));
  for (i=0;i<ISDN_MAX_DRIVERS;i++)
    dev->drvmap[i] = -1;
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    dev->chanmap[i] = -1;
    dev->m_idx[i] = -1;
    strcpy(dev->num[i],"???");
  }
  if (register_chrdev(ISDN_MAJOR, "isdn", &isdn_fops)) {
    printk(KERN_WARNING "isdn: Could not register ISDN-controldevice\n");
    kfree(dev);
    return -EIO;
  }
  if ((i =modem_init())<0) {
    printk(KERN_WARNING "Could not register ISDN-modemdevices\n");
    if (i==-3)
      tty_unregister_driver(&dev->mdm.cua_modem);
    if (i<=-2)
      tty_unregister_driver(&dev->mdm.tty_modem);
    kfree(dev);
    unregister_chrdev(ISDN_MAJOR, "isdn");    
    return -EIO;
  }
#ifdef ISYNCPPP
  if (init_ippp() < 0) {
    printk(KERN_WARNING "Could not register PPP-devices\n");
    tty_unregister_driver(&dev->mdm.tty_modem);
    tty_unregister_driver(&dev->mdm.cua_modem);
    for (i=0;i<ISDN_MAX_CHANNELS;i++)
      kfree(dev->mdm.info[i].xmit_buf);
    unregister_chrdev(ISDN_MAJOR, "isdn");
    kfree(dev);
    return -EIO;
  }
#endif /* ISYNCPPP */
  if ((p = strchr(revision,':'))) {
    strcpy(rev,p+1);
    p = strchr(rev,'$');
    *p = 0;
  } else
    strcpy(rev," ??? ");
  printk(KERN_INFO "ISDN-linklevel-driver Rev%sloaded\n",rev);
  isdn_info_update();
  return 0;
}

/*
 * Try unloading module
 */
void
cleanup_module( void) {
  int flags;
  int i;

#ifdef ISYNCPPP
  cleanup_ippp();
#endif
  save_flags(flags);
  cli();
  if (isdn_net_rmall()<0) {
    printk(KERN_WARNING "isdn: net-device busy, remove cancelled\n");
    restore_flags(flags);
    return;
  }
  if (tty_unregister_driver(&dev->mdm.tty_modem)) {
    printk(KERN_WARNING "isdn: modem-device busy, remove cancelled\n");
    restore_flags(flags);
    return;
  }
  if (tty_unregister_driver(&dev->mdm.cua_modem)) {
    printk(KERN_WARNING "isdn: modem-device busy, remove cancelled\n");
    restore_flags(flags);
    return;
  }
  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    kfree(dev->mdm.info[i].xmit_buf-4);
  if (unregister_chrdev(ISDN_MAJOR, "isdn") != 0) {
    printk(KERN_WARNING "isdn: cleanup_module failed\n");
  } else {
    del_timer(&dev->timer);
    kfree(dev);
    printk(KERN_INFO "ISDN-linklevel-driver unloaded\n");
  }
  restore_flags(flags);
}

/*********************************************************************
 Modem-Emulator-Routines
 *********************************************************************/

#define cmdchar(c) ((c>' ')&&(c<=0x7f))

/*
 * Put a message from the AT-emulator into receive-buffer of tty,
 * convert CR, LF, and BS to values in modem-registers 3, 4 and 5.
 */
static void
modem_cout(char *msg, modem_info *info) {
  struct tty_struct *tty;
  atemu *m = &(dev->mdm.atmodem[info->line]);
  char *p;
  char c;
  ulong flags;
  
  if (!msg) {
    printk(KERN_WARNING "isdn: Null-Message in modem_cout\n");
    return;
  }
  save_flags(flags);
  cli();
  tty = info->tty;
  for (p=msg;*p;p++) {
    switch (*p) {
      case '\r':
	c = m->mdmreg[3];
	break;
      case '\n':
	c = m->mdmreg[4];
	break;
      case '\b':
	c = m->mdmreg[5];
	break;
      default:
	c = *p;
    }
    if ((info->flags & ISDN_ASYNC_CLOSING) || (!tty)) {
      restore_flags(flags);
      return;
    }
    if (tty->flip.count >= TTY_FLIPBUF_SIZE)
      break;
    tty_insert_flip_char(tty,c,0);
  }
  restore_flags(flags);
  queue_task(&tty->flip.tqueue, &tq_timer);
}

/*
 * Perform ATH Hangup
 */
static void
on_hook(modem_info *info) {
  if (info->isdn_channel>=0) {
#ifdef ISDN_DEBUG_MODEM_HUP
    printk(KERN_DEBUG "Mhup in on_hook\n");
#endif
    modem_hup(info);
    modem_result(3,info);
  }
}

static void
off_hook(void) {
  printk(KERN_DEBUG "off_hook\n");
}

#define PLUSWAIT1 (HZ/2)   /* 0.5 sec.*/
#define PLUSWAIT2 (HZ*3/2) /* 1.5 sec */

/*
 * Check Buffer for Modem-escape-sequence, activate timer-callback to
 * modem_plus() if sequence found.
 *
 * Parameters:
 *   p          pointer to databuffer
 *   plus       escape-character
 *   count      length of buffer
 *   pluscount  count of valid escape-characters so far
 *   lastplus   timestamp of last character
 */
static void
modem_chk3plus(const u_char *p, u_char plus, int count, int *pluscount,
	       int *lastplus, int from_user) {
  char cbuf[3];

  if (plus>127)
    return;
  if (count>3) {
    p += count-3;
    count = 3;
    *pluscount = 0;
  }
  if (from_user) {
    memcpy_fromfs(cbuf,p,count);
    p = cbuf;
  }
  while (count>0) {
    if (*(p++) == plus) {
      if ((*pluscount)++) {
	/* Time since last '+' > 0.5 sec. ? */
	if ((jiffies - *lastplus) > PLUSWAIT1)
	  *pluscount = 1;
      } else {
	/* Time since last non-'+' < 1.5 sec. ? */
	if ((jiffies - *lastplus) < PLUSWAIT2)
	  *pluscount = 0;
      }
      if ((*pluscount == 3) && (count = 1))
	isdn_timer_ctrl(ISDN_TIMER_MODEMPLUS,1);
      if (*pluscount > 3)
	*pluscount = 1;
    } else
      *pluscount = 0;
    *lastplus = jiffies;
    count--;
  }
}

/*
 * Return result of AT-emulator to tty-receive-buffer, depending on
 * modem-register 12, bit 0 and 1.
 * For CONNECT-messages also switch to online-mode.
 * For RING-message handle auto-ATA if register 0 != 0
 */
static void
modem_result(int code, modem_info *info) {
  atemu *m = &dev->mdm.atmodem[info->line];
  static char *msg[] = {"OK","CONNECT","RING","NO CARRIER","ERROR",
			"CONNECT 64000","NO DIALTONE","BUSY","NO ANSWER",
			"RINGING","NO MSN/EAZ"};
  ulong flags;
  char s[4];
  
  switch (code) {
    case 2:
      m->mdmreg[1]++; /* RING */
      if (m->mdmreg[1] == m->mdmreg[0]) {
	/* Accept incoming call */
	isdn_ctrl cmd;
	m->mdmreg[1] = 0;
	dev->mdm.msr[info->line] &= ~UART_MSR_RI;
	cmd.driver  = info->isdn_driver;
	cmd.arg     = info->isdn_channel;
	cmd.command = ISDN_CMD_ACCEPTD;
	dev->drv[info->isdn_driver]->interface->command(&cmd);
      }
      break;
    case 3:
      /* NO CARRIER */
      save_flags(flags);
      cli();
      dev->mdm.msr[info->line] &= ~(UART_MSR_DCD | UART_MSR_RI);
      if ((info->flags & ISDN_ASYNC_CLOSING) || (!info->tty)) {
	restore_flags(flags);
	return;
      }
      restore_flags(flags);
      break;
    case 1:
    case 5:
      dev->mdm.online[info->line] = 1;
      break;
  }
  if (m->mdmreg[12] & 1) {
    /* Show results */
    modem_cout("\r\n",info);
    if (m->mdmreg[12] & 2) {
      /* Show numeric results */
      sprintf(s,"%d",code);
      modem_cout(s,info);
    } else {
      if (code==2) {
	modem_cout("CALLER NUMBER: ",info);
	modem_cout(dev->num[info->drv_index],info);
	modem_cout("\r\n",info);
      }
      modem_cout(msg[code],info);
      if (code==5) {
	/* Append Protocol to CONNECT message */
	modem_cout((m->mdmreg[14] != 3)?"/X.75":"/HDLC",info);
	if (m->mdmreg[13] & 2)
	  modem_cout("/T.70",info);
      }
    }
    modem_cout("\r\n",info);
  }
  if (code==3) {
    save_flags(flags);
    cli();
    if ((info->flags & ISDN_ASYNC_CLOSING) || (!info->tty)) {
      restore_flags(flags);
      return;
    }
    if ((info->flags & ISDN_ASYNC_CHECK_CD) &&
	(!((info->flags & ISDN_ASYNC_CALLOUT_ACTIVE) &&
	   (info->flags & ISDN_ASYNC_CALLOUT_NOHUP))))
      tty_hangup(info->tty);
    restore_flags(flags);
  }
}

/*
 * Display a modem-register-value.
 */
static void
show_register(int ridx, modem_info *info) {
  char v[6];

  sprintf(v,"%d\r\n",dev->mdm.atmodem[info->line].mdmreg[ridx]);
  modem_cout(v,info);
}

/*
 * Get integer from char-pointer, set pointer to end of number
 */
static int
getnum(char **p) {
  int v = -1;

  while (*p[0]>='0' && *p[0]<='9')
    v = ((v<0)?0:(v*10))+(int)((*p[0]++)-'0');
  return v;
}

/*
 * Get MSN-string from char-pointer, set pointer to end of number
 */
static void
getmsnstr(char *n, char **p) {
  while ((*p[0]>='0' && *p[0]<='9') || (*p[0]==','))
    *n++ = *p[0]++;
  *n = '\0';
}

/*
 * Get phone-number from modem-commandbuffer
 */
static void
getdial(char *p, char *q) {
  int first = 1;

  while (strchr("0123456789,#.*WPTS-",*p) && *p) {
    if ((*p>='0' && *p<='9') || ((*p=='S') && first))
      *q++ = *p;
    p++;
    first = 0;
  }
  *q = 0;
}

/*
 * Parse and perform an AT-command-line.
 *
 * Parameter:
 *   channel   index to line (minor-device)
 */
static void
do_parse(modem_info *info) {
  atemu *m = &dev->mdm.atmodem[info->line];
  char *p;
  int  mreg;
  int  mval;
  int  i;
  char rb[100];
  char ds[40];
  isdn_ctrl cmd;

#ifdef ISDN_DEBUG_AT
  printk(KERN_DEBUG "AT: '%s'\n",m->mdmcmd);
#endif
  for (p=&m->mdmcmd[2];*p;) {
    switch (*p) {
      case 'A':
	/* A - Accept incoming call */
	p++;
	if (m->mdmreg[1]) {
#define FIDOBUG
#ifdef FIDOBUG
/* Variables fido... defined temporarily for finding a strange bug */
	  driver     *fido_drv;
	  isdn_if    *fido_if;
	  int         fido_isdn_driver;
	  modem_info *fido_modem_info;
          int        (*fido_command)(isdn_ctrl*);
#endif
	  /* Accept incoming call */
	  m->mdmreg[1] = 0;
	  dev->mdm.msr[info->line] &= ~UART_MSR_RI;
	  cmd.driver  = info->isdn_driver;
	  cmd.command = ISDN_CMD_SETL2;
	  cmd.arg     = info->isdn_channel + (m->mdmreg[14] << 8);
	  dev->drv[info->isdn_driver]->interface->command(&cmd);
	  cmd.driver  = info->isdn_driver;
	  cmd.command = ISDN_CMD_SETL3;
	  cmd.arg     = info->isdn_channel + (m->mdmreg[15] << 8);
	  dev->drv[info->isdn_driver]->interface->command(&cmd);
	  cmd.driver  = info->isdn_driver;
	  cmd.arg     = info->isdn_channel;
	  cmd.command = ISDN_CMD_ACCEPTD;
#ifdef FIDOBUG
	  fido_modem_info  = info;
	  fido_isdn_driver = fido_modem_info->isdn_driver;
	  fido_drv         = dev->drv[fido_isdn_driver];
	  fido_if          = fido_drv->interface;
	  fido_command     = fido_if->command;
	  fido_command(&cmd);
#else
	  dev->drv[info->isdn_driver]->interface->command(&cmd);
#endif
	} else {
	  modem_result(8,info);
	  return;
	}
	break;
      case 'D':
	/* D - Dial */
        getdial(++p,ds);
	p += strlen(p);
	if (!strlen(m->msn))
	  modem_result(10,info);
	else
	  if (strlen(ds))
	      modem_dial(ds,info,m);
	  else
	    modem_result(4,info);
	return;
      case 'E':
	/* E - Turn Echo on/off */
	p++;
	switch (*p) {
	  case '0':
	    p++;
	    m->mdmreg[12] &= ~4;
	    break;
	  case '1':
	    p++;
	    m->mdmreg[12] |= 4;
	    break;
	  default:
	    modem_result(4,info);
	    return;
	}
	break;
      case 'H':
	/* H - On/Off-hook */
	p++;
	switch (*p) {
	  case '0':
	    p++;
	    on_hook(info);
	    break;
	  case '1':
	    p++;
	    off_hook();
	    break;
	  default:
	    on_hook(info);
	    break;
	}
	break;
      case 'I':
	/* I - Information */
	p++;
	modem_cout("ISDN for Linux  (c) by Fritz Elfert\r\n",info);
	switch (*p) {
	  case '0':
	  case '1':
	    p++;
	    break;
	  default:
	}
	break;
      case 'O':
	/* O - Go online */
	p++;
	if (dev->mdm.msr[info->line] & UART_MSR_DCD) /* if B-Channel is up */
	  modem_result(5,info);
	else
	  modem_result(3,info);
	return;
      case 'Q':
	/* Q - Turn Emulator messages on/off */
	p++;
	switch (*p) {
	  case '0':
	    p++;
	    m->mdmreg[12] |= 1;
	    break;
	  case '1':
	    p++;
	    m->mdmreg[12] &= ~1;
	    break;
	  default:
	    modem_result(4,info);
	    return;
	}
	break;
      case 'S':
	/* S - Set/Get Register */
	p++;
	mreg = getnum(&p);
	if (mreg<0 || mreg>ISDN_MODEM_ANZREG) {
	  modem_result(4,info);
	  return;
	}
	switch (*p) {
	  case '=':
	    p++;
	    mval = getnum(&p);
	    if (mval>=0 && mval<=255) {
	      if ((mreg==16) && ((mval*16)>ISDN_SERIAL_XMIT_SIZE)) {
		modem_result(4,info);
		return;
	      }
	      m->mdmreg[mreg] = mval;
	    } else {
	      modem_result(4,info);
	      return;
	    }
	    break;
	  case '?':
	    p++;
	    show_register(mreg,info);
	    return;
	    break;
	  default:
	    modem_result(4,info);
	    return;
	}
	break;
      case 'V':
	/* V - Numeric or ASCII Emulator-messages */
	p++;
	switch (*p) {
	  case '0':
	    p++;
	    m->mdmreg[12] |= 2;
	    break;
	  case '1':
	    p++;
	    m->mdmreg[12] &= ~2;
	    break;
	  default:
	    modem_result(4,info);
	    return;
	}
	break;
      case 'Z':
	/* Z - Load Registers from Profile */
	p++;
	modem_reset_regs(m,1);
	break;
      case '+':
	p++;
	switch (*p) {
	  case 'F':
	    break;
	}
	break;
      case '&':
	p++;
	switch (*p) {
	  case 'B':
	    /* &B - Set Buffersize */
	    p++;
	    i = getnum(&p);
	    if ((i<0) || (i>ISDN_SERIAL_XMIT_SIZE)) {
	      modem_result(4,info);
	      return;
	    }
	    m->mdmreg[16] = i / 16;
	    break;
	  case 'D':
	    /* &D - Set DCD-Low-behavior */
	    p++;
	    switch (getnum(&p)) {
	      case 2:
		m->mdmreg[12] &= ~32;
		break;
	      case 3:
		m->mdmreg[12] |= 32;
		break;
	      default:
		modem_result(4,info);
		return;
	    }
	    break;
	  case 'E':
	    /* &E -Set EAZ/MSN */
	    p++;
	    getmsnstr(m->msn,&p);
	    break;
	  case 'F':
	    /* &F -Set Factory-Defaults */
	    p++;
	    modem_reset_profile(m);
	    modem_reset_regs(m,1);
	    break;
	  case 'S':
	    /* &S - Set Windowsize */
	    p++;
	    i = getnum(&p);
	    if ((i>0) && (i<9))
	      m->mdmreg[17] = i;
	    else {
	      modem_result(4,info);
	      return;
	    }
	    break;
	  case 'V':
	    /* &V - Show registers */
	    p++;
	    for (i=0;i<ISDN_MODEM_ANZREG;i++) {
	      sprintf(rb,"S%d=%d%s",i,m->mdmreg[i],(i==6)?"\r\n":" ");
	      modem_cout(rb,info);
	    }
	    sprintf(rb,"\r\nEAZ/MSN: %s\r\n",strlen(m->msn)?m->msn:"None");
	    modem_cout(rb,info);
	    break;
	  case 'W':
	    /* &W - Write Profile */
	    p++;
	    switch (*p) {
	      case '0':
		p++;
		modem_write_profile(m);
		break;
	      default:
		modem_result(4,info);
		return;
	    }
	    break;
	  case 'X':
	    /* &X - Switch to BTX-Mode */
	    p++;
	    switch (*p) {
	      case '0':
		p++;
		m->mdmreg[13] &= ~2;
		break;
	      case '1':
		p++;
		m->mdmreg[13] |= 2;
		m->mdmreg[14] = 0;
		m->mdmreg[16] = 7;
		m->mdmreg[18] = 7;
		m->mdmreg[19] = 0;
		break;
	      default:
		modem_result(4,info);
		return;
	    }
	    break;
	  default:
	    modem_result(4,info);
	    return;
	}
	break;
      default:
	modem_result(4,info);
	return;
    }
  }  
  modem_result(0,info);
}

/* Need own toupper() because standard-toupper is not available
 * within modules.
 */
#define my_toupper(c) (((c>='a')&&(c<='z'))?(c&0xdf):c)

/*
 * Perform line-editing of AT-commands
 *
 * Parameters:
 *   p        inputbuffer
 *   count    length of buffer
 *   channel  index to line (minor-device)
 *   user     flag: buffer is in userspace
 */
static int
modem_parse(const char *p, int count, modem_info *info, int user) {
  atemu *m = &dev->mdm.atmodem[info->line];
  int  total = 0;
  u_char c;
  char eb[2];
  int  cnt;

  for (cnt=count;cnt>0;p++,cnt--) {
    if (user)
      c = get_fs_byte(p);
    else
      c = *p;
    total ++;
    if (c==m->mdmreg[3] || c==m->mdmreg[4]) {
      /* Separator (CR oder LF) */
      m->mdmcmd[m->mdmcmdl] = 0;
      if (m->mdmreg[12] & 4) {
	eb[0] = c;
	eb[1] = 0;
	modem_cout(eb,info);
      }
      if (m->mdmcmdl>=2)
	do_parse(info); 
      m->mdmcmdl = 0;
      continue;
    }
    if (c==m->mdmreg[5] && m->mdmreg[5]<128) {
      /* Backspace-Funktion */
      if ((m->mdmcmdl>2) || (!m->mdmcmdl)) {
	if (m->mdmcmdl)
	  m->mdmcmdl--;
	if (m->mdmreg[12] & 4)
	  modem_cout("\b",info);
      }
      continue;
    }
    if (cmdchar(c)) {
      if (m->mdmreg[12] & 4) {
	eb[0] = c;
	eb[1] = 0;
	modem_cout(eb,info);
      }
      if (m->mdmcmdl < 255) {
        c = my_toupper(c);
	switch (m->mdmcmdl) {
	  case 0:
	    if (c=='A')
	      m->mdmcmd[m->mdmcmdl++] = c;
	    break;
	  case 1:
	    if (c=='T')
	      m->mdmcmd[m->mdmcmdl++] = c;
	    break;
	  default:
	    m->mdmcmd[m->mdmcmdl++] = c;
	}
      }
    }
  }
  return total;
}

/*
 * Switch all modem-channels who are online and got a valid
 * escape-sequence 1.5 seconds ago, to command-mode.
 * This function is called every second via timer-interrupt from within 
 * timer-dispatcher isdn_timer_function()
 */
static void
modem_plus(void) {
  int ton = 0;
  int i;
  int midx;

  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (USG_MODEM(dev->usage[i]))
      if ((midx = dev->m_idx[i])>=0)
	if (dev->mdm.online[midx]) {
	  ton = 1;
	  if ((dev->mdm.atmodem[midx].pluscount == 3) &&
	      ((jiffies - dev->mdm.atmodem[midx].lastplus) > PLUSWAIT2)) {
	    dev->mdm.atmodem[midx].pluscount = 0;
	    dev->mdm.online[midx] = 0;
	    modem_result(0,&dev->mdm.info[midx]);
	  }
	}
  isdn_timer_ctrl(ISDN_TIMER_MODEMPLUS,ton);
}

/*
 * Put a RING-message to all modem-channels who have the RI-bit set.
 * This function is called every second via timer-interrupt from within 
 * timer-dispatcher isdn_timer_function()
 */
static void
modem_ring(void) {
  int ton = 0;
  int i;
  int midx;

  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (USG_MODEM(dev->usage[i]))
      if ((midx = dev->m_idx[i])>=0)
	if (dev->mdm.msr[midx] & UART_MSR_RI) {
	  ton = 1;
	  modem_result(2,&dev->mdm.info[midx]);
	}
  isdn_timer_ctrl(ISDN_TIMER_MODEMRING,ton);
}

static void
modem_xmit(void) {
  int ton = 0;
  int i;
  int midx;
  char *bufptr;
  int  buflen;

  for (i=0;i<ISDN_MAX_CHANNELS;i++)
    if (USG_MODEM(dev->usage[i]))
      if ((midx = dev->m_idx[i])>=0)
	if (dev->mdm.online[midx]) {
	  modem_info *info = &(dev->mdm.info[midx]);
	  ulong flags;

	  save_flags(flags);
	  cli();
	  if (info->xmit_count>0) {
	    struct tty_struct *tty = info->tty;
	    ton = 1;
#if 0
	    printk(KERN_DEBUG "WB2: %d\n",info->xmit_count);
#endif
	    bufptr = info->xmit_buf;
	    buflen = info->xmit_count;
	    if (dev->mdm.atmodem[midx].mdmreg[13] & 2) {
	      /* Add T.70 simplyfied header */
#ifdef ISDN_DEBUG_MODEM_DUMP
	      isdn_dumppkt("T70pack3:",bufptr,buflen,40);
#endif
	      bufptr -= 4;
	      buflen += 4;
	      memcpy(bufptr,"\1\0\1\0",4);
#ifdef ISDN_DEBUG_MODEM_DUMP
	      isdn_dumppkt("T70pack4:",bufptr,buflen,40);
#endif
	    }
	    if (dev->drv[info->isdn_driver]->interface->
		writebuf(info->isdn_channel,bufptr,buflen,0)>0) {
	      info->xmit_count = 0;
	      info->xmit_size = dev->mdm.atmodem[midx].mdmreg[16] * 16;
#if FUTURE
	      info->send_outstanding++;
	      dev->mdm.msr[midx] &= ~UART_MSR_CTS;
#endif
	      if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
		  tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);
	      wake_up_interruptible(&tty->write_wait);
	    }
	  }
	  restore_flags(flags);
	}
  isdn_timer_ctrl(ISDN_TIMER_MODEMXMIT,ton);
}

#if FUTURE
/*
 * A packet has been output successfully.
 * Search the tty-devices for an aproppriate device, decrement it's
 * counter for outstanding packets, and set CTS if this counter reaches 0.
 */
static void
isdn_modem_bsent(int drv, int chan) {
  int i;
  ulong flags;

  save_flags(flags);
  cli();
  for (i=0;i<ISDN_MAX_CHANNELS;i++) {
    modem_info *info = &dev->mdm.info[i];
    if ((info->isdn_driver == drv) &&
	(info->isdn_channel == chan) &&
	(info->send_outstanding)) {
      if (!(--info->send_outstanding))
	dev->mdm.msr[i] |= UART_MSR_CTS;
      restore_flags(flags);
      return;
    }
  }
  restore_flags(flags);
  return;
}
#endif /* FUTURE */

#ifdef ISYNCPPP

/*****************************************************************************
 * fake device for ISDN PPP Daemon v0.1
 *
 * copyrights (c) 1995 by Michael Hipp (Michael.Hipp@student.uni-tuebingen.de)
 * copy policy: GNU PUBLIC LICENSE Version 2 (see file 'COPYING' for more)
 *
 * 7.July 1995: removed a few bugs .. 
 * Mai/June 95: inital version
 */

static int
free_ippp(isdn_net_local *lp) {
  if(lp->ppp_minor < 0)
    return 0;
  ippp_hangup(lp->ppp_minor);
  return 0;
}

static int
bind_ippp(isdn_net_local *lp) {
  int i;

  if(lp->p_encap != ISDN_NET_ENCAP_SYNCPPP)
    return 0;

  /* save_flags ... cli();  */

  for(i=0;i<ISDN_MAX_CHANNELS;i++)
  {
    if(ippp_table[i].state == IPPP_OPENSLEEP)
    {
      printk(KERN_DEBUG "find_minor, %d lp: %08lx\n",i,(long)lp);
      break;
    }
  }

  if(i >= ISDN_MAX_CHANNELS)
    return -1;

  lp->ppp_minor = i;
  ippp_table[lp->ppp_minor].lp = lp;
  
  ippp_table[lp->ppp_minor].state = IPPP_OPENWAKEDUP;
  wake_up_interruptible(&ippp_table[lp->ppp_minor].wq);

  return lp->ppp_minor;
}


static int
ippp_hangup(int minor) {
  if(minor < 0 || minor >= ISDN_MAX_CHANNELS)
    return 0;

  if(ippp_table[minor].state == IPPP_OPENSLEEP)
    wake_up_interruptible(&ippp_table[minor].wq);

  ippp_table[minor].state = IPPP_CLOSEWAIT;
  send_sig(SIGIO,ippp_table[minor].tk,1);

  return 1;
}


static int
ippp_state(int minor) {
  if(minor < 0 || minor >= ISDN_MAX_CHANNELS) {
    printk(KERN_DEBUG "ippp: illegal minor %d in ippp_state().\n",minor);
    return 0;
  }
  return( ippp_table[minor].pppcfg & SC_ENABLE_IP );
}


static int
ippp_open(int minor, struct file *file) {
  struct task_struct *p;
  
  p = current;

/*  printk(KERN_DEBUG "ippp, open, minor: %d %d\n",minor,p->session);  */

  if(ippp_table[minor].state)
    return -EBUSY;

  MOD_INC_USE_COUNT;

/* set IPPP_OPENSLEEP and calling interruptible_sleep_on() 
   should be done while cli()ing */

  ippp_table[minor].pppcfg = 0;
  ippp_table[minor].unit   = 0;         /* TEST: currently always zero */
  ippp_table[minor].mru    = 1524;      /* MRU, default 1524*/
  ippp_table[minor].maxcid = 16;        /* VJ: maxcid */
  ippp_table[minor].tk     = current;
  ippp_table[minor].wq     = NULL;
  ippp_table[minor].first  = ippp_table[minor].rq+NUM_RCV_BUFFS-1;
  ippp_table[minor].last   = ippp_table[minor].rq;

  ippp_table[minor].state  = IPPP_OPENSLEEP;

  interruptible_sleep_on(&ippp_table[minor].wq);

  if(ippp_table[minor].state == IPPP_OPENWAKEDUP)
    ippp_table[minor].state = IPPP_OPEN;

  /* return error if state==IPPP_CLOSEWAIT ? */
  return 0;
}


static void
ippp_release(int minor, struct file *file) {

  if(minor < 0 || minor >= ISDN_MAX_CHANNELS)
    return;

/* in the future: call isdn_net_hangup
                  cleanup read_queue  */

  if(ippp_table[minor].lp)
  {
    ippp_table[minor].lp->ppp_minor = -1;
    ippp_table[minor].lp = NULL;
  }
 
  ippp_table[minor].state = 0;
  MOD_DEC_USE_COUNT;
}

static int
get_arg(void *b,unsigned long *val) {
  int r;
  if((r=verify_area(VERIFY_READ,(void*)b,sizeof(unsigned long))))
    return r;
  memcpy_fromfs((void*) val,b,sizeof(unsigned long));
  return 0;
}

static int
set_arg(void *b,unsigned long val) {
  int r;
  if((r=verify_area(VERIFY_WRITE,b,sizeof(unsigned long))))
    return r;
  memcpy_tofs(b,(void*)&val,sizeof(unsigned long));
  return 0;
}

static int
ippp_ioctl(int minor, struct file *file,unsigned int cmd, unsigned long arg) {
  unsigned long val;
  int r;

/*  printk(KERN_DEBUG "ippp, ioctl, minor: %d %x ",minor,cmd); */

  if(ippp_table[minor].state != IPPP_OPEN &&
     ippp_table[minor].state != IPPP_CONNECT )
    return -EINVAL;

  switch(cmd) {
    case PPPIOCSINPSIG:	/* set input ready signal */
       /* usual: sig = SIGIO */ /* we always deliver a SIGIO */
      break;
    case PPPIOCGUNIT:		/* get ppp/isdn unit number */
      /* here "isdnX" not "pppX" */
      if((r=set_arg( (void*) arg,ippp_table[minor].unit)))
	return r;
      break;
    case PPPIOCGFLAGS:		/* get configuration flags */
      if( (r=set_arg( (void*)arg,ippp_table[minor].pppcfg )))
	return r;
      break;
    case PPPIOCSFLAGS:		/* set configuration flags */
      if( (r=get_arg( (void*)arg , &val) ))
	return r;
      ippp_table[minor].pppcfg = val;
      break;
    case PPPIOCGSTAT:		/* read PPP statistic information */
      break;
    case PPPIOCGTIME:		/* read time delta information */
      break;
    case PPPIOCSMRU:		/* set receive unit size for PPP */
      if( (r=get_arg( (void*)arg , &val) ))
	return r;
      ippp_table[minor].mru = val;
      break;
    case PPPIOCSMAXCID:	/* set the maximum compression slot id */
      if( (r=get_arg( (void*)arg , &val) ))
	return r;
      ippp_table[minor].maxcid = val;
      break;
    case PPPIOCGDEBUG:
      break;
    case PPPIOCSDEBUG:
      break;
    default:
      break;
#if 0
#define PPPIOCGASYNCMAP  0x5492 /* get async map */
#define PPPIOCSASYNCMAP  0x5493 /* set async map */
#define PPPIOCGXASYNCMAP 0x549B /* get async table */
#define PPPIOCSXASYNCMAP 0x549C /* set async table */
#define PPPIOCRASYNCMAP  0x549E /* set receive async map */
#endif
	
  }
  return 0;
}

static int
ippp_select(int minor, struct file *file,int type,select_table *st) {
  printk(KERN_DEBUG "ippp, select %d\n",type);
  return 1;
}

/*
 *  fill up ippp_read() queue .. send SIGIO to process .. 
 */

static int
ippp_fill_rq(char *buf,int len,int minor) {
  struct ippp_buf_queue *bf,*bl;
  unsigned long flags;

  if(minor < 0 ||minor >= ISDN_MAX_CHANNELS) {
    printk(KERN_WARNING "ippp: illegal minor.\n");
    return 0;
  }

/* better: IPPP_OPENWAKEDUP, IPPP_OPEN, IPPP_CONNECT */
  if(ippp_table[minor].state == IPPP_OPENSLEEP) {
    printk(KERN_WARNING "ippp: device not activated.\n");
    return 0;
  }

  save_flags(flags); 
  cli();

  bf = ippp_table[minor].first;
  bl = ippp_table[minor].last;

/*  printk(KERN_DEBUG "ipppf: %d %d %08lx %08lx\n",minor,len,(long) bf,(long)bl);  */

  if(bf == bl) {
    printk(KERN_WARNING "ippp: Queue is full; discarding first buffer\n");
    bf = bf->next;
    kfree(bf->buf);
    ippp_table[minor].first = bf;
  }

  bl->buf = (char *) kmalloc(len,GFP_ATOMIC);
  if(!bl->buf) {
    printk(KERN_WARNING "ippp: Can't alloc buf\n");    
    restore_flags(flags);
    return 0;
  }
  bl->len = len;

  memcpy(bl->buf,buf,len);
  
  ippp_table[minor].last = bl->next; 
  restore_flags(flags);

  send_sig(SIGIO,ippp_table[minor].tk,1);

  return len;
}

/*
 * read() .. pppd calls it only after receiption of an interrupt 
 */

static int
ippp_read(int minor, struct file *file,char *buf,int count) {
  ippp_struct *c = &ippp_table[minor];
  struct ippp_buf_queue *b;
  int r;
  unsigned long flags;

  if(ippp_table[minor].state != IPPP_OPEN &&
     ippp_table[minor].state != IPPP_CONNECT)
    return 0;

  if((r=verify_area(VERIFY_WRITE, (void *) buf,count)))
    return r;

  save_flags(flags);
  cli();

  b = c->first->next;
  if(!b->buf) {
    restore_flags(flags);
    return -EAGAIN;
  }
  if(b->len < count)
    count = b->len;
  memcpy_tofs(buf,b->buf,count);
  kfree(b->buf);
  b->buf = NULL;
  c->first = b;
  restore_flags(flags);

  return count;
}

static int
ippp_write(int minor, struct file *file,char *buf,int count) {
  isdn_net_local *lp;

  if(ippp_table[minor].state != IPPP_OPEN &&
     ippp_table[minor].state != IPPP_CONNECT)
    return 0;

  lp = ippp_table[minor].lp;

  /* -> push it directly to the lowlevel interface */

  if(!lp)
    printk(KERN_WARNING "lp == NULL\n"); 
  else {
/*    printk(KERN_DEBUG "lp: %08lx %d %d\n",(long) lp,lp->isdn_device,lp->isdn_channel);  */

    if(lp->isdn_device < 0 || lp->isdn_channel < 0)
      return 0;

/*
    printk(KERN_DEBUG "OK1: %08lx ",(long)dev->drv[lp->isdn_device]->interface);
    printk(KERN_DEBUG " %08lx ",(long) dev->drv[lp->isdn_device]->interface->writebuf);
    printk(KERN_DEBUG " %d\n", dev->drv[lp->isdn_device]->running);
*/

    if(dev->drv[lp->isdn_device]->running && lp->dialstate == 0 && 
       (lp->flags & ISDN_NET_CONNECTED) )
      dev->drv[lp->isdn_device]->interface->writebuf(lp->isdn_channel,buf,count,1);
  }

  return count;
}

static int
init_ippp(void) {
  int i,j;

  if (!(ippp_table =
	(ippp_struct *)kmalloc(sizeof(ippp_struct) * ISDN_MAX_CHANNELS,
			      GFP_KERNEL))) {
    printk(KERN_WARNING "init_ippp: Could not alloc ippp_table\n");
    return -1;
  }
  for(i=0;i<ISDN_MAX_CHANNELS;i++) {
    ippp_table[i].state = 0;
 /*   ippp_table[i].wq = NULL; */
    ippp_table[i].first = ippp_table[i].rq+NUM_RCV_BUFFS-1;
    ippp_table[i].last = ippp_table[i].rq;

#if 0
    ippp_table[i].pppcfg = 0;
    ippp_table[i].unit = 0;         /* TEST: currently always zero */
    ippp_table[i].mru = 1524;         /* MRU, default 1524*/
    ippp_table[i].maxcid = 16;	     /* VJ: maxcid */
    ippp_table[i].lp = NULL;
#endif

    for(j=0;j<NUM_RCV_BUFFS;j++)    {
      ippp_table[i].rq[j].buf = NULL;
      ippp_table[i].rq[j].last = ippp_table[i].rq +
	(NUM_RCV_BUFFS+j-1)%NUM_RCV_BUFFS;
      ippp_table[i].rq[j].next = ippp_table[i].rq + (j+1)%NUM_RCV_BUFFS;
    }
  }
/*
  if (register_chrdev(IPPP_MAJOR,"ippp",&ippp_fops)) {
    printk(KERN_WARNING "Can't get major %d for IPPP device.\n", IPPP_MAJOR);
    return -EIO;
  }
*/
  return 0;
}

static void cleanup_ippp(void) {
/*
  unregister_chrdev(IPPP_MAJOR,"ippp");
  */
  kfree(ippp_table);
}

#endif /* ISYNCPPP */
















