#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/major.h>
#include <asm/segment.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/signal.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <asm/bitops.h>
#include <asm/segment.h>

#include <linux/time.h>
#include <linux/types.h>
#include <linux/unistd.h>
#include <linux/timer.h>
#include <linux/ioport.h>
#define INCLUDE_INLINE_FUNCS
#include <linux/tqueue.h>

#define byte unsigned char
			
extern int printk( const char* fmt, ...);

#include "isdnif.h"
#include "teles.h"

extern void tei_handler(struct PStack *st,byte pr,
  struct BufHeader *ibh);
extern struct IsdnCard cards[];
extern int nrcards;

static void teles_xmitframe(struct PStack *st,
  struct BufHeader *ibh);
static void hscx_xmitframe(struct PStack *st,
  struct BufHeader *ibh);

static inline byte readisac(byte *cardm,byte offset) {
  return *(byte *)(cardm+0x100+((offset&1)?0x1ff:0)+offset);
}

static inline void writeisac(byte *cardm,byte offset,byte value) {
  *(byte *)(cardm+0x100+((offset&1)?0x1ff:0)+offset)=value;
}


static inline byte readhscx(byte *base, byte hscx, byte offset) 
{
  return *(byte *)(base+0x180+((offset&1)?0x1FF:0)+
    ((hscx&1)?0x40:0)+offset);
}

static inline void writehscx(byte *base, byte hscx, byte offset, 
  byte data) 
{
  *(byte *)(base+0x180+((offset&1)?0x1FF:0)+
    ((hscx&1)?0x40:0)+offset)=data;
}

#define ISAC_MASK 0x20
#define ISAC_ISTA 0x20
#define ISAC_STAR 0x21
#define ISAC_CMDR 0x21
#define ISAC_EXIR 0x24

#define ISAC_RBCH 0x2a

#define ISAC_ADF2 0x39
#define ISAC_SPCR 0x30
#define ISAC_ADF1 0x38
#define ISAC_CIX0 0x31
#define ISAC_STCR 0x37
#define ISAC_MODE 0x22
#define ISAC_RSTA 0x27
#define ISAC_RBCL 0x25
#define ISAC_TIMR 0x23

#define HSCX_ISTA 0x20
#define HSCX_CCR1 0x2f 
#define HSCX_CCR2 0x2c
#define HSCX_TSAR 0x31
#define HSCX_TSAX 0x30
#define HSCX_XCCR 0x32
#define HSCX_RCCR 0x33
#define HSCX_MODE 0x22
#define HSCX_CMDR 0x21
#define HSCX_EXIR 0x24
#define HSCX_XAD1 0x24
#define HSCX_XAD2 0x25
#define HSCX_RAH2 0x27
#define HSCX_RSTA 0x27
#define HSCX_TIMR 0x23
#define HSCX_STAR 0x21
#define HSCX_RBCL 0x25
#define HSCX_XBCH 0x2d
#define HSCX_VSTR 0x2e
#define HSCX_MASK 0x20

static inline void waitforCEC(byte *base,byte hscx)
{
  long to=200;
  while ((readhscx(base,hscx,HSCX_STAR)&0x04)&&(to--)) ; 
  if (!to) printk("waitforCEC timeout\n");
}

static inline void waitforXFW(byte *base,byte hscx)
{
  long to=200;
  waitforCEC(base,hscx);
  while ((!(readhscx(base,hscx,HSCX_STAR)&0x40))&&(to--)) ; 
  if (!to) printk("waitforXFW timeout\n");
}

static inline void writehscxCMDR(byte *base, byte hscx,
  byte data)
{
  waitforCEC(base,hscx);
  writehscx(base,hscx,HSCX_CMDR,data);
}

/* fast interrupt here */

#define ISAC_RCVBUFREADY 0
#define ISAC_XMTBUFREADY 1
#define ISAC_PHCHANGE    2

#define HSCX_RCVBUFREADY 0
#define HSCX_XMTBUFREADY 1

void teles_hscxreport(struct IsdnCardState *sp,int hscx)
{
  printk("HSCX %d\n",hscx);
  printk("  ISTA %x\n",readhscx(sp->membase,hscx,HSCX_ISTA));
  printk("  STAR %x\n",readhscx(sp->membase,hscx,HSCX_STAR));
  printk("  EXIR %x\n",readhscx(sp->membase,hscx,HSCX_EXIR));
}

void teles_report(struct IsdnCardState *sp)
{
  printk("ISAC\n");
  printk("  ISTA %x\n",readisac(sp->membase,ISAC_ISTA));
  printk("  STAR %x\n",readisac(sp->membase,ISAC_STAR));
  printk("  EXIR %x\n",readisac(sp->membase,ISAC_EXIR));
  teles_hscxreport(sp,0);
  teles_hscxreport(sp,1);    
}

struct tq_struct *tq_isdn=&tq_last;
struct tq_struct run_isdn_bh= {
  0,0,(void *)(void *)run_task_queue,&tq_isdn,
};

/* HSCX stuff goes here */

static void hscx_sched_event(struct HscxState *hsp,
  int event)
{
  hsp->event|=1<<event;
  queue_task_irq_off(&hsp->tqueue,&tq_isdn);
  queue_task_irq_off(&run_isdn_bh,&tq_immediate);
  mark_bh(IMMEDIATE_BH);
}

static void hscx_empty_fifo(struct HscxState *hsp,int count)
{
  byte *ptr;
  struct BufHeader *ibh=hsp->rcvibh;
    
  if (hsp->sp->debug)
    printk("hscx_empty_fifo\n");

  if (hsp->rcvptr+count>BUFFER_SIZE(HSCX_RBUF_ORDER,
    HSCX_RBUF_BPPS)) {
    printk("hscx_empty_fifo: incoming packet too large\n");
    return;
  }

  ptr=DATAPTR(ibh);ptr+=hsp->rcvptr;  
 
  hsp->rcvptr+=count;
  while (count--) *ptr++=readhscx(hsp->membase,hsp->hscx,0x0);
  writehscxCMDR(hsp->membase,hsp->hscx,0x80);
}

static void hscx_fill_fifo(struct HscxState *hsp)
{
  struct BufHeader *ibh;
  int i,more;
  byte *ptr;

  if (hsp->sp->debug)
    printk("hscx_fill_fifo\n");

  ibh=hsp->xmtibh;
  if (!ibh) return; 

  i=ibh->datasize-hsp->sendptr;
  if (i<=0) return;

#if 0  
  if (!hsp->sendptr) {
    ptr=DATAPTR(ibh);
    printk("snd bytes %2x %2x %2x %2x %2x\n",ptr[0],ptr[1],ptr[2],
      ptr[3],ptr[4]);
  }
#endif
  
  more=0;
  if (i>32) {
    more=!0;
    i=32;
  }

  ptr=DATAPTR(ibh);
  ptr+=hsp->sendptr;
  hsp->sendptr+=i;

  waitforXFW(hsp->membase,hsp->hscx);
  while (i--) writehscx(hsp->membase,hsp->hscx,0x0,*ptr++);
    
  writehscxCMDR(hsp->membase,hsp->hscx,more?0x8:0xa);
}

static inline void hscx_interrupt(struct IsdnCardState *sp,
  byte val,byte hscx)
{
  byte r;
  struct HscxState *hsp=sp->hs+hscx;
  int count;

  if (!hsp->init) return;
  
  if (val&0x80) { /* RME */

    r=readhscx(hsp->membase,hsp->hscx,HSCX_RSTA);
    if ((r&0xf0)!=0xa0) {
      if (!r&0x80) printk("Teles: HSCX invalid frame\n");
      if (r&0x40)  printk("Teles: HSCX RDO\n");
      if (!r&0x20) printk("Teles: HSCX CRC error\n");
#if 0  /* happens on every connection close */    
      if (r&0x10)  printk("Teles: HSCX RAB\n");
#endif
      if (hsp->rcvibh) BufPoolRelease(hsp->rcvibh);
      hsp->rcvibh=NULL;
      writehscxCMDR(hsp->membase,hsp->hscx,0x80);
      goto afterRME;
    }

    if (!hsp->rcvibh) 
      if (BufPoolGet(&hsp->rcvibh,&hsp->rbufpool,GFP_ATOMIC,
        (void *)1,1)) {
        printk("HSCX RME out of buffers at %ld\n",jiffies);
        writehscxCMDR(hsp->membase,hsp->hscx,0x80);
        goto afterRME;
      }
      else
        hsp->rcvptr=0;

    count=readhscx(hsp->membase,hsp->hscx,HSCX_RBCL)&0x1f;
    if (count==0) count=32;
    hscx_empty_fifo(hsp,count);
    hsp->rcvibh->datasize=hsp->rcvptr-1;
    BufQueueLink(&hsp->rq,hsp->rcvibh);
    hsp->rcvibh=NULL;
    hscx_sched_event(hsp,HSCX_RCVBUFREADY);
  }

afterRME:
  if (val&0x40) {  /* RPF */
    if (!hsp->rcvibh) 
      if (BufPoolGet(&hsp->rcvibh,&hsp->rbufpool,
        GFP_ATOMIC,(void *)1,2)) {
        printk("HSCX RME out of buffers at %ld\n",jiffies);
        writehscxCMDR(hsp->membase,hsp->hscx,0x80);
        goto afterRPF;
      }
      else
       hsp->rcvptr=0;

    hscx_empty_fifo(hsp,32);
  }      

afterRPF:
  if (val&0x10) {  /* XPR */  
     if (hsp->xmtibh) 
      if (hsp->xmtibh->datasize>hsp->sendptr) {
        hscx_fill_fifo(hsp);
	goto afterXPR;
      }
      else {
	if (hsp->xmtibh->primitive) 
          BufPoolRelease(hsp->xmtibh);
        hscx_sched_event(hsp,HSCX_XMTBUFREADY);
        hsp->xmtibh=NULL;
	hsp->sendptr=0;
      }
      
    if (!BufQueueUnlink(&hsp->xmtibh,&hsp->sq)) 
      hscx_fill_fifo(hsp);
   }

afterXPR:
}

/* ISAC stuff goes here */

static void isac_sched_event(struct IsdnCardState *sp,
  int event)
{
  sp->event|=1<<event;
  queue_task_irq_off(&sp->tqueue,&tq_isdn);
  queue_task_irq_off(&run_isdn_bh,&tq_immediate);
  mark_bh(IMMEDIATE_BH);
}

static void empty_fifo(struct IsdnCardState *sp,int count)
{
  byte *ptr;
  struct BufHeader *ibh=sp->rcvibh;

  if (sp->debug)
    printk("empty_fifo\n");

  if (sp->rcvptr>=3072) {
    printk("empty_fifo rcvptr %d\n",sp->rcvptr);
    return;
  }
  
  ptr=DATAPTR(ibh);ptr+=sp->rcvptr;  
  sp->rcvptr+=count;
  while (count--) *ptr++=readisac(sp->membase,0x0);
  writeisac(sp->membase,ISAC_CMDR,0x80);
}

static void fill_fifo(struct IsdnCardState *sp)
{
  struct BufHeader *ibh;
  int i,more;
  byte *ptr;

  if (sp->debug)
    printk("fill_fifo\n");

  ibh=sp->xmtibh;
  if (!ibh) return; 
  
  i=ibh->datasize-sp->sendptr;
  if (i<=0) return;
  if (i>=3072) return;

  more=0;
  if (i>32) {
    more=!0;
    i=32;
  }

  ptr=DATAPTR(ibh);
  ptr+=sp->sendptr;
  sp->sendptr+=i;
  while (i--) writeisac(sp->membase,0x0,*ptr++);
   
  writeisac(sp->membase,ISAC_CMDR,more?0x8:0xa);
}

static void teles_interrupt(int intno,struct pt_regs * regs)
{
  byte val,r;
  struct IsdnCardState *sp;
  unsigned int count;
  struct HscxState *hsp;
  
  sp=(struct IsdnCardState *)irq2dev_map[intno];
  
  if (!sp) return;

  val=readhscx(sp->membase,1,HSCX_ISTA);

  if (val&0x01) {
      hsp=sp->hs+1;
      printk("HSCX B EXIR %x xmitbh %lx rcvibh %lx\n",
	 readhscx(sp->membase,1,HSCX_EXIR),(long)hsp->xmtibh,
	 (long)hsp->rcvibh);
  }

  if (val&0xf8) {
    if (sp->debug) printk("HSCX B interrupt %x\n",val);
    hscx_interrupt(sp,val,1);
  }

  if (val&0x02) {
    printk("HSCX A EXIR %x\n",readhscx(sp->membase,0,HSCX_EXIR));
  }
  
  if (val&0x04) {
    val=readhscx(sp->membase,0,HSCX_ISTA);
    if (sp->debug) printk("HSCX A interrupt %x\n",val);
    hscx_interrupt(sp,val,0);
  }
    
  val=readisac(sp->membase,ISAC_ISTA);

  if (sp->debug)
    printk("ISAC interrupt %x\n",val);
  
  if (val&0x80) { /* RME */

    r=readisac(sp->membase,ISAC_RSTA);
    if ((r&0x70)!=0x20) {
      if (r&0x40)  printk("Teles: ISAC RDO\n");
      if (!r&0x20) printk("Teles: ISAC CRC error\n");
#if 0    /* see HSCX comment */
      if (r&0x10)  printk("Teles: ISAC RAB\n");
#endif
      if (sp->rcvibh) BufPoolRelease(sp->rcvibh);
      sp->rcvibh=NULL;
      writeisac(sp->membase,ISAC_CMDR,0x80);
      goto afterRME;
    }

    if (!sp->rcvibh) 
      if (BufPoolGet(&(sp->rcvibh),&(sp->rbufpool),GFP_ATOMIC,
        (void *)1,3)) {
        printk("ISAC RME out of buffers!\n");
        writeisac(sp->membase,ISAC_CMDR,0x80);
        goto afterRME;
      }
      else
        sp->rcvptr=0;
    
    count=readisac(sp->membase,ISAC_RBCL)&0x1f;
    if (count==0) count=32;
    empty_fifo(sp,count);
    sp->rcvibh->datasize=sp->rcvptr;
    BufQueueLink(&(sp->rq),sp->rcvibh);
    sp->rcvibh=NULL;
    isac_sched_event(sp,ISAC_RCVBUFREADY);
  }

afterRME:
  if (val&0x40) {  /* RPF */
    if (!sp->rcvibh) 
      if (BufPoolGet(&(sp->rcvibh),&(sp->rbufpool),GFP_ATOMIC,
        (void *)1,4)) {
        printk("ISAC RME out of buffers!\n");
        writeisac(sp->membase,ISAC_CMDR,0x80);
        goto afterRPF;
      }
      else
        sp->rcvptr=0;

    empty_fifo(sp,32);
  }      

afterRPF:
  if (val&0x20) {
  }

  if (val&0x10) {  /* XPR */ 
    if (sp->xmtibh)
      if (sp->xmtibh->datasize>sp->sendptr) {
        fill_fifo(sp);
	goto afterXPR;
      }
      else {
	if (sp->xmtibh->primitive)
	  BufPoolRelease(sp->xmtibh);
	isac_sched_event(sp,ISAC_XMTBUFREADY);
	sp->xmtibh=NULL;
        sp->sendptr=0;
      }
    if (!BufQueueUnlink(&sp->xmtibh,&sp->sq))
      fill_fifo(sp);
  }  
    
afterXPR:
  if (val&0x04) {  /* CISQ */
    sp->ph_state=(readisac(sp->membase,ISAC_CIX0)>>2)&0xf;
    isac_sched_event(sp,ISAC_PHCHANGE);
  }

  writeisac(sp->membase,ISAC_MASK,0xFF);
  writehscx(sp->membase,0,HSCX_MASK,0xFF);
  writehscx(sp->membase,1,HSCX_MASK,0xFF);
  writeisac(sp->membase,ISAC_MASK,0x0);
  writehscx(sp->membase,0,HSCX_MASK,0x0);
  writehscx(sp->membase,1,HSCX_MASK,0x0);

}

/* soft interrupt */

static void process_new_ph(struct IsdnCardState *sp) 
{
  int enq;
  long flags;

  if (sp->debug)
    printk("new ph_state %d\n",sp->ph_state);

  enq=sp->sq.head||sp->xmtibh;

  switch(sp->ph_state) {
  case(0):
  case(6):
    if (enq)
      writeisac(sp->membase,ISAC_CIX0,3);
    else 
      writeisac(sp->membase,ISAC_CIX0,(0xf<<2)|3);
    break;
  case(7):
    if (enq)
      writeisac(sp->membase,ISAC_CIX0,(9<<2)|3);
    break;
  case(12):
  case(13):
    sp->ph_active=5;
    save_flags(flags);cli();
    if (!sp->xmtibh)
      if (!BufQueueUnlink(&sp->xmtibh,&sp->sq)) 
        sp->sendptr=0;
    if (sp->xmtibh) 
      fill_fifo(sp);
    restore_flags(flags);
    break;
  case(4):
  case(8):
    break;
  default:
     sp->ph_active=0;
     break;
  }
}

static void process_xmt(struct IsdnCardState *sp)
{
}

static void process_rcv(struct IsdnCardState *sp)
{
  struct BufHeader *ibh,*cibh;
  struct PStack *stptr;
  byte *ptr;
  int found,broadc;

  while (!BufQueueUnlink(&ibh,&(sp->rq))) {
    stptr=sp->stlist;
    ptr=DATAPTR(ibh);
    broadc=(ptr[1]>>1)==127;

    if (broadc&&sp->dlogflag&&(!(ptr[0]>>2))) 
      dlogframe(sp,ptr+3,ibh->datasize-3,
        "Q.931 frame network->user broadcast");
    
    if (broadc) {
      while (stptr!=NULL) { 
        if ((ptr[0]>>2)==stptr->l2.sap)
	  if (!BufPoolGet(&cibh,&sp->rbufpool,GFP_ATOMIC,
            (void *)1,5)) {
	    memcpy(DATAPTR(cibh),DATAPTR(ibh),ibh->datasize);
            cibh->datasize=ibh->datasize;
	    stptr->l1.acceptph(stptr,cibh);
          }
          else
            printk("isdn broadcast buffer shortage\n");
        stptr=stptr->next;
      }
      BufPoolRelease(ibh);
    }
    else {
      found=0;
      while (stptr!=NULL) 
        if ( ((ptr[0]>>2)==stptr->l2.sap) &&
	   ((ptr[1]>>1)==stptr->l2.tei) ) { 
          stptr->l1.acceptph(stptr,ibh); 
	  found=!0;
          break;
        }
        else
        stptr=stptr->next;
      if (!found)
        BufPoolRelease(ibh);
    }

  }  


}

static void isac_bh(struct IsdnCardState *sp)
{
  if (!sp) return;

  if (clear_bit(ISAC_RCVBUFREADY,&sp->event))
    process_rcv(sp);
  if (clear_bit(ISAC_XMTBUFREADY,&sp->event))
    process_xmt(sp);
  if (clear_bit(ISAC_PHCHANGE,&sp->event)) 
    process_new_ph(sp);
}


static void hscx_process_xmt(struct HscxState *hsp)
{
}

static void hscx_process_rcv(struct HscxState *hsp)
{
  struct BufHeader *ibh;

#ifdef DEBUG_MAGIC  
  if (hsp->magic!=301270) {
    printk("hscx_process_rcv magic not 301270\n");
    return;
  }
#endif  

  while (!BufQueueUnlink(&ibh,&hsp->rq)) 
    hsp->st->l1.acceptph(hsp->st,ibh); 
}

static void hscx_bh(struct HscxState *hsp)
{

  if (!hsp) return;

  if (clear_bit(HSCX_RCVBUFREADY,&hsp->event))
    hscx_process_rcv(hsp);
  if (clear_bit(HSCX_XMTBUFREADY,&hsp->event))
    hscx_process_xmt(hsp);

}

/* interrupt stuff ends here */

static void restart_ph(struct IsdnCardState *sp)
{
  switch(sp->ph_active) {
  case(0):
    if (sp->ph_state==6)
      writeisac(sp->membase,ISAC_CIX0,3);
    else   
      writeisac(sp->membase,ISAC_CIX0,(0x1<<2)|3);
    sp->ph_active=1;
    break;
  }
}

static void initisac(byte *cardmem) 
{
  writeisac(cardmem,ISAC_MASK,0xff);
  writeisac(cardmem,ISAC_ADF2,0x0);
  writeisac(cardmem,ISAC_SPCR,0xa);
  writeisac(cardmem,ISAC_ADF1,0x2);
  writeisac(cardmem,ISAC_STCR,0x70);
  writeisac(cardmem,ISAC_MODE,0xc9);
  writeisac(cardmem,ISAC_CMDR,0x41);
  writeisac(cardmem,ISAC_CIX0,(1<<2)|3);
}

#define byteout(addr,val) outb_p(val,addr)
#define bytein(addr) inb_p(addr)

static int checkcard(int cardnr) 
{
  int timout;
  byte cfval;
  struct IsdnCard *card=cards+cardnr;
  
  if (!card->ioaddr) {
    printk("Teles card assumed at %lx irq %d\n",
      (long)card->membase,card->interrupt);
    printk("HSCX version A: %x B:%x\n",
      readhscx(card->membase,0,HSCX_VSTR)&0xf,
      readhscx(card->membase,1,HSCX_VSTR)&0xf);
  }
  else {
	   
    switch(card->interrupt) {
      case  2: cfval = 0x00; break;
      case  3: cfval = 0x02; break;
      case  4: cfval = 0x04; break;
      case  5: cfval = 0x06; break;
      case 10: cfval = 0x08; break;
      case 11: cfval = 0x0A; break;
      case 12: cfval = 0x0C; break;
      case 15: cfval = 0x0E; break;
      default: cfval = 0x00; break;
    }

    cfval |= (((unsigned int)card->membase >> 9) & 0xF0);

    if(bytein(card->ioaddr+0) != 0x51) 
    { printk("XXX Byte at %x is %x\n",card->ioaddr+0,
        bytein(card->ioaddr+0)); 
      return 0; 
    }

    if(bytein(card->ioaddr+1) != 0x93) 
    { printk("XXX Byte at %x is %x\n",card->ioaddr+1,
        bytein(card->ioaddr+1)); 
      return 0; 
    }

    if(bytein(card->ioaddr+2) != 0x1E) 
    { printk("XXX Byte at %x is %x\n",card->ioaddr+2,
        bytein(card->ioaddr+2)); 
      return 0; 
    }

    cli();
    timout = jiffies+(HZ/10)+1;
    byteout(card->ioaddr+4,cfval);
    sti();
    while(jiffies <= timout) ;

    cli();
    timout = jiffies+(HZ/10)+1;
    byteout(card->ioaddr+4,cfval|1);
    sti();
    while(jiffies <= timout) ;

    printk("Teles card found at io %x membase %lx irq %d\n",
      card->ioaddr,(long)card->membase,card->interrupt);
    printk("HSCX version A:%x B:%x\n",
      readhscx(card->membase,0,HSCX_VSTR)&0xf,
      readhscx(card->membase,1,HSCX_VSTR)&0xf);

  }

  cli();
  timout = jiffies+(HZ/5)+1;
  *(byte *)(card->membase+0x80) = 0;
  sti();
  while(jiffies <= timout) ;

  cli();
  *(byte *)(card->membase+0x80) = 1;
  timout = jiffies+(HZ/5)+1;
  sti();
  while(jiffies <= timout) ;

  return(!0);
}

void modehscx(struct HscxState *hs,int mode,
  int ichan)
{
  struct IsdnCardState *sp=hs->sp;
  int hscx=hs->hscx;

#if 0  
  printk("modehscx hscx %d mode %d ichan %d\n",
    hscx,mode,ichan);
#endif
  
  if (hscx==0) ichan=1-ichan;  /* raar maar waar... */

  writehscx(sp->membase,hscx,HSCX_CCR1, 0x85);
  writehscx(sp->membase,hscx,HSCX_XAD1, 0xFF);
  writehscx(sp->membase,hscx,HSCX_XAD2, 0xFF);
  writehscx(sp->membase,hscx,HSCX_RAH2, 0xFF);
  writehscx(sp->membase,hscx,HSCX_XBCH, 0x0);
  
  switch(mode) {
  case(0):
    writehscx(sp->membase,hscx,HSCX_CCR2, 0x30); /* 0x00 */
    writehscx(sp->membase,hscx,HSCX_TSAX, 0xff); /* 0 */
    writehscx(sp->membase,hscx,HSCX_TSAR, 0xff); /* 0 */
    writehscx(sp->membase,hscx,HSCX_XCCR, 7);
    writehscx(sp->membase,hscx,HSCX_RCCR, 7);
    writehscx(sp->membase,hscx,HSCX_MODE, 0x84);
    break;
  case(1):
    if (ichan==0) {
      writehscx(sp->membase,hscx,HSCX_CCR2, 0x30); /* 0x00 */
      writehscx(sp->membase,hscx,HSCX_TSAX, 0x7); /* 0 */
      writehscx(sp->membase,hscx,HSCX_TSAR, 0x7); /* 0 */
      writehscx(sp->membase,hscx,HSCX_XCCR, 7);
      writehscx(sp->membase,hscx,HSCX_RCCR, 7);
    }
    else {
      writehscx(sp->membase,hscx,HSCX_CCR2, 0x30); /* 0x00 */
      writehscx(sp->membase,hscx,HSCX_TSAX, 0x3); /* 0 */
      writehscx(sp->membase,hscx,HSCX_TSAR, 0x3); /* 0 */
      writehscx(sp->membase,hscx,HSCX_XCCR, 7);
      writehscx(sp->membase,hscx,HSCX_RCCR, 7);
    }
    writehscx(sp->membase,hscx,HSCX_MODE,0xe4);
    writehscx(sp->membase,hscx,HSCX_CMDR,0x41);
    break;
  case(2):
    if (ichan==0) {
      writehscx(sp->membase,hscx,HSCX_CCR2, 0x30); /* 0x00 */
      writehscx(sp->membase,hscx,HSCX_TSAX, 0x7); /* 0 */
      writehscx(sp->membase,hscx,HSCX_TSAR, 0x7); /* 0 */
      writehscx(sp->membase,hscx,HSCX_XCCR, 7);
      writehscx(sp->membase,hscx,HSCX_RCCR, 7);
    }
    else {
      writehscx(sp->membase,hscx,HSCX_CCR2, 0x30); /* 0x00 */
      writehscx(sp->membase,hscx,HSCX_TSAX, 0x3); /* 0 */
      writehscx(sp->membase,hscx,HSCX_TSAR, 0x3); /* 0 */
      writehscx(sp->membase,hscx,HSCX_XCCR, 7);
      writehscx(sp->membase,hscx,HSCX_RCCR, 7);
    }
    writehscx(sp->membase,hscx,HSCX_MODE,0x8c);
    writehscx(sp->membase,hscx,HSCX_CMDR,0x41);
    break;
  }
  writehscx(sp->membase,hscx,HSCX_ISTA,0x00);
}   

void teles_addlist(struct IsdnCardState *sp,
  struct PStack *st)
{
  st->next=sp->stlist;
  sp->stlist=st;
}

void teles_rmlist(struct IsdnCardState *sp,
  struct PStack *st)
{
  struct PStack *p;

  if (sp->stlist==st)
    sp->stlist=st->next;
  else {
    p=sp->stlist;
    while (p) 
      if (p->next==st) {
        p->next=st->next;
        return;
      }
      else
	p=p->next;
  }
}


static void teles_xmitframe(struct PStack *st,
  struct BufHeader *ibh)
{
  struct IsdnCardState *sp=(struct IsdnCardState *)
    st->l1.hardware;
  long flags;

#ifdef DEBUG_MAGIC  
  if (sp->magic!=301271) {
    printk("teles_xmitframe magic not 301271\n");
    return;
  }
#endif  

  if (sp->debug)
    printk("teles_xmitframe\n");

  BufQueueLink(&sp->sq,ibh);
  if (sp->xmtibh) return;

  save_flags(flags);cli();
  if (sp->ph_active==5) {
    if (!sp->xmtibh)
      if (!BufQueueUnlink(&sp->xmtibh,&sp->sq)) {
        sp->sendptr=0;
        fill_fifo(sp);
      }
  }
  else
    restart_ph(sp);
  restore_flags(flags);
}

static void isac_discardq(struct PStack *st,int pr,void *heldby,
  int releasetoo)
{
  struct IsdnCardState *sp=(struct IsdnCardState *)st->l1.hardware;

#ifdef DEBUG_MAGIC  
  if (sp->magic!=301271) {
    printk("isac_discardq magic not 301271\n");
    return;
  }
#endif
  
  BufQueueDiscard(&sp->sq,pr,heldby,releasetoo); 
}

void setstack_teles(struct PStack *st,struct IsdnCardState *sp)
{
  st->l1.hardware=sp;
  st->l1.sbufpool=&(sp->sbufpool);
  st->l1.rbufpool=&(sp->rbufpool);
  st->l1.smallpool=&(sp->smallpool);

  st->l1.teistack=sp->teistack;
  st->l1.stlistp=&(sp->stlist);
  
  st->l1.xmitframe=teles_xmitframe;
  st->l1.discardq=isac_discardq;

}

static void init_tei(struct IsdnCardState *sp)
{
  struct PStack *st;
  
  st=(struct PStack *)Smalloc(sizeof(struct PStack),GFP_KERNEL,
    "struct PStack");     

  setstack_teles(st,sp);
  
  st->l2.extended=!0;
  st->l2.laptype=LAPD;
  st->l2.state=4;
  st->l2.window=1;
  st->l2.orig=!0;

  st->l2.sap=63;
  st->l2.tei=127;
  
  setstack_isdnl2(st);
  st->l2.debug=0;
  st->l3.debug=0;
  
  st->l3.hup=(void *)tei_handler;  
  st->l3.hdown=(void *)tei_handler;
  st->l4.writewakeup=NULL;
  
  teles_addlist(sp,st);
  sp->teistack=st;
}

static void release_tei(struct IsdnCardState *sp)
{
  struct PStack *st=sp->teistack;

  teles_rmlist(sp,st);
  Sfree((void *)st);
}

void init_hscxstate(struct IsdnCardState *sp,
  int hscx)
{
  struct HscxState *hsp=sp->hs+hscx;

  hsp->sp=sp;
  hsp->hscx=hscx;
  hsp->membase=sp->membase;
  
  hsp->tqueue.next=0;
  hsp->tqueue.sync=0;
  hsp->tqueue.routine=(void *)(void *)hscx_bh;
  hsp->tqueue.data=hsp;

  hsp->inuse=0;
  hsp->init=0;

#ifdef DEBUG_MAGIC
  hsp->magic=301270;
#endif
}

void initcard(int cardnr)
{
  struct IsdnCardState *sp;
  struct IsdnCard *card=cards+cardnr;
  
  sp=(struct IsdnCardState *)
    Smalloc(sizeof(struct IsdnCardState),GFP_KERNEL,
    "struct IsdnCardState");
  sp->membase=card->membase;
  sp->cardnr=cardnr;

  BufPoolInit(&sp->sbufpool,ISAC_SBUF_ORDER,ISAC_SBUF_BPPS,
    ISAC_SBUF_MAXPAGES);
  BufPoolInit(&sp->rbufpool,ISAC_RBUF_ORDER,ISAC_RBUF_BPPS,
    ISAC_RBUF_MAXPAGES);
  BufPoolInit(&sp->smallpool,ISAC_SMALLBUF_ORDER,ISAC_SMALLBUF_BPPS,
    ISAC_SMALLBUF_MAXPAGES);

  sp->dlogspace=Smalloc(4096,GFP_KERNEL,"dlogspace");

  initisac(card->membase);

  sp->rcvibh=NULL;
  sp->rcvptr=0;
  sp->xmtibh=NULL;
  sp->sendptr=0;
  sp->event=0;
  sp->tqueue.next=0;
  sp->tqueue.sync=0;
  sp->tqueue.routine=(void *)(void *)isac_bh;
  sp->tqueue.data=sp;

  BufQueueInit(&sp->rq);
  BufQueueInit(&sp->sq);

  sp->stlist=NULL;
    
  sp->ph_active=0;

  sp->dlogflag=0;
  sp->debug=0;
  
#ifdef DEBUG_MAGIC
  sp->magic=301271;
#endif  

  cards[sp->cardnr].sp=sp;
  init_tei(sp);

  init_hscxstate(sp,0);
  init_hscxstate(sp,1);
  
  modehscx(sp->hs,0,0);
  modehscx(sp->hs+1,0,0);

  writeisac(sp->membase,ISAC_MASK,0x0); 
}

static int get_irq(int cardnr)
{
  struct IsdnCard *card=cards+cardnr;
  long flags;

  save_flags(flags);
  cli();
  irq2dev_map[card->interrupt]=(void *)card->sp;
  if (request_irq(card->interrupt,&teles_interrupt,
    SA_INTERRUPT,"teles")) {
    printk("Teles couldn't get interrupt\n");
    irq2dev_map[card->interrupt]=NULL;
    restore_flags(flags);
    return(!0);
  }
  restore_flags(flags);
  return(0);
}

static int release_irq(int cardnr)
{
  struct IsdnCard *card=cards+cardnr;
  
  irq2dev_map[card->interrupt]=NULL;
  free_irq(card->interrupt);
}

void close_hscxstate(struct HscxState *hs)
{
  hs->inuse=0;

  if (hs->init) {
    BufPoolFree(&hs->smallpool);
    BufPoolFree(&hs->rbufpool);
    BufPoolFree(&hs->sbufpool);
  }
  
  hs->init=0;
}
    
void closecard(int cardnr)
{
  struct IsdnCardState *sp=cards[cardnr].sp;

  release_tei(sp);
  cards[cardnr].sp=NULL;
  

  Sfree(sp->dlogspace);
  
  BufPoolFree(&sp->smallpool);
  BufPoolFree(&sp->rbufpool);
  BufPoolFree(&sp->sbufpool);

  close_hscxstate(sp->hs+1);
  close_hscxstate(sp->hs);
  
  Sfree((void *)sp);
}

void teles_inithardware(void)
{
  int i;
  
  for(i=0;i<nrcards;i++)
    if (checkcard(i)) {
      initcard(i);
      if (get_irq(i)) 
	closecard(i);
  }
}

void teles_closehardware(void)
{
  int i;
  
  for(i=0;i<nrcards;i++)
    if (cards[i].sp) {
      release_irq(i);
      closecard(i);
  }
}

static void hscx_xmitframe(struct PStack *st,
  struct BufHeader *ibh)
{
  struct IsdnCardState *sp=(struct IsdnCardState *)
    st->l1.hardware;
  struct HscxState *hsp=sp->hs+st->l1.hscx;
  long flags;

#ifdef DEBUG_MAGIC  
  if (hsp->magic!=301270) {
    printk("hscx_xmitframe magic not 301270\n");
    return;
  }
#endif
  
  if (sp->debug)
    printk("hscx_xmitframe\n");

  BufQueueLink(&hsp->sq,ibh);
  if (hsp->xmtibh) return;

  save_flags(flags);cli();
  if (!hsp->xmtibh)
    if (!BufQueueUnlink(&hsp->xmtibh,&hsp->sq)) {
      hsp->sendptr=0;
      hscx_fill_fifo(hsp);
    }
  restore_flags(flags);
}

extern struct IsdnBuffers *tracebuf;

static void hscx_discardq(struct PStack *st,int pr,void *heldby,
  int releasetoo)
{
  struct IsdnCardState *sp=(struct IsdnCardState *)
    st->l1.hardware;
  struct HscxState *hsp=sp->hs+st->l1.hscx;

#ifdef DEBUG_MAGIC
  if (hsp->magic!=301270) {
    printk("hscx_discardq magic not 301270\n");
    return;
  }
#endif
  
  BufQueueDiscard(&hsp->sq,pr,heldby,releasetoo); 
}

static int open_hscxstate(struct IsdnCardState *sp,
  int hscx)
{
  struct HscxState *hsp=sp->hs+hscx;
 
  if (!hsp->init) {
    BufPoolInit(&hsp->sbufpool,HSCX_SBUF_ORDER,HSCX_SBUF_BPPS,
      HSCX_SBUF_MAXPAGES);
    BufPoolInit(&hsp->rbufpool,HSCX_RBUF_ORDER,HSCX_RBUF_BPPS,
      HSCX_RBUF_MAXPAGES);
    BufPoolInit(&hsp->smallpool,HSCX_SMALLBUF_ORDER,HSCX_SMALLBUF_BPPS,
      HSCX_SMALLBUF_MAXPAGES);
  }
  hsp->init=!0;
  
  BufQueueInit(&hsp->rq);
  BufQueueInit(&hsp->sq);

  hsp->rcvibh=NULL;
  hsp->xmtibh=NULL;
  hsp->rcvptr=0;
  hsp->sendptr=0;
  hsp->event=0;
  return(0);
}

int setstack_hscx(struct PStack *st,struct HscxState *hs)
{
  if (open_hscxstate(st->l1.hardware,hs->hscx))
    return(-1);
  
  st->l1.hscx=hs->hscx;
  st->l1.xmitframe=hscx_xmitframe;
  st->l1.discardq=hscx_discardq;

  st->l1.sbufpool=&hs->sbufpool;
  st->l1.rbufpool=&hs->rbufpool;
  st->l1.smallpool=&hs->smallpool;

  hs->st=st;
  return(0);
}

void teles_reportcard(int cardnr)
{
  struct IsdnCardState *sp=cards[cardnr].sp;

  printk("teles_reportcard\n");

}

