#include <linux/config.h>
#include <linux/module.h>

#include <asm/segment.h>

#define byte unsigned char

#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/timer.h>
#include <linux/sched.h>

#include "isdnif.h"
#include "teles.h"

extern struct IsdnCard cards[];
extern int nrcards;
extern int drid;
extern isdn_if iif;

static int  init_ds(int chan,int incoming);
static void release_ds(int chan); 
static void strcpyupto(char *dest,char *src,char upto);

static struct Fsm callcfsm={NULL,0,0};
struct Channel *chanlist;
static int chancount=0;

static void stat_debug(struct Channel *chanp,char *s)
{
  char tmp[64];
  sprintf(tmp,"%d HL->LL channel %d: %s\n",jiffies,chanp->chan,s);
  teles_putstatus(tmp);
}

enum {
  ST_NULL,        /*  0 inactive                                              */
  ST_OUT,         /*  1 outgoing, awaiting SETUP confirm                      */
  ST_CLEAR,       /*  2 call release, awaiting RELEASE confirm                */
  ST_OUT_W,       /*  3 outgoing, awaiting d-channel establishment            */
  ST_REL_W,       /*  4 awaiting d-channel release                            */
  ST_IN_W,        /*  5 incoming, awaiting d-channel establishment            */
  ST_IN,          /*  6 incoming call received                                */
  ST_IN_SETUP,    /*  7 incoming, SETUP response sent                         */
  ST_IN_DACT,     /*  8 incoming connected, no b-channel prot.                */
  ST_OUT_DACT,    /*  9 outgoing connected, no b-channel prot.                */
  ST_OUT_ESTB,    /* 10 outgoing connected, awaiting b-channel prot. estbl.   */
  ST_ACTIVE,      /* 11 active, b channel prot. established                   */
  ST_BC_HANGUP,   /* 12 call clear. (initiator), awaiting b channel prot. rel.*/
  ST_PRO_W,       /* 13 call clear. (initiator), DISCONNECT req. sent         */
  ST_ANT_W,       /* 14 call clear. (receiver), awaiting DISCONNECT ind.      */
};

#define STATE_COUNT (ST_ANT_W+1)

enum {
  EV_DIAL,
  EV_SETUP_CNF,
  EV_ACCEPTB,
  EV_DL_ESTABLISH,
  EV_DL_RELEASE,
  EV_DISCONNECT_CNF,
  EV_DISCONNECT_IND,
  EV_RELEASE_CNF,
  EV_DLEST,
  EV_DLRL,
  EV_SETUP_IND,
  EV_RELEASE_IND,
  EV_ACCEPTD,
  EV_SETUP_CMPL_IND,
  EV_BC_EST,
  EV_WRITEBUF,
  EV_DATAIN,
  EV_HANGUP,
  EV_BC_REL,
};

#define EVENT_COUNT (EV_BC_REL+1)

static void r1(struct FsmInst *fi,int event,void *arg)
{
  isdn_ctrl *ic=arg;
  struct Channel *chanp=fi->userdata;

  chanp->para.itc=8;
  strcpyupto(chanp->para.called,ic->num,',');

  chanp->is.l3.hdown(&chanp->is,CC_DLEST,NULL);
  FsmChangeState(fi,ST_OUT_W);  
}

static void r2(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  chanp->is.l3.hdown(&chanp->is,CC_RELEASE_REQ,NULL);

  if (chanp->debug&1) stat_debug(chanp,"STAT_DHUP");
  ic.driver=drid;
  ic.command=ISDN_STAT_DHUP;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
  
  FsmChangeState(fi,ST_CLEAR);  
}

static void r3(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  chanp->is.l3.hdown(&chanp->is,CC_DLRL,NULL);

  FsmChangeState(fi,ST_REL_W);  
}

static void r4(struct FsmInst *fi,int event,void *arg)
{
  FsmChangeState(fi,ST_NULL);  
}

static void r5(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  chanp->para.callref=chanp->outcallref;

  chanp->outcallref++;
  if (chanp->outcallref==128) chanp->outcallref=64;
  
  chanp->is.l3.hdown(&chanp->is,CC_SETUP_REQ,NULL);

  FsmChangeState(fi,ST_OUT);  
}

static void r6(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  chanp->is.l3.hdown(&chanp->is,CC_DLEST,NULL);
  FsmChangeState(fi,ST_IN_W);  
}

static void r7(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  chanp->is.l3.hdown(&chanp->is,CC_ALERTING_REQ,NULL);

  FsmChangeState(fi,ST_IN);  

  if (chanp->debug&1) stat_debug(chanp,"STAT_ICALL");
  ic.driver=drid;
  ic.command=ISDN_STAT_ICALL;
  ic.arg=chanp->chan;
  sprintf(ic.num,"%s,7,0,%s",chanp->para.calling,chanp->para.called);
  iif.statcallb(&ic);
}

static void r8(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  chanp->is.l3.hdown(&chanp->is,CC_SETUP_RSP,NULL);
  FsmChangeState(fi,ST_IN_SETUP);  

}

static void r9(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  init_ds(chanp->chan,!0);

  switch(chanp->l2_active_protocol) {
  case(ISDN_PROTO_L2_X75I):
    FsmChangeState(fi,ST_IN_DACT);  
    if (chanp->debug&1) stat_debug(chanp,"STAT_DCONN");
    ic.driver=drid;
    ic.command=ISDN_STAT_DCONN;
    ic.arg=chanp->chan;
    iif.statcallb(&ic);
    break;
  case(ISDN_PROTO_L2_HDLC):
    FsmChangeState(fi,ST_ACTIVE);  
    if (chanp->debug&1) stat_debug(chanp,"STAT_DCONN");
    ic.driver=drid;
    ic.command=ISDN_STAT_DCONN;
    ic.arg=chanp->chan;
    iif.statcallb(&ic);
    if (chanp->debug&1) stat_debug(chanp,"STAT_BCONN");
    ic.driver=drid;
    ic.command=ISDN_STAT_BCONN;
    ic.arg=chanp->chan;
    iif.statcallb(&ic);
    break;
  }
}

static void r10(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  FsmChangeState(fi,ST_OUT_DACT);  

  init_ds(chanp->chan,0);

  if (chanp->debug&1) stat_debug(chanp,"STAT_DCONN");
  ic.driver=drid;
  ic.command=ISDN_STAT_DCONN;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
}

static void r11(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  switch(chanp->l2_active_protocol) {
  case(ISDN_PROTO_L2_X75I):
    FsmChangeState(fi,ST_OUT_ESTB);  
    chanp->ds.l2.hdown(&chanp->ds,DL_ESTABLISH);
    break;
  case(ISDN_PROTO_L2_HDLC):
    FsmChangeState(fi,ST_ACTIVE);
    if (chanp->debug&1) stat_debug(chanp,"STAT_BCONN");
    ic.driver=drid;
    ic.command=ISDN_STAT_BCONN;
    ic.arg=chanp->chan;
    iif.statcallb(&ic);
    break;
  }
}

static void r12(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  FsmChangeState(fi,ST_ACTIVE);  

  if (chanp->debug&1) stat_debug(chanp,"STAT_BCONN");
  ic.driver=drid;
  ic.command=ISDN_STAT_BCONN;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
}

static void r13(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  chanp->ds.l2.hdown(&chanp->ds,DL_DATA,arg);
}

static void prp(byte *p,int size)
{
  while(size--) printk("%2x ",*p++);
  printk("\n");
}

static void r14(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  struct BufHeader *ibh=arg;
  byte *ptr;
  int size;

  ptr=DATAPTR(ibh);
  ptr+=chanp->ds.l2.ihsize;
  size=ibh->datasize-chanp->ds.l2.ihsize;
#if 0
  prp(ptr,size);
#endif
  iif.rcvcallb(drid,chanp->chan,ptr,size);
  BufPoolRelease(ibh);    
}

static void r15(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  switch(chanp->l2_active_protocol) {
  case(ISDN_PROTO_L2_X75I):
    FsmChangeState(fi,ST_BC_HANGUP);  
    chanp->ds.l2.hdown(&chanp->ds,DL_RELEASE,NULL);
    break;
  case(ISDN_PROTO_L2_HDLC):
    FsmChangeState(fi,ST_PRO_W);  
    chanp->is.l3.hdown(&chanp->is,CC_DISCONNECT_REQ,NULL);

    break;
  }
}

static void r16(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;

  FsmChangeState(fi,ST_PRO_W);  
  chanp->is.l3.hdown(&chanp->is,CC_DISCONNECT_REQ,NULL);
}

static void r17(struct FsmInst *fi,int event,void *arg)
{
  FsmChangeState(fi,ST_ANT_W);
}

static void r18(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  release_ds(chanp->chan);
  chanp->is.l3.hdown(&chanp->is,CC_DLRL,NULL);

  FsmChangeState(fi,ST_REL_W);  

  if (chanp->debug&1) stat_debug(chanp,"STAT_BHUP");
  ic.driver=drid;
  ic.command=ISDN_STAT_BHUP;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);

  if (chanp->debug&1) stat_debug(chanp,"STAT_DHUP");
  ic.driver=drid;
  ic.command=ISDN_STAT_DHUP;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
}

static void r19(struct FsmInst *fi,int event,void *arg)
{
  struct Channel *chanp=fi->userdata;
  isdn_ctrl ic;

  release_ds(chanp->chan);
  chanp->is.l3.hdown(&chanp->is,CC_RELEASE_REQ,NULL);

  FsmChangeState(fi,ST_CLEAR);  

  if (chanp->debug&1) stat_debug(chanp,"STAT_DHUP");
  ic.driver=drid;
  ic.command=ISDN_STAT_DHUP;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
}


static struct FsmNode fnlist[]={
  {ST_NULL,EV_DIAL,r1},
  {ST_OUT_W,EV_DLEST,r5},
  {ST_OUT,EV_DISCONNECT_IND,r2},
  {ST_CLEAR,EV_RELEASE_CNF,r3},
  {ST_REL_W,EV_DLRL,r4},
  {ST_NULL,EV_SETUP_IND,r6},
  {ST_IN_W,EV_DLEST,r7},
  {ST_IN,EV_RELEASE_IND,r3},
  {ST_IN,EV_ACCEPTD,r8},
  {ST_IN_SETUP,EV_SETUP_CMPL_IND,r9},
  {ST_OUT,EV_SETUP_CNF,r10},
  {ST_OUT_DACT,EV_ACCEPTB,r11},
  {ST_OUT_ESTB,EV_BC_EST,r12},
  {ST_IN_DACT,EV_BC_EST,r12},
  {ST_ACTIVE,EV_WRITEBUF,r13},
  {ST_ACTIVE,EV_DATAIN,r14},
  {ST_ACTIVE,EV_HANGUP,r15},
  {ST_BC_HANGUP,EV_BC_REL,r16},
  {ST_ACTIVE,EV_BC_REL,r17},
  {ST_ACTIVE,EV_DISCONNECT_IND,r19},
  {ST_PRO_W,EV_RELEASE_IND,r18},
  {ST_ANT_W,EV_DISCONNECT_IND,r19},
};
#define FNCOUNT (sizeof(fnlist)/sizeof(struct FsmNode))

void CallcNew(void)
{
  FsmNew(&callcfsm,STATE_COUNT,EVENT_COUNT,fnlist,
    FNCOUNT);
}

void CallcFree(void)
{
  FsmFree(&callcfsm);
}

static void release_ds(int chan) 
{
  struct PStack *st=&chanlist[chan].ds;
  struct IsdnCardState *sp;
  struct HscxState *hsp;
  
  sp=st->l1.hardware;
  hsp=sp->hs+chanlist[chan].hscx;

  modehscx(hsp,0,0);
  close_hscxstate(hsp);

  switch(chanlist[chan].l2_active_protocol) {
  case(ISDN_PROTO_L2_X75I):
    releasestack_isdnl2(st);
    break;
  case(ISDN_PROTO_L2_HDLC):
    releasestack_transl2(st);
    break;
  }  
}

static void ll_handler(struct PStack *st,int pr,
  struct BufHeader *ibh)
{
  struct Channel *chanp=(struct Channel *)st->l4.userdata;

  switch(pr) {
  case(CC_DISCONNECT_IND):
    FsmEvent(&chanp->fi,EV_DISCONNECT_IND,NULL);
    break;
  case(CC_RELEASE_CNF):
    FsmEvent(&chanp->fi,EV_RELEASE_CNF,NULL);
    break;
  case(CC_DLEST):
    FsmEvent(&chanp->fi,EV_DLEST,NULL);
    break;
  case(CC_DLRL):
    FsmEvent(&chanp->fi,EV_DLRL,NULL);
    break;
  case(CC_SETUP_IND):
    FsmEvent(&chanp->fi,EV_SETUP_IND,NULL);
    break;
  case(CC_RELEASE_IND):
    FsmEvent(&chanp->fi,EV_RELEASE_IND,NULL);
    break;
  case(CC_SETUP_COMPLETE_IND):
    FsmEvent(&chanp->fi,EV_SETUP_CMPL_IND,NULL);
    break;
  case(CC_SETUP_CNF):
    FsmEvent(&chanp->fi,EV_SETUP_CNF,NULL);
    break;
  }
}

static void init_is(int chan,unsigned int ces)
{
  struct PStack *st=&(chanlist[chan].is);
  struct IsdnCardState *sp=chanlist[chan].sp;

  setstack_teles(st,sp);

  st->l2.sap=0;

  st->l2.tei=255;
  st->l2.state=1;

  st->l2.ces=ces;
  st->l2.extended=!0;
  st->l2.laptype=LAPD;
  st->l2.window=1;
  st->l2.orig=!0;

  setstack_isdnl2(st);
  setstack_isdnl3(st);
  st->l2.debug=0;
  st->l3.debug=0;
  
  st->l4.hup=ll_handler;
  st->l4.userdata=chanlist+chan;
  st->l4.writewakeup=NULL;
  st->pa=&chanlist[chan].para;
  teles_addlist(sp,st);
}

static void callc_debug(struct FsmInst *fi,char *s)
{
  char str[64];
  struct Channel *chanp=fi->userdata;

  sprintf(str,"Channel %d: %s\n",chanp->chan,s);
  teles_putstatus(str);
}

static void init_chan(int chan,int cardnr,int hscx,
  unsigned int ces)
{
  struct IsdnCard *card=cards+cardnr;
  struct Channel *chanp=chanlist+chan;

  chanp->sp=card->sp;
  chanp->hscx=hscx;
  chanp->chan=chan;
  chanp->incoming=0;
  chanp->debug=0;
  init_is(chan,ces);

  chanp->fi.fsm=&callcfsm;
  chanp->fi.state=ST_NULL;
  chanp->fi.debug=0;
  chanp->fi.userdata=chanp;
  chanp->fi.printdebug=callc_debug;
  chanp->outcallref=64;
}

int CallcNewChan(void)
{
  int i,ces,c;

  chancount=0;
  for(i=0;i<nrcards;i++)
    if (cards[i].sp) 
      chancount+=2;

  chanlist=(struct Channel *)Smalloc(sizeof(struct Channel)*
    chancount,GFP_KERNEL,"chanlist");

  c=0;  
  ces=randomces();
  for(i=0;i<nrcards;i++)
    if (cards[i].sp) {
      init_chan(c++,i,1,ces++);
      ces%=0xffff;
      init_chan(c++,i,0,ces++);
      ces%=0xffff;
    }
  printk("channels %d\n",chancount);
  return(chancount);

}

static void release_is(int chan)
{
  struct PStack *st=&chanlist[chan].is;

  releasestack_isdnl2(st);
    
  teles_rmlist(st->l1.hardware,st);  

  BufQueueRelease(&st->l2.i_queue);
}

static void release_chan(int chan)
{
#if 0
  release_ds(chan);
#endif
  release_is(chan);
}
  
void CallcFreeChan(void)
{
  int i;

  for(i=0;i<chancount;i++)
    release_chan(i);
  Sfree((void *)chanlist);
}

static void lldata_handler(struct PStack *st,int pr,
  struct BufHeader *ibh)
{
  struct Channel *chanp=(struct Channel *)st->l4.userdata;

  switch(pr) {
  case(DL_DATA):
    if (FsmEvent(&chanp->fi,EV_DATAIN,ibh))
      BufPoolRelease(ibh);
    break;
  case(DL_ESTABLISH):
    FsmEvent(&chanp->fi,EV_BC_EST,NULL);
    break;
  case(DL_RELEASE):
    FsmEvent(&chanp->fi,EV_BC_REL,NULL);
    break;
  }
}

static void ll_writewakeup(struct PStack *st)
{
  struct Channel *chanp=st->l4.userdata;
  isdn_ctrl ic;

  ic.driver=drid;
  ic.command=ISDN_STAT_BSENT;
  ic.arg=chanp->chan;
  iif.statcallb(&ic);
}

static int init_ds(int chan,int incoming)
{
  struct PStack *st=&(chanlist[chan].ds);
  struct IsdnCardState *sp=(struct IsdnCardState *)
    chanlist[chan].is.l1.hardware;
  struct HscxState *hsp=sp->hs+chanlist[chan].hscx;

  st->l1.hardware=sp;

  hsp->mode=2;
  hsp->transbufsize=4000;
  
  if (setstack_hscx(st,hsp))
    return(-1);
  
  st->l2.state=4;
  st->l2.extended=0;
  st->l2.laptype=LAPB;
  st->l2.window=3;
  st->l2.orig=!incoming;

  switch(chanlist[chan].l2_protocol) {
  case(ISDN_PROTO_L2_X75I):
    setstack_isdnl2(st);
    break;
  case(ISDN_PROTO_L2_HDLC):
    setstack_transl2(st);
    break;
  }

  chanlist[chan].l2_active_protocol=chanlist[chan].l2_protocol;

  st->l2.debug=0;
  st->l3.debug=0;
  
  st->l3.hup=lldata_handler;
  st->l4.userdata=chanlist+chan;
  st->l4.writewakeup=ll_writewakeup;

  modehscx(hsp,2,chanlist[chan].para.bchannel-1);
  return(0);
  
}

static void channel_report(int i)
{
  struct Channel *chanp=chanlist+i;
}

static void command_debug(struct Channel *chanp,char *s)
{
  char tmp[64];
  sprintf(tmp,"%d LL->HL channel %d: %s\n",jiffies,chanp->chan,s);
  teles_putstatus(tmp);
}

int teles_command(isdn_ctrl *ic)
{
  struct Channel *chanp;
  char tmp[64];
  int i;
  unsigned int num;

  switch(ic->command) {
  case(ISDN_CMD_SETEAZ):
    chanp=chanlist+ic->arg;
    if (chanp->debug&1) command_debug(chanp,"SETEAZ");
    return(0);
  case(ISDN_CMD_DIAL):
    chanp=chanlist+(ic->arg&0xff);
    if (chanp->debug&1) {
      sprintf(tmp,"DIAL %s",ic->num);
      command_debug(chanp,tmp);
    }
    FsmEvent(&chanp->fi,EV_DIAL,ic);
    return(0);
  case(ISDN_CMD_ACCEPTB):
    chanp=chanlist+ic->arg;
    if (chanp->debug&1) command_debug(chanp,"ACCEPTB");
    FsmEvent(&chanp->fi,EV_ACCEPTB,NULL);
    break;
  case(ISDN_CMD_ACCEPTD):
    chanp=chanlist+ic->arg;
    if (chanp->debug&1) command_debug(chanp,"ACCEPTD");
    FsmEvent(&chanp->fi,EV_ACCEPTD,NULL);
    break;
  case(ISDN_CMD_HANGUP):
    chanp=chanlist+ic->arg;
    if (chanp->debug&1) command_debug(chanp,"HANGUP");
    FsmEvent(&chanp->fi,EV_HANGUP,NULL);
    break;
  case(ISDN_CMD_LOCK):
    MOD_INC_USE_COUNT; 
    break;
  case(ISDN_CMD_UNLOCK):
    MOD_DEC_USE_COUNT;
    break;
  case(ISDN_CMD_IOCTL):
    switch(ic->arg) {
    case(0):
      for(i=0;i<nrcards;i++)
        if (cards[i].sp)
          teles_reportcard(i);
      for(i=0;i<chancount;i++)
        channel_report(i);
      break;
    case(1):
      num=*(unsigned int *)ic->num;
      for(i=0;i<chancount;i++) {
        chanlist[i].debug=num&1;
        chanlist[i].fi.debug=num&2;
      }
      for(i=0;i<nrcards;i++)
        if (cards[i].sp)
          cards[i].sp->dlogflag=num&4;
      sprintf(tmp,"debugging flags set to %x\n",num);
      teles_putstatus(tmp);
      break;
    }
    break;
  case(ISDN_CMD_SETL2):
    chanp=chanlist+(ic->arg&0xff);
    if (chanp->debug&1) {
      sprintf(tmp,"SETL2 %d",ic->arg>>8);
      command_debug(chanp,tmp);
    }
    chanp->l2_protocol=ic->arg>>8;
    break;  
  default:
    break;
  }

  return(0);
}

int teles_writebuf(int chan,u_char *buf,int count,
  int user)
{
  struct Channel *chanp=chanlist+chan;
  struct PStack *st=&chanp->ds;
  struct BufHeader *ibh;
  int err;
  byte *ptr;
  
  err=BufPoolGet(&ibh,st->l1.sbufpool,GFP_ATOMIC,st,21);
  if (err) 
    return(0);

  if (count>BUFFER_SIZE(HSCX_SBUF_ORDER,HSCX_SBUF_BPPS)) {  
    printk("teles_writebuf: packet too large!\n");
    return(0);
  }

  ptr=DATAPTR(ibh);
  ptr+=st->l2.ihsize;
  if (user)
    memcpy_fromfs(ptr,buf,count);
  else
    memcpy(ptr,buf,count);
  ibh->datasize=count+st->l2.ihsize;
  
  if (FsmEvent(&chanp->fi,EV_WRITEBUF,ibh)) {
    BufPoolRelease(ibh);
    return(0);
  }
  else
    return(count);
}

static void strcpyupto(char *dest,char *src,char upto)
{
  while (*src&&(*src!=upto)) *dest++=*src++;
  *dest='\0';
}