/* $Id: isdnctrl.c,v 1.7 1995/04/29 13:13:44 fritz Exp fritz $
 *
 * ISDN driver for Linux. (Control-Utility)
 *
 * Copyright 1994,95 by Fritz Elfert (fritz@wuemaus.franken.de)
 * Copyright 1995 Thinking Objects Software GmbH Wuerzburg
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Log: isdnctrl.c,v $
 * Revision 1.7  1995/04/29  13:13:44  fritz
 * Added new command verbose.
 *
 * Revision 1.6  1995/04/23  13:38:34  fritz
 * Adapted addphone and delphone to support changes in isdn.c
 *
 * Revision 1.5  1995/03/25  23:35:35  fritz
 * Added ihup-Feature.
 *
 * Revision 1.4  1995/03/15  12:44:15  fritz
 * Added generic conversion-routines for keyword<->value conversion.
 * Added display of phone-numbers in list-routine
 * Corrected some typos
 *
 * Revision 1.3  1995/02/20  03:38:59  fritz
 * Added getmax and rmax for performance-tests of tty-driver.
 *
 * Revision 1.2  1995/01/29  23:27:52  fritz
 * Added keywords: list, l2_proto, l3_proto, huptimeout, chargehup and
 * encap.
 *
 * Revision 1.1  1995/01/09  07:35:35  fritz
 * Initial revision
 *
 *
 */

#undef  ISDN_DEBUG_MODEM_SENDOPT

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <isdn.h>
#include <isdnif.h>

char *l2protostr[] = {"x75i", "x75ui", "x75bui", "hdlc", "\0"};
int  l2protoval[]  = {ISDN_PROTO_L2_X75I, ISDN_PROTO_L2_X75UI,
		      ISDN_PROTO_L2_X75BUI, ISDN_PROTO_L2_HDLC, -1};

char *l3protostr[] = {"trans", "\0"};
int  l3protoval[]  = {ISDN_PROTO_L3_TRANS, -1};

char *pencapstr[]  = {"ethernet", "rawip", "ip", "\0"};
int  pencapval[]   = {ISDN_NET_ENCAP_ETHER, ISDN_NET_ENCAP_RAWIP,
		      ISDN_NET_ENCAP_IPTYP, -1};
char *cmd;

void
usage(void) {
  fprintf(stderr,"usage: %s <command> <options>\n",cmd);
  fprintf(stderr,"\n");
  fprintf(stderr,"where <command> is one of the following:\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"          addif [name]               (add net-interface)\n");
  fprintf(stderr,"          delif name                 (remove net-interface)\n");
  fprintf(stderr,"          addphone name in|out num   (add phone-number to interface)\n");
  fprintf(stderr,"          delphone name in|out num   (remove phone-number from interface)\n");
  fprintf(stderr,"          eaz name [eaz]             (get/set eaz for interface)\n");
  fprintf(stderr,"          huptimeout name [seconds]  (get/set hangup-timeout for interface)\n");
  fprintf(stderr,"          ihup name [on|off]         (get/set incoming-hangup for interface)\n");
  fprintf(stderr,"          chargehup name [on|off]    (get/set charge-hangup for interface)\n");
  fprintf(stderr,"          secure name [on|off]       (get/set secure-feature for interface)\n");
  fprintf(stderr,"          callback name [out|in|off] (get/set active callback-feature for interface)\n");
  fprintf(stderr,"          encap name [encapname]     (get/set packet-encapsulation for interface)\n");
  fprintf(stderr,"          l2_prot name [protocol]    (get/set layer-2-protocol for interface)\n");
  fprintf(stderr,"          l3_prot name [protocol]    (get/set layer-3-protocol for interface)\n");
  fprintf(stderr,"          list name                  (show current setup of interface)\n");
  fprintf(stderr,"          verbose <num>              (set verbose-level)\n");
  exit(-2);
}

static int
key2num(char *key, char **keytable, int *numtable) {
  int i = -1;
  while (strlen(keytable[++i]))
    if (!strcmp(keytable[i],key))
      return numtable[i];
  return -1;
}

static char*
num2key(int num, char **keytable, int *numtable) {
  int i = -1;
  while (numtable[++i]>=0)
    if (numtable[i]==num)
      return keytable[i];
  return "???";
}

static void
listif(int isdnctrl, char *name, int errexit) {
  isdn_net_ioctl_cfg   cfg;
  union p {
    isdn_net_ioctl_phone phone;
    char                 n[1024];
  } ph;
  char nn[1024];

  strcpy(cfg.name,name);
  if (ioctl(isdnctrl,ISDN_IOCTL_NET_GETCFG,&cfg)<0) {
    if (errexit) {
      perror(name);
      exit(-1);
    } else
      return;
  }
  strcpy(ph.phone.name,name);
  ph.phone.outgoing = 0;
  if (ioctl(isdnctrl,ISDN_IOCTL_NET_GETNUM,&ph.phone)<0) {
    if (errexit) {
      perror(name);
      exit(-1);
    } else
      return;
  }
  strcpy(nn,ph.n);
  strcpy(ph.phone.name,name);
  ph.phone.outgoing = 1;
  if (ioctl(isdnctrl,ISDN_IOCTL_NET_GETNUM,&ph.phone)<0) {
    if (errexit) {
      perror(name);
      exit(-1);
    } else
      return;
  }
  printf("\nCurrent setup of interface '%s':\n\n",cfg.name);
  printf("EAZ:              %s\n",cfg.eaz);
  printf("Phone number(s):\n");
  printf("  Outgoing:       %s\n",ph.n);
  printf("  Incoming:       %s\n",nn);
  printf("Secure:           %s\n",cfg.secure?"Yes":"No");
  printf("Callback:         %s\n",cfg.callback?"Yes":"No");
  printf("Hangup-Timeout:   %d\n",cfg.onhtime);
  printf("Incoming-Hangup:  %s\n",cfg.ihup?"Yes":"No");
  printf("ChargeHangup:     %s\n",cfg.chargehup?"Yes":"No");
  printf("Charge-Units:     %d\n",cfg.charge);
  printf("Layer-2-Protocol: %s\n",num2key(cfg.l2_proto,l2protostr,l2protoval));
  printf("Layer-3-Protocol: %s\n",num2key(cfg.l3_proto,l3protostr,l3protoval));
  printf("Encapsulation:    %s\n",num2key(cfg.p_encap,pencapstr,pencapval));
}

void
main(int argc, char **argv) {
  int fd;
  int i;
  int result;
  FILE *iflst;
  char *p;
  char s[255];
  isdn_net_ioctl_phone phone;
  isdn_net_ioctl_cfg   cfg;

  cmd = argv[0];
  if (argc>=2) {
    fd = open("/dev/isdnctrl",O_RDWR);
    if (fd < 0) {
      perror("/dev/isdnctrl");
      exit(-1);
    }
#ifdef  ISDN_DEBUG_MODEM_SENDOPT
    if (!strcmp(argv[1],"gmax")) {
      if ((result=ioctl(fd,ISDN_IOCTL_GETMAX,&i))<0) {
	perror("getmax");
	exit(-1);
      }
      printf("maxch = %d\n",i);
      close(fd);
      return;
    }
    if (!strcmp(argv[1],"rmax")) {
      if ((result=ioctl(fd,ISDN_IOCTL_RESETMAX,0))<0) {
	perror("resetmax");
	exit(-1);
      }
      printf("maxch set to 0\n");
      close(fd);
      return;
    }
#endif
    if (!strcmp(argv[1],"addif")) {
      if (argc==2)
	strcpy(s,"  ");
      else {
	if (strlen(argv[2])>8) {
	  fprintf(stderr,"name must not exceed 8 characters\n");
	  exit(-1);
	}
	strcpy(s,argv[2]);
      }
      if ((result=ioctl(fd,ISDN_IOCTL_NET_ADDIF,s))<0) {
	perror("addif");
	exit(-1);
      }
      printf("%s added\n",s);
      close(fd);
      return;
    }
    if (!strcmp(argv[1],"delif")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      if ((result=ioctl(fd,ISDN_IOCTL_NET_DELIF,argv[2]))<0) {
	perror(argv[2]);
	exit(-1);
      }
      printf("%s deleted\n",argv[2]);
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"addphone")) {
      if (argc<5)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      if (strcmp(argv[3],"in") && strcmp(argv[3],"out")) {
	fprintf(stderr,"Direction must be \"in\" or \"out\"\n");
	exit(-1);
      }
      phone.outgoing = strcmp(argv[3],"out")?0:1;
      if (strlen(argv[4])>20) {
	fprintf(stderr,"phone-number must not exceed 20 characters\n");
	exit(-1);
      }
      strcpy(phone.name,argv[2]);
      strcpy(phone.phone,argv[4]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_ADDNUM,&phone))<0) {
	perror(argv[2]);
	exit(-1);
      }
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"delphone")) {
      if (argc<5)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      if (strcmp(argv[3],"in") && strcmp(argv[3],"out")) {
	fprintf(stderr,"Direction must be \"in\" or \"out\"\n");
	exit(-1);
      }
      phone.outgoing = strcmp(argv[3],"out")?0:1;
      if (strlen(argv[4])>20) {
	fprintf(stderr,"phone-number must not exceed 20 characters\n");
	exit(-1);
      }
      strcpy(phone.name,argv[2]);
      strcpy(phone.phone,argv[4]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_DELNUM,&phone))<0) {
	perror(argv[2]);
	exit(-1);
      }
      close(fd);
      return;
    }
    if (!strcmp(argv[1],"list")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      if (!strcmp(argv[2],"all")) {
	char name[10];
	if ((iflst = fopen("/proc/net/dev","r"))==NULL) {
	  perror("/proc/net/dev");
	  exit(-1);
	}
	while (!feof(iflst)) {
	  fgets(s,sizeof(s),iflst);
	  if ((p = strchr(s,':'))) {
	    *p = 0;
	    sscanf(s,"%s",name);
	    listif(fd,name,0);
	  }
	}
	fclose(iflst);
      } else
	listif(fd,argv[2],1);
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"eaz")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	strncpy(cfg.eaz,argv[3],sizeof(cfg.eaz)-1);
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("EAZ for %s is %s\n",cfg.name,cfg.eaz);
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"verbose")) {
      if (argc<2)
	usage();
      i = -1;
      sscanf(argv[2],"%d",&i);
      if (i<0) {
	fprintf(stderr,"Verbose-level must be >= 0\n");
	exit(-1);
      }
      if ((result=ioctl(fd,ISDN_IOCTL_SET_VERBOSE,i))<0) {
	perror("IOCTL_SET_VERBOSE");
	exit(-1);
      }
      printf("Verbose-level set to %d.\n",i);
      close(fd);
      return;
    }
    if (!strcmp(argv[1],"huptimeout")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	sscanf(argv[3],"%d",&i);
	if (i<0) {
	  fprintf(stderr,"Hangup-Timeout must be >= 0\n");
	  exit(-1);
	}
	cfg.onhtime = i;
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Hangup-Timeout for %s is %d sec.\n",cfg.name,cfg.onhtime);
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"chargehup")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	if (strcmp(argv[3],"on") && strcmp(argv[3],"off")) {
	  fprintf(stderr,"Charge-Hangup must be 'on' or 'off'\n");
	  exit(-1);
	}
	cfg.chargehup = strcmp(argv[3],"off");
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Charge-Hangup for %s is %s\n",cfg.name,cfg.chargehup?"on":"off");
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"ihup")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	if (strcmp(argv[3],"on") && strcmp(argv[3],"off")) {
	  fprintf(stderr,"Incoming-Hangup must be 'on' or 'off'\n");
	  exit(-1);
	}
	cfg.ihup = strcmp(argv[3],"off");
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Incoming-Hangup for %s is %s\n",cfg.name,cfg.ihup?"on":"off");
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"secure")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	if (strcmp(argv[3],"on") && strcmp(argv[3],"off")) {
	  fprintf(stderr,"Secure-parameter must be 'on' or 'off'\n");
	  exit(-1);
	}
	cfg.secure = strcmp(argv[3],"off");
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Security for %s is %s\n",cfg.name,cfg.secure?"on":"off");
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"callback")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = -1;
	if (strcmp(argv[3],"on") && strcmp(argv[3],"off")) {
	  fprintf(stderr,"Callback-parameter must be 'on' or 'off'\n");
	  exit(-1);
	}
	cfg.callback = strcmp(argv[3],"off");
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Callback for %s is %s\n",cfg.name,cfg.callback?"on":"off");
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"l2_prot")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = key2num(argv[3],l2protostr,l2protoval);
	if (i<0) {
	  fprintf(stderr,"Layer-2-Protocol must be one of the following:\n");
	  i = 0;
	  while (strlen(l2protostr[i]))
	    fprintf(stderr,"\t\"%s\"\n",l2protostr[i++]);
	  exit(-1);
	}
	cfg.l2_proto = i;
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Layer-2-Protocol for %s is %s\n",cfg.name,
	     num2key(cfg.l2_proto,l2protostr,l2protoval));
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"l3_prot")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = key2num(argv[3],l3protostr,l3protoval);
	if (i<0) {
	  fprintf(stderr,"Layer-3-Protocol must be one of the following:\n");
	  i = 0;
	  while (strlen(l3protostr[i]))
	    fprintf(stderr,"\t\"%s\"\n",l3protostr[i++]);
	  exit(-1);
	}
	cfg.l3_proto = i;
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Layer-3-Protocol for %s is %s\n",cfg.name,
	     num2key(cfg.l3_proto,l3protostr,l3protoval));
      close(fd);
      return;
    }  
    if (!strcmp(argv[1],"encap")) {
      if (argc<3)
	usage();
      if (strlen(argv[2])>8) {
	fprintf(stderr,"name must not exceed 8 characters\n");
	exit(-1);
      }
      strcpy(cfg.name,argv[2]);
      if ((result=ioctl(fd,ISDN_IOCTL_NET_GETCFG,&cfg))<0) {
	perror(argv[2]);
	exit(-1);
      }
      if (argc>3) {
	i = key2num(argv[3],pencapstr,pencapval);
	if (i<0) {
	  fprintf(stderr,"Encapsulation must be one of the following:\n");
	  i = 0;
	  while (strlen(pencapstr[i]))
	    fprintf(stderr,"\t\"%s\"\n",pencapstr[i++]);
	  exit(-1);
	}
	cfg.p_encap = i;
	if ((result=ioctl(fd,ISDN_IOCTL_NET_SETCFG,&cfg))<0) {
	  perror(argv[2]);
	  exit(-1);
	}
      }
      printf("Encapsulation for %s is %s\n",cfg.name,
	     num2key(cfg.p_encap,pencapstr,pencapval));
      close(fd);
      return;
    }  
    close(fd);
  }
  usage();
}



