/*
 *  ntfs.h
 *
 *  Copyright (C) 1995 Martin von Lwis
 */

#include <linux/types.h>

typedef struct _ntfs_volume{
	int blocksize;
	int clusterfactor;
	int clustersize;
	int mft_recordsize;
	int mft_clusters_per_record;
	int index_recordsize;
	int index_clusters_per_record;
	int mft_cluster;
	unsigned char *mft;
	uid_t uid;
	gid_t gid;
	umode_t umask;
	union{
		int fd;		/* file descriptor for the tools */
		void *sb;	/* pointer to super block for the kernel */
	}u;
}ntfs_volume;

#define NTFS_FD(vol)	((vol)->u.fd)
#define NTFS_SB(vol)	((struct super_block*)(vol)->u.sb)
#define NTFS_SB2VOL(sb)	((ntfs_volume*)(sb)->u.generic_sbp)
#define NTFS_INO2VOL(ino)	((ntfs_volume*)((ino)->i_sb->u.generic_sbp))

typedef struct {
	int i_number;
	unsigned char* attr;
}ntfs_inode;

typedef void* (*copyfunction)(void*,const void*,size_t);

#define AT_STANDARD_INFORMATION	0x10
#define AT_ATTRIBUTE_LIST		0x20
#define AT_FILE_NAME			0x30
#define AT_VOLUME_VERSION		0x40
#define AT_SECURITY_DESCRIPTOR	0x50
#define AT_VOLUME_NAME			0x60
#define AT_VOLUME_INFORMATION	0x70
#define AT_DATA					0x80
#define AT_INDEX_ROOT			0x90
#define AT_INDEX_ALLOCATION		0xA0
#define AT_BITMAP				0xB0
#define AT_SYMBOLIC_LINK		0xC0
#define AT_EA_INFORMATION		0xD0
#define AT_EA					0xE0

#define FILE_MFT				0
#define FILE_MFTMIRR			1
#define FILE_LOGFILE			2
#define FILE_VOLUME				3
#define FILE_ATTRDEF			4
#define FILE_ROOT				5
#define FILE_BITMAP				6
#define FILE_BOOT				7
#define FILE_BADCLUS			8
#define FILE_QUOTA				9
#define FILE_UPCASE				10

extern ntfs_volume *the_vol;

void print_time(long long*);

void init_volume(ntfs_volume *vol,char *boot);
int ntfs_get_inode(ntfs_inode *ino);
int ntfs_read_mft_record(ntfs_volume *vol,int mftno,char *buf);
int ntfs_has_attr(char*ino,int attr,char*name);
int ntfs_get_attr_size(ntfs_volume *,char*ino,int inum,int attr,char*name);
int ntfs_attr_is_resident(ntfs_volume *,char*ino,int inum,int attr,char*name);
char *ntfs_get_attr(char *inode,int attr,char *name);
int ntfs_read_attr(ntfs_volume *vol,char *ino,int inum,int attr,
	char *name,int offset,char *buf,int length,copyfunction);
int ntfs_vcn_to_lcn(ntfs_volume*,char*,int);
int ntfs_get_clusters(ntfs_volume *vol,int start_cluster,size_t start_offs,
	size_t length,void *buf,copyfunction);
int ntfs_get_name(char*,char*);
int ntfs_getdir_byposition(ntfs_volume *vol,char*ino,int inum,long long *num,char *ret);
int ntfs_getdir_byname(ntfs_volume *vol,char*ino,int inum,const char *name,
	int namelen,char *ret);
void dumpdir(char*,int);
void list_attr_mem(char*);
time_t ntfs_ntutc2unixutc(long long);
void ntfs_uni2ascii(char*,char*,int);
void *ntfs_allocate(int size);
int ntfs_fixup_record(ntfs_volume*,char *record,int count);
int ntfs_check_index_record(ntfs_volume*,char *record);
int ntfs_check_mft_record(ntfs_volume*,char *record);
int ntfs_get_volumesize(ntfs_volume *);
int ntfs_get_free_cluster_count(ntfs_volume *vol,char*bitmap);
int ntfs_get_free_inode_count(ntfs_volume *,char*mft);
void* ntfs_memcpy(void*a,const void*b,size_t);

#define min(a,b)	((a) <= (b) ? (a) : (b))
#define max(a,b)	((a) >= (b) ? (a) : (b))
#define RESIDENT(attr)	(*((char*)attr+8)=='\0')
#define COMPRESSED(attr)	(*((char*)attr+0xC)!='\0')
#define DATASIZE(attr)	\
	(RESIDENT(attr)?*(short*)(attr+0x10):*(int*)(attr+0x30))
#define IS_MAGIC(a,b)	(*(int*)(a)==*(int*)(b))
#define IS_MFT_RECORD(a)	IS_MAGIC((a),"FILE")
#define IS_NTFS_VOLUME(a)	IS_MAGIC((a)+3,"NTFS")
#define IS_INDEX_RECORD(a)	IS_MAGIC((a),"INDX")

/* 'NTFS' in little endian */
#define NTFS_SUPER_MAGIC	0x5346544E
