/*
 * Copyright (C) 1995  Peter L Jones
 * See file COPYING before you even think about using this program.
 */
static char version[] = "$Id: userdel.c,v 1.2 1995/12/19 20:54:02 thanatos Exp $";
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <dirent.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*
 * I have not written supplementary groups code yet
 * - please update /etc/group by hand :-(
 * (options are accepted and ignored)
 */
#define NO_SUPPGROUPS

#define PASSWD_FILE "/etc/passwd"
#define GROUP_FILE "/etc/group"
const char SHADOW_SYSV[] = "/etc/shadow";
const char SHADOW_BSD[] = "/etc/master.passwd";	/* I got this name from "adduser" */

/*
 * Private local functions
 */
static int Shadow(void);
static int Parse(int argc, char *argv[]);
static void Unlock(void);
static int Validate(void);
static int Remove(void);

static void Usage(void);
static void Long_usage(void);
static int LockPW(void);
static int BackupPW(void);
static int DeleteUser(void);
static int DestroyHome(void);
#ifndef NO_SUPPGROUPS
static int LockGrp(void);
static int BackupGrp(void);
static int DeleteGroups(void);
#endif

static struct mainOptions {
	int	rmhome;
	const char*	username;
	const char*	homedir;
	const char*	pname;
} Opt = { 0, NULL, NULL, NULL, };


int main(int argc, char *argv[]) {
	Opt.pname = argv[0];
	if (atexit(Unlock)) {
		perror(argv[0]);
		return 1;
	}
	if (Shadow()) return 2;
	if (Parse(argc, argv)) return 3;
	if (Validate()) return 4;
	if (Remove()) return 5;
	return 0;
}


static int Remove(void) {
	if (LockPW()) return 1;
	if (BackupPW()) return 1;
	if (DeleteUser()) return 1;
#ifndef NO_SUPPGROUPS
	if (Opt.suppgroups) {
		if (LockGrp()) return 1;
		if (BackupGrp()) return 1;
		if (DeleteGroups()) return 1;
	}
#endif
	if (Opt.rmhome) if (DestroyHome()) return 1;
	return 0;
}

static int DeleteUser(void) {
	FILE *PASSWD_IN;
	FILE *PASSWD_OUT;
	static struct passwd *pwp;
	
	PASSWD_IN = fopen(PASSWD_FILE ".OLD", "r");
	if (!PASSWD_IN) {
		fprintf(stderr,"%s: could not read from %s: %s\n", Opt.pname, PASSWD_FILE ".OLD", strerror(errno));
		fclose(PASSWD_IN);
		return 1;
	}
	PASSWD_OUT = fopen(PASSWD_FILE, "w");
	if (!PASSWD_OUT) {
		fprintf(stderr,"%s: could not write to %s: %s\n", Opt.pname, PASSWD_FILE, strerror(errno));
		fclose(PASSWD_IN);
		return 1;
	}
	while ((pwp = fgetpwent(PASSWD_IN)) != NULL) {
		if (strcmp(pwp->pw_name,Opt.username)) {
			putpwent(pwp, PASSWD_OUT);
		}
	}
	fclose(PASSWD_IN);
	fclose(PASSWD_OUT);
	return 0;
}

static int DestroyHome(void) {
	char buffer[1024];
	sprintf(buffer, "/bin/rm -rf %s", Opt.homedir);
	if (system(buffer)) return 1;
	return 0;
}

#ifndef NO_SUPPGROUPS
static int DeleteGroups(void) {
	return 0;
}

static int BackupGrp(void) {
	if (system("/bin/cp " GROUP_FILE " " GROUP_FILE ".OLD")) return 1;
	chmod(GROUP_FILE ".OLD", 0);
	return 0;
}

static int LockGrp(void) {
	static int fd;
	if ((fd = open(GROUP_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create group file lock: %s\n", Opt.pname, strerror(errno));
		remove(PASSWD_FILE "~");
		return 1;
	} else close(fd);
	return 0;
}
#endif

static int Validate() {
	struct passwd *pwp;
	if (!(pwp = getpwnam(Opt.username))) {
		fprintf(stderr,"%s: '%s' is not a valid user name.\n", Opt.pname, Opt.username);
		return 1;
	}
	if (Opt.rmhome) Opt.homedir = strdup(pwp->pw_dir);
	return 0;
}

static int Parse(int argc, char *argv[]) {
	int c;
	char short_options[] = "hVr";
	struct option long_options[] = {
		{"help",		no_argument, 0, 'h'},
		{"version",		no_argument, 0, 'V'},
		{"remove",		no_argument, 0, 'r'},
		{0, 0, 0, 0}
	};

	while ((c = getopt_long(argc, argv, short_options,
			long_options, NULL)) != -1) switch (c) {
		case 'r':	Opt.rmhome++; break;

		case 'V':
			printf("%s\n",version);
			exit(0);

		case 'h':
			Long_usage();
			exit(0);

		default:
			Usage();
			return 1;
	}

	if (optind != argc-1) {
		fprintf(stderr,"%s: user name not specified\n", Opt.pname);
		Usage();
		return 1;
	}

	Opt.username=argv[optind];

	return 0;
}

static void Usage(void) {
	fprintf(stderr,"Try `%s --help' for more information.\n",Opt.pname);
}

static void Long_usage(void) {
	fprintf(stderr,"Usage: %s [option ...] username\n", Opt.pname);
	fprintf(stderr,
		"where 'option' is:\n"
		"  -h, --help\n"
		"  -V, --version\n"
		"  -r, --remove\n"
	);
}

static int Shadow(void) {
	/* if either /etc/shadow or /etc/master.passwd exists, this is shadow system - we don't work! */
	if (!access(SHADOW_SYSV,F_OK) || !access(SHADOW_BSD,F_OK)) {
		fprintf(stderr,"%s: Detected a SHADOW PASSWORD file on this system.  "
			"If you are not using shadow passwords, please remove the shadow file.\n", Opt.pname);
		return 1;
	}
	return 0;
}

static int BackupPW(void) {
	if (system("/bin/cp " PASSWD_FILE " " PASSWD_FILE ".OLD")) return 1;
	chmod(PASSWD_FILE ".OLD", 0);
	return 0;
}

static int LockPW(void) {
	/* lock the password and group files */
	static int fd;
	if ((fd = open(PASSWD_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create password lock: %s\n", Opt.pname, strerror(errno));
		return 1;
	} else close(fd);
	return 0;
}

static void Unlock(void) {
	remove(PASSWD_FILE "~");
	remove(GROUP_FILE "~");
}
