/*
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
/*static char sccsid[] = "from: @(#)announce.c  8.3 (Berkeley) 4/28/95";*/
static char rcsid[] = "$Id: announce.c,v 1.2 1993/08/01 18:29:37 mycroft Exp $";
#endif /* not lint */

#include "talk.h"
#include <pwd.h>
#include <signal.h>


extern char hostname[];
static int 
announce_proc(CTL_MSG *request, const char *remote_machine, const char *uthost);
static int announce_X(CTL_MSG *request, const char *remote_machine);
static int announce_noX(CTL_MSG *request, const char *remote_machine);
static void print_mesg(FILE *tf, CTL_MSG *request, const char *remote_machine);


/*
 * Announce an invitation to talk.
 *
 * Because the tty driver insists on attaching a terminal-less
 * process to any terminal that it writes on, we must fork a child
 * to protect ourselves
 */
int announce(CTL_MSG *request, const char *remote_machine, const char *uthost)
{
	int pid, val, status;

	if ((strstr(request->l_name,"\033"))||(strstr(remote_machine,"\033")))
        {
	    if (strstr(remote_machine,"\033"))
	    {
		syslog(LOG_WARNING, 
		"blocked VT100 BOMB to user: %s (from field in contains bomb)",
		 request->r_name);
	    } else {
		syslog(LOG_WARNING, 
		"blocked VT100 BOMB to user: %s (apparently from: %s)", 
		request->r_name, remote_machine);
	    }
	    return;
        }

	if (pid = fork()) {
		/* we are the parent, so wait for the child */
		if (pid == -1)		/* the fork failed */
			return (FAILED);
		do {
			val = wait(&status);
			if (val == -1) {
				if (errno == EINTR)
					continue;
				/* shouldn't happen */
				syslog(LOG_WARNING, "announce: wait: %m");
				return (FAILED);
			}
		} while (val != pid);
		if (status&0377 > 0)	/* we were killed by some signal */
			return (FAILED);
		/* Get the second byte, this is the exit/return code */
		return ((status >> 8) & 0377);
	}
	/* we are the child, go and do it */
	_exit(announce_proc(request, remote_machine, uthost));
}



/* See if user is on local X display 
 * run announce_X or announce_noX 
 */ 
static int 
announce_proc( CTL_MSG *request, const char *remote_machine, const char *uthost)
{
	if (strcmp(uthost,":0.0"))
		return( announce_noX(request, remote_machine));
	else
		return( announce_X(request, remote_machine));
}



/*
 * Announcement for X-windows:
 * Look wether user has an executable ~/.talkd.sh; if so, call it,
 * else fall back to non-X-announcement
 */
static int announce_X( CTL_MSG *request, const char *remote_machine)
{
 	struct passwd *recipient;
	char rcfile[256];
	struct stat st;
	char full_tty[32];

	if (recipient=getpwnam(request->r_name))
	{
		snprintf(rcfile,sizeof rcfile,"%s/.talkd.sh",recipient->pw_dir);
		/* check, whether file exists & is executable for owner;
                   we cannot use access(...,X_OK) because this will
                   always succeed if the file exists, since we're running
                   as root ... */
		if (stat(rcfile,&st)!= -1 && (st.st_mode&0100)!=0)
		{
			/* script ok, execute it */
			int pid, status;

			if (pid=fork()) {
				if (pid==-1) return(FAILED);

				/* parent, wait! */
#if 0				/* have to think about this.... */
				signal(SIGCHLD, SIG_IGN);
				waitpid(pid, &status, 0);
				syslog(LOG_ERR, "waitpid %0x", status);
				if (status==0 || status==0x1000)
					return (SUCCESS);
				return (PERMISSION_DENIED);
#else
					return (SUCCESS);
#endif

			} else {
			    snprintf(full_tty, 
				    sizeof(full_tty), "%s/%s", _PATH_DEV, 
				    request->r_tty);
		
			    /* we are the child, go and do it */
			    setgid(recipient->pw_gid);
			    setuid(recipient->pw_uid);

			    execl (rcfile, rcfile,
				TALK_PROG, 
				request->l_name, remote_machine,
				request->r_name, full_tty, NULL);
			    /* execl failed? ... */
			    _exit(-1);
			}
		}
	}
	return(announce_noX(request, remote_machine));
}



/*
 * Announcement for non X-windows:
 * See if the user is accepting messages. If so, announce that 
 * a talk is requested.
 */
static int 
announce_noX(CTL_MSG *request, const char *remote_machine)
{
	int pid, status;
	char full_tty[32];
	FILE *tf;
	struct stat stbuf;

	(void)snprintf(full_tty, sizeof(full_tty), "%s/%s", _PATH_DEV, 
		request->r_tty);
	if (access(full_tty, 0) != 0)
		return (FAILED);
	if ((tf = fopen(full_tty, "w")) == NULL)
		return (PERMISSION_DENIED);
	/*
	 * On first tty open, the server will have
	 * it's pgrp set, so disconnect us from the
	 * tty before we catch a signal.
	 */
	ioctl(fileno(tf), TIOCNOTTY, NULL);
	if (fstat(fileno(tf), &stbuf) < 0)
		return (PERMISSION_DENIED);
	if ((stbuf.st_mode&020) == 0)
		return (PERMISSION_DENIED);
	print_mesg(tf, request, remote_machine);
	fclose(tf);
	return (SUCCESS);
}

#define max(a,b) ( (a) > (b) ? (a) : (b) )
#define N_LINES 5
#define N_CHARS 256

/*
 * Build a block of characters containing the message. 
 * It is sent blank filled and in a single block to
 * try to keep the message in one piece if the recipient
 * in in vi at the time
 */
static void
print_mesg(FILE *tf, CTL_MSG *request, const char *remote_machine)
{
	struct timeval clock;
	struct timezone zone;
	struct tm *localtime();
	struct tm *localclock;
	char line_buf[N_LINES][N_CHARS];
	int sizes[N_LINES];
	char big_buf[N_LINES*N_CHARS];
        char *bptr, *lptr, *vis_user;
	int i, j, max_size;

	i = 0;
	max_size = 0;
	gettimeofday(&clock, &zone);
	localclock = localtime( (time_t*) &clock.tv_sec );
        (void)snprintf(line_buf[i], N_CHARS, " ");
	sizes[i] = strlen(line_buf[i]);
	max_size = max(max_size, sizes[i]);
	i++;
        (void)snprintf(line_buf[i], N_CHARS,
		"Message from Talk_Daemon@%s at %d:%02d ...",
		hostname, localclock->tm_hour , localclock->tm_min );
	sizes[i] = strlen(line_buf[i]);
	max_size = max(max_size, sizes[i]);
	i++;
 
        vis_user = (char*) malloc(strlen(request->l_name) * 4 + 1);
        /*strvis(vis_user, request->l_name, VIS_CSTYLE);*/
        *strcpy(vis_user, request->l_name);
        (void)snprintf(line_buf[i], N_CHARS,
		"%s: connection requested by %s@%s", 
		TALK_PROG, vis_user, remote_machine);

	sizes[i] = strlen(line_buf[i]);
	max_size = max(max_size, sizes[i]);
	i++;
	(void)snprintf(line_buf[i], N_CHARS, 
		"%s: respond with:  %s %s@%s",
		TALK_PROG, TALK_PROG, vis_user, remote_machine);
	sizes[i] = strlen(line_buf[i]);
	max_size = max(max_size, sizes[i]);
	i++;
	(void)snprintf(line_buf[i], N_CHARS, " ");
	sizes[i] = strlen(line_buf[i]);
	max_size = max(max_size, sizes[i]);
	i++;
	bptr = big_buf;
	*bptr++ = ''; /* send something to wake them up */
	*bptr++ = '\r';	/* add a \r in case of raw mode */
	*bptr++ = '\n';
	for (i = 0; i < N_LINES; i++) {
		/* copy the line into the big buffer */
		lptr = line_buf[i];
		while (*lptr != '\0')
			*(bptr++) = *(lptr++);
		/* pad out the rest of the lines with blanks */
		for (j = sizes[i]; j < max_size + 2; j++)
			*(bptr++) = ' ';
		*(bptr++) = '\r';	/* add a \r in case of raw mode */
		*(bptr++) = '\n';
	}
	*bptr = '\0';
	fprintf(tf, big_buf);
	fflush(tf);
	ioctl(fileno(tf), TIOCNOTTY, NULL);
}
